package commands;

import static javax.swing.JOptionPane.*;
import gui.Application;
import java.awt.print.PrinterException;
import java.io.File;
import javax.swing.*;
import javax.swing.text.JTextComponent;
import neutrino.dialogs.DocumentRestorer;
import neutrino.text.ITextComponent;
import neutrino.text.components.plain.PlainTextArea;

/**
 * Gateway to file commands.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
class FileCommandGateway implements ICommandGateway {

	private Application application = null;
	
	public FileCommandGateway(Application application) {
		this.application = application;
	}

    /**
     * Performs command appropriate to given type
     * @param command - CommandType
     */
    public void performCommand(CommandType command) {
        switch (command) {
            case NEW:
                newText();
                break;
            case OPEN:
                open();
                break;
            case RESTORE:
                restore();
                break;
            case SAVE:
                save();
                break;
            case SAVE_AS:
                saveAs();
                break;
            case RELOAD:
                reload();
                break;
            case PRINT:
                print();
                break;
            case COPY_FILE:
                copy();
                break;
            case MOVE:
                move();
                break;
            case REMOVE:
                remove();
                break;
            case RENAME:
                rename();
                break;
            case EXECUTE:
                toggleExecute();
                break;
            case READ_ONLY:
                toggleReadOnly();
                break;
            case WRITE_ONLY:
                toggleWriteOnly();
                break;
            case HIDDEN:
                toggleHidden();
                break;
            case EXIT:
                exit();
                break;
        }
    }

    /**
     * Returns true when the command gateway contains the command
     * @param command - CommandType
     * @return - boolean
     */
    public boolean containsCommand(CommandType command) {
        switch (command) {
            case NEW:
            case OPEN:
            case RESTORE:
            case SAVE:
            case SAVE_AS:
            case RELOAD:
            case PRINT:
            case COPY_FILE:
            case MOVE:
            case REMOVE:
            case RENAME:
            case EXECUTE:
            case READ_ONLY:
            case WRITE_ONLY:
            case HIDDEN:
            case EXIT:
                return true;
            default:
                return false;
        }
    }

    /**
     * Returns true when can perform command appropriate to given type
     * @param command - CommandType
     * @return boolean
     */
    public boolean canPerformCommand(CommandType command) {
        switch (command) {
            case NEW:
                return canNewText();
            case RESTORE:
                return canRestore();
            case SAVE:
                return canSave();
            case SAVE_AS:
                return canSaveAs();
            case RELOAD:
                return canReload();
            case PRINT:
                return canPrint();
            case COPY_FILE:
                return canCopy();
            case MOVE:
                return canMove();
            case REMOVE:
                return canRemove();
            case RENAME:
                return canRename();
            case EXECUTE:
                return canToggleExecute();
            case READ_ONLY:
                return canToggleReadOnly();
            case WRITE_ONLY:
                return canToggleWriteOnly();
            case HIDDEN:
                return canToggleHidden();
            default:
                return true;
        }
    }

    private boolean canNewText() {
        return application.getTextEditor().canClear();
    }

	private void newText() {
		if (!application.getTextEditor().isTextSaved()) return;
		application.getTextEditor().clear();
	}

    private void open() {
		if (!application.getTextEditor().isTextSaved()) return;
		application.getTextEditor().open();
	}

    private boolean canRestore() {
        return DocumentRestorer.hasBackup(application.getTextEditor());
    }

    private void restore() {
		File file = DocumentRestorer.showRestoreDialog(application, application.getTextEditor());
		application.getTextEditor().restore(file);
	}

    private boolean canSave() {
        return application.getTextEditor().canSave();
    }

    private void save() {
		application.getTextEditor().save();
	}

    private boolean canSaveAs() {
        return application.getTextEditor().canSaveAs();
    }

    private void saveAs() {
		application.getTextEditor().saveAs();
	}

    private boolean canReload() {
        return application.getTextEditor().canReload();
    }

    private void reload() {
		application.getTextEditor().reload();
	}

    private boolean canPrint() {
        JTextComponent textComponent = application.getTextEditor().getTextComponent();
        if (textComponent instanceof ITextComponent) {
            return ((ITextComponent) textComponent).canPrint();
        } else {
            return true;
        }
    }

    private void print() {
		try {
			PlainTextArea textComponent = (PlainTextArea) application.getTextEditor().getTextComponent();
			if (textComponent.isTextEmpty()) {
				showMessageDialog(application, "Nothing to print.", "Information", INFORMATION_MESSAGE);
			} else {
				JTextArea view = new JTextArea();
				view.setTabSize(textComponent.getTabSize());
				view.setText(textComponent.getText());
				view.setFont(textComponent.getFont());
				view.setLineWrap(true);
				view.setWrapStyleWord(false);
				view.print();
				view = null;
				System.gc();
			}
		} catch (PrinterException e) {
			e.printStackTrace();
		}
	}

    private boolean canCopy() {
        return application.getTextEditor().canCopy();
    }

    private void copy() {
        application.getTextEditor().copy();
    }

    private boolean canMove() {
        return application.getTextEditor().canMove();
    }

    private void move() {
        application.getTextEditor().move();
    }

    private boolean canRename() {
        return application.getTextEditor().canRename();
    }

    private void rename() {
        application.getTextEditor().rename();
    }

    private boolean canRemove() {
        return application.getTextEditor().canRemove();
    }

    private void remove() {
        application.getTextEditor().remove();
    }

    private boolean canToggleExecute() {
        return application.getTextEditor().canToggleExecute();
    }

    private void toggleExecute() {
        application.getTextEditor().toggleExecute();
    }

    private boolean canToggleReadOnly() {
        return application.getTextEditor().canToggleReadOnly();
    }

    private void toggleReadOnly() {
        application.getTextEditor().toggleReadOnly();
    }

    private boolean canToggleWriteOnly() {
        return application.getTextEditor().canToggleWriteOnly();
    }

    private void toggleWriteOnly() {
        application.getTextEditor().toggleWriteOnly();
    }

    private boolean canToggleHidden() {
        return application.getTextEditor().canToggleHidden();
    }

    private void toggleHidden() {
        application.getTextEditor().toggleHidden();
    }

    private void exit() {
		if (application.isConfirmExit()) {
			JPanel panel = new JPanel();
			BoxLayout layout = new BoxLayout(panel, BoxLayout.Y_AXIS);
			panel.setLayout(layout);
			panel.add(new JLabel("Do you want to exit?"));
			panel.add(Box.createVerticalStrut(5));
			JCheckBox checkBox = new JCheckBox("Always exit without prompt");
			checkBox.setSelected(false);
			panel.add(checkBox);
			int option = showConfirmDialog(application, panel, "Confirm exit", YES_NO_OPTION);
			application.setConfirmingExit(!checkBox.isSelected());
			if (option == NO_OPTION) return;
		}
		if (!application.getTextEditor().isTextSaved()) return;
		application.getPropertiesManager().saveProperties();
        application.getTextEditor().getBackupManager().turnOff();
		application.setVisible(false);
		application.dispose();
		System.exit(0);
	}

}
