package neutrino;

import java.awt.*;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

/**
 * Component with title that combines text field and list
 * It's displayed when inserted in JPanel with GridLayaut
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class OpenList extends JPanel implements ListSelectionListener {
	
	protected JLabel m_title;
	protected JTextField m_text;
	protected JList m_list;
	protected JScrollPane m_scroll;
	private boolean fixedSelectedValue = false;
	private boolean autoSelectingMode = true;
	
	public OpenList(String title, String[] data, boolean dataSortingMode) {
		setLayout(null);
	    m_title = new JLabel(title, JLabel.LEFT);
		add(m_title);
		m_text = new JTextField();
		m_title.setLabelFor(m_text);
		add(m_text);
		if (data != null) {
			if (dataSortingMode) {
				List<String> list = new ArrayList<String>();
				for (int i = 0; i < data.length; i++) {
					list.add(data[i]);
				}
				Collections.sort(list);
				m_list = new JList(list.toArray());
				if (!list.isEmpty()) m_text.setText(list.get(0).toString());
			} else {
				m_list = new JList(data);
				if (data.length > 0) m_text.setText(data[0].toString());
			}
		} else {
			m_list = new JList();
		}
		m_list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		m_list.setVisibleRowCount(6);
		m_list.addListSelectionListener(this);
		m_scroll = new JScrollPane(m_list);
		m_text.addKeyListener(textTypeListener);
		add(m_scroll);
	}
	
	public OpenList(String title, String[] data) {
		this(title, data, true);
	}
	
	public OpenList(String title) {
		this(title, null, true);
	}

    public OpenList() {
        this(null, null, true);
    }
	
	private KeyListener textTypeListener = new KeyListener() {
		@Override
		public void keyTyped(KeyEvent e) {
		}
		@Override
		public void keyReleased(KeyEvent e) { 
			if (autoSelectingMode) {
				String key = m_text.getText();
				ListModel model = m_list.getModel();
				for (int i = 0; i < model.getSize(); i++) {
					if (model.getElementAt(i).toString().toLowerCase().startsWith(key.toLowerCase())) {
						m_list.setSelectedValue(model.getElementAt(i), true);
						break;
					}
				}
			}
		}
		@Override
		public void keyPressed(KeyEvent e) { 
		}
	}; 
	
	public void setTitle(String title) {
		m_title.setText(title);
	}
	
	public String getTitle() {
		return m_title.getText();
	}
	
	public void setListData(Object[] listData, boolean dataSortingMode) {
		if (listData instanceof String[] && dataSortingMode) {
			final List<String> list = new ArrayList<String>();
			for (int i = 0; i < listData.length; i++) {
				list.add((String) listData[i]);
			}
			Collections.sort(list);
			m_list.setModel(new AbstractListModel() {
				@Override
				public int getSize() {
					return list.size();
				}
				@Override
				public Object getElementAt(int index) {
					return list.get(index);
				}
			});
			if (!list.isEmpty()) m_text.setText(list.get(0).toString());
		} else {
			m_list.setListData(listData);
			if (listData.length > 0) m_text.setText(listData[0].toString());
		}
	}
	
	public void setListData(Vector<?> listData, boolean dataSortingMode) {
		if (!listData.isEmpty() && listData.firstElement() instanceof String && dataSortingMode) {
			final List<String> list = new ArrayList<String>();
			for (int i = 0; i < listData.size(); i++) {
				list.add((String) listData.get(i));
			}
			Collections.sort(list);
			m_list.setModel(new AbstractListModel() {
				@Override
				public int getSize() {
					return list.size();
				}
				@Override
				public Object getElementAt(int index) {
					return list.get(index);
				}
			});
			if (!list.isEmpty()) m_text.setText(list.get(0).toString());
		} else {
			m_list.setListData(listData);
			if (!listData.isEmpty()) m_text.setText(listData.get(0).toString());
		}
	}

	public void setModel(ListModel model, boolean dataSortingMode) {
		if (model.getSize() > 0 && model.getElementAt(0) instanceof String && dataSortingMode) {
			final List<String> list = new ArrayList<String>();
			for (int i = 0; i < model.getSize(); i++) {
				list.add((String) model.getElementAt(i));
			}
			Collections.sort(list);
			m_list.setModel(new AbstractListModel() {
				@Override
				public int getSize() {
					return list.size();
				}
				@Override
				public Object getElementAt(int index) {
					return list.get(index);
				}
			});
		} else {
			m_list.setModel(model);
		}
		if (model.getSize() > 0) m_text.setText(model.getElementAt(0).toString());
	}
	
	public ListModel getModel() {
		return m_list.getModel();
	}
	
	public void setSelectedIndex(int index) {
		m_list.setSelectedIndex(index);
	}

    /**
     works when setSize is not used
      */
	public void setVisibleRowCount(int rowCount) {
		m_list.setVisibleRowCount(rowCount);
	}
	
	public void setSelectedValue(String sel) {
		m_list.setSelectedValue(sel, true);
		m_text.setText(sel);
	}
	
	public String getSelectedValue() { 
		if (fixedSelectedValue) {
			return m_list.getSelectedValue().toString();
		} else {
			return m_text.getText(); 
		}
	}
	
	public boolean isFixedSelectedValue() {
		return fixedSelectedValue;
	}

	public void setFixedSelectedValue(boolean fixedSelectedValue) {
		this.fixedSelectedValue = fixedSelectedValue;
	}

	public boolean isAutoSelectingMode() {
		return autoSelectingMode;
	}

	public void setAutoSelectingMode(boolean autoSelectingMode) {
		this.autoSelectingMode = autoSelectingMode;
	}

	@Override
	public void valueChanged(ListSelectionEvent e) {
		if (m_list.hasFocus()) {
		    Object obj = m_list.getSelectedValue();
		    if (obj != null) m_text.setText(obj.toString());
		}
	}
	
	public void setDisplayedMnemonic(char aChar) {
		m_title.setDisplayedMnemonic(aChar);
	}
	
	public void setDisplayedMnemonic(int aChar) {
		m_title.setDisplayedMnemonic(aChar);
	}
	
	public void setDisplayedMnemonicIndex(int index) throws IllegalArgumentException {
		m_title.setDisplayedMnemonicIndex(index);
	}
	
	public void addListSelectionListener(ListSelectionListener lst) {
		m_list.addListSelectionListener(lst);
	}
	
	public void removeListSelectionListener(ListSelectionListener lst) {
		m_list.removeListSelectionListener(lst);
	}
	
	public void addKeyListener(KeyListener keyListener) {
		m_text.addKeyListener(keyListener);
	}
	
	public void removeKeyListener(KeyListener keyListener) {
		m_text.removeKeyListener(keyListener);
	}
	
	public Dimension getPreferredSize() {
		Insets ins = getInsets();
		Dimension d1 = m_title.getPreferredSize();
		Dimension d2 = m_text.getPreferredSize();
		Dimension d3 = m_scroll.getPreferredSize();
		int w = Math.max(Math.max(d1.width, d2.width), d3.width);
		int h = d1.height + d2.height + d3.height;
		return new Dimension(w+ins.left+ins.right, 
				h+ins.top+ins.bottom);
	}

	public Dimension getMaximumSize() {
		Insets ins = getInsets();
		Dimension d1 = m_title.getMaximumSize();
		Dimension d2 = m_text.getMaximumSize();
		Dimension d3 = m_scroll.getMaximumSize();
		int w = Math.max(Math.max(d1.width, d2.width), d3.width);
		int h = d1.height + d2.height + d3.height;
		return new Dimension(w+ins.left+ins.right, 
				h+ins.top+ins.bottom);
	}

	public Dimension getMinimumSize() {
		Insets ins = getInsets();
		Dimension d1 = m_title.getMinimumSize();
		Dimension d2 = m_text.getMinimumSize();
		Dimension d3 = m_scroll.getMinimumSize();
		int w = Math.max(Math.max(d1.width, d2.width), d3.width);
		int h = d1.height + d2.height + d3.height;
		return new Dimension(w+ins.left+ins.right,
				h+ins.top+ins.bottom);
	}

    public void setSize(int width, int height) {
        Insets ins = getInsets();
        int w = width-ins.left-ins.right;
        int h = height-ins.top-ins.bottom;

        Dimension d1 = m_title.getPreferredSize();
        m_title.setPreferredSize(new Dimension(w, d1.height));
        Dimension d2 = m_text.getPreferredSize();
        m_text.setPreferredSize(new Dimension(w, d2.height));
        int y = d1.height + d2.height;
        m_scroll.setPreferredSize(new Dimension(w, h - y));
    }

	public void doLayout() {
		Insets ins = getInsets();
		Dimension d = getSize();
		int x = ins.left;
		int y = ins.top;
		int w = d.width-ins.left-ins.right;
		int h = d.height-ins.top-ins.bottom;
		
		Dimension d1 = m_title.getPreferredSize();
		m_title.setBounds(x, y, w, d1.height);
		y += d1.height;
		Dimension d2 = m_text.getPreferredSize();
		m_text.setBounds(x, y, w, d2.height);
		y += d2.height;
		m_scroll.setBounds(x, y, w, h-y);
	}	
	
}
