package neutrino.autocomplete;

import org.fife.ui.autocomplete.Completion;
import org.fife.ui.autocomplete.CompletionProvider;

import javax.swing.*;
import javax.swing.text.JTextComponent;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

/**
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class TagCompletion implements Completion {

    /**
     * The provider that created this completion;
     */
    private CompletionProvider provider;

    /**
     * The icon to use for this completion.
     */
    private Icon icon;

    /**
     * The relevance of this completion.  Completion instances with higher
     * "relevance" values are inserted higher into the list of possible
     * completions than those with lower values.  Completion instances with
     * equal relevance values are sorted alphabetically.
     */
    private int relevance;

    private String replacementText;
	private String shortDesc;
	private String summary;
    private String tagName;

    private List<TagChild> children;
    private List<Completion> attributes;


	/**
	 * Constructor.
	 *
	 * @param provider The parent completion provider.
	 * @param replacementText The text to replace.
	 * @param shortDesc A short description of the completion.  This will be
	 *        displayed in the completion list.  This may be <code>null</code>.
	 */
	public TagCompletion(CompletionProvider provider, String replacementText,
                         String shortDesc) {
		this(provider, replacementText, shortDesc, null);
	}


	/**
	 * Constructor.
	 *
	 * @param provider The parent completion provider.
	 * @param replacementText The text to replace.
	 * @param shortDesc A short description of the completion.  This will be
	 *        displayed in the completion list.  This may be <code>null</code>.
	 * @param summary The summary of this completion.  This should be HTML.
	 *        This may be <code>null</code>.
	 */
	public TagCompletion(CompletionProvider provider, String replacementText,
                         String shortDesc, String summary) {
		this.provider = provider;
        this.tagName = replacementText;
		this.replacementText = "<" + replacementText + ">";
        if (shortDesc != null && shortDesc.isEmpty()) shortDesc = null;
		this.shortDesc = shortDesc;
        this.children = getTagChildrenFromDescription(shortDesc);
		this.summary = summary;
        this.attributes = new ArrayList<Completion>();
	}

    public List<Completion> getAttributes() {
        return this.attributes;
    }

    private int multiplicity = TagChild.ONLY_ONE;

    public int getMultiplicity() {
        return this.multiplicity;
    }

    private List<TagChild> getTagChildrenFromDescription(String description) {
        ArrayList<TagChild> children = new ArrayList<TagChild>();
        if (description == null || description.isEmpty()) return children;
        multiplicity = TagChild.ONLY_ONE;
        if (description.endsWith("*")) {
            description = description.substring(0, description.length() - 1);
            multiplicity = TagChild.ZERO_OR_MORE;
        }
        if (description.endsWith("+")) {
            description = description.substring(0, description.length() - 1);
            multiplicity = TagChild.ONE_OR_MORE;
        }
        StringTokenizer tokenizer = new StringTokenizer(description, "(,|)");
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            if (token.trim().equals("#PCDATA")) continue;
            TagChild tagChild = new TagChild(token.trim());
            children.add(tagChild);
        }
        return children;
    }

    public List<TagChild> getChildren() {
        return this.children;
    }

    /**
	 * {@inheritDoc}
	 */
	public String getReplacementText() {
		return replacementText;
	}


	/**
	 * Returns the short description of this completion, usually used in
	 * the completion choices list.
	 *
	 * @return The short description, or <code>null</code> if there is none.
	 * @see #setShortDescription(String)
	 */
	public String getShortDescription() {
		return shortDesc;
	}


	/**
	 * {@inheritDoc}
	 */
	public String getSummary() {
		return summary;
	}


	/**
	 * Sets the short description of this completion.
	 *
	 * @param shortDesc The short description of this completion.
	 * @see #getShortDescription()
	 */
	public void setShortDescription(String shortDesc) {
		this.shortDesc = shortDesc;
	}


	/**
	 * Sets the summary for this completion.
	 *
	 * @param summary The summary for this completion.
	 * @see #getSummary()
	 */
	public void setSummary(String summary) {
		this.summary = summary;
	}

    /**
     * {@inheritDoc}
     */
    public int compareTo(Completion c2) {
        if (c2==this) {
            return 0;
        }
        else if (c2!=null) {
            return toString().compareToIgnoreCase(c2.toString());
        }
        return -1;
    }


    /**
     * {@inheritDoc}
     */
    public String getAlreadyEntered(JTextComponent comp) {
        return provider.getAlreadyEnteredText(comp);
    }


    /**
     * {@inheritDoc}
     */
    public Icon getIcon() {
        return icon;
    }


    /**
     * Returns the text the user has to (start) typing for this completion
     * to be offered.  The default implementation simply returns
     * {@link #getReplacementText()}.
     *
     * @return The text the user has to (start) typing for this completion.
     * @see #getReplacementText()
     */
    public String getInputText() {
        return getReplacementText();
    }


    /**
     * {@inheritDoc}
     */
    public CompletionProvider getProvider() {
        return provider;
    }


    /**
     * {@inheritDoc}
     */
    public int getRelevance() {
        return relevance;
    }


    /**
     * The default implementation returns <code>null</code>.  Subclasses
     * can override this method.
     *
     * @return The tool tip text.
     */
    public String getToolTipText() {
        return null;
    }


    /**
     * Sets the icon to use for this completion.
     *
     * @param icon The icon to use.
     * @see #getIcon()
     */
    public void setIcon(Icon icon) {
        this.icon = icon;
    }


    /**
     * Sets the relevance of this completion.
     *
     * @param relevance The new relevance of this completion.
     * @see #getRelevance()
     */
    public void setRelevance(int relevance) {
        this.relevance = relevance;
    }

	/**
	 * Returns a string representation of this completion.  If the short
	 * description is not <code>null</code>, this method will return:
	 * 
	 * <code>getInputText() + " - " + shortDesc</code>
	 * 
	 * otherwise, it will return <tt>getInputText()</tt>.
	 *
	 * @return A string representation of this completion.
	 */
	@Override
	public String toString() {
		if (shortDesc==null) {
			return " " + getTagName();
		}
		return " " + getTagName() + " - " + shortDesc;
	}


    public String getTagName() {
        return tagName;
    }

    public void setTagName(String tagName) {
        this.tagName = tagName;
    }

}