package neutrino.dialogs;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import static javax.swing.JOptionPane.*; 
import javax.swing.border.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.text.JTextComponent;

import neutrino.text.IPlainTextComponent;
import neutrino.text.ITextComponent;

/**
 * Supply dialog for performing complex editing operation in instance of TextComponent. 
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class ComplexOperationChooser {
	
	private static boolean isTextEmpty(JTextComponent textComponent) {
		return textComponent.getDocument().getLength() == 0;
	}
	
	private static boolean isTextSelected(JTextComponent textComponent) {
		return textComponent.getSelectionStart() != textComponent.getSelectionEnd();
	}
	
	private class ComplexOperationDialog extends JDialog {
		
		private JButton bOk = new JButton("Ok");
		private JButton bCancel = new JButton("Cancel");
		private JTextComponent textComponent = null;
		private ActionListener alOk = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if (!validateInput()) {
					return;
				}
				if (textComponent instanceof IPlainTextComponent) {
					performComplexOperation((IPlainTextComponent) textComponent);
				}
				setVisible(false);
				dispose();
			}
		};

		/**
		 * precondition : input is valid
		 */
		private void performComplexOperation(IPlainTextComponent textComponent) {
            if (textComponent instanceof ITextComponent) {
                ((ITextComponent) textComponent).beginEdit();
            }
			if (cbTransformText.isSelected()) {
				if (rbMakeUppercase.isSelected()) {
					textComponent.makeUpperCase();
				} else {
					textComponent.makeLowerCase();
				}
			}
			if (cbPerformReplacement.isSelected()) {
				if (rbTabifySelectedLines.isSelected()) {
					textComponent.tabifySelectedLines();
				} else {
					textComponent.untabifySelectedLines();
				}
			}
			if (cbDeleteHorizontalWhiteSpace.isSelected()) {
				textComponent.deleteHorizontalWhiteSpace();
			}
			if (cbRemoveTrailingWhitespaces.isSelected()) {
				textComponent.removeTrailingWhitespaces();
			}
			if (cbChangeLineIndent.isSelected()) {
				int count = Integer.parseInt(tfChangeLineIndentBy.getText());
				for (int i = 0; i < count; i++) {
					if (rbIncreaseLineIndent.isSelected()) {
						textComponent.shiftInRight();
					} else {
						textComponent.shiftInLeft();
					}
				}
			}
            if (textComponent instanceof ITextComponent) {
                ((ITextComponent) textComponent).endEdit();
            }
		}
		
		private boolean validateInput() {
			if (cbChangeLineIndent.isSelected()) {
				int count;
				try {
					count = Integer.parseInt(tfChangeLineIndentBy.getText());
				} catch (NumberFormatException e) {
					showMessageDialog(getOwner(), "The count of changing line indent operations is not number", "Error message", ERROR_MESSAGE);
					return false;
				}
				if (count < 1) {
					showMessageDialog(getOwner(), "The count of changing line indent operations is less then 1", "Error message", ERROR_MESSAGE);
					return false;
				}
				if (count > 512) {
					showMessageDialog(getOwner(), "The count of changing line indent operations is greater then 512", "Error message", ERROR_MESSAGE);
					return false;
				}
			}
			return true; 
		}
		
		private ActionListener alCancel = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				setVisible(false);
				dispose();
			}
		};
		private JCheckBox cbTransformText = new JCheckBox("Transform text");
		private JPanel pTextTransformation = new JPanel();
		private ButtonGroup bgTextTransformation = new ButtonGroup();
		private JRadioButton rbMakeUppercase = new JRadioButton("Make uppercase");
		private JRadioButton rbMakeLowercase = new JRadioButton("Make lowercase");
		private JCheckBox cbPerformReplacement = new JCheckBox("Perform replacement");
		private JPanel pTextReplacement = new JPanel();
		private ButtonGroup bgTextReplacement = new ButtonGroup();
		private JRadioButton rbTabifySelectedLines = new JRadioButton("Tabify selected lines");
		private JRadioButton rbUntabifySelectedLines = new JRadioButton("Untabify selected lines");
		private JCheckBox cbDeleteHorizontalWhiteSpace = new JCheckBox("Delete horizontal white space");
		private JCheckBox cbRemoveTrailingWhitespaces = new JCheckBox("Remove trailing whitespaces");
		private JCheckBox cbChangeLineIndent = new JCheckBox("Change line indent");
		private JPanel pLineIndent = new JPanel();
		private ButtonGroup bgLineIndent = new ButtonGroup();
		private JRadioButton rbIncreaseLineIndent = new JRadioButton("Increase line indent");
		private JRadioButton rbDecreaseLineIndent = new JRadioButton("Decrease line indent");
		private JLabel lChangeLineIndentBy = new JLabel("By :");
		private JTextField tfChangeLineIndentBy = new JTextField(5);		
		
		public ComplexOperationDialog(Window owner, JTextComponent textComponent) {
			super(owner);
			this.textComponent = textComponent;
			setTitle("Perform complex operation");
			setModal(true);
			setResizable(false);
			setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.anchor = GridBagConstraints.NORTH;
			c.insets = new Insets(10, 10, 5, 0);
			add(getLeftPanel(), c);
			c.gridx = 1;
			c.insets = new Insets(10, 5, 5, 10);
			add(getRightPanel(), c);
			c.gridx = 0;
			c.gridy = 1;
			c.gridwidth = 2;			
			c.insets = new Insets(0, 5, 10, 5);
			add(getButtonsPanel(), c);
			pack();
		}
				
		private JPanel getLeftPanel() {
			cbTransformText.setMnemonic(KeyEvent.VK_N);
			cbTransformText.addChangeListener(new  ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					boolean isEnabled = cbTransformText.isSelected(); 
					rbMakeUppercase.setEnabled(isEnabled);
					rbMakeLowercase.setEnabled(isEnabled);
				}
			});
			cbTransformText.setEnabled(isTextSelected(textComponent));
			bgTextTransformation.add(rbMakeUppercase);
			bgTextTransformation.add(rbMakeLowercase);
			rbMakeUppercase.setSelected(true);
			rbMakeUppercase.setMnemonic(KeyEvent.VK_U);
			rbMakeLowercase.setMnemonic(KeyEvent.VK_L);
			rbMakeUppercase.setEnabled(false);
			rbMakeLowercase.setEnabled(false);
			pTextTransformation.setBorder(new CompoundBorder (new TitledBorder("Text transformation"), 
					new EmptyBorder(0, 5, 5, 5)));
			BoxLayout lTextTransformation = new BoxLayout(pTextTransformation, BoxLayout.Y_AXIS);
			pTextTransformation.setLayout(lTextTransformation);
			pTextTransformation.add(rbMakeUppercase);
			pTextTransformation.add(rbMakeLowercase);
			cbPerformReplacement.setMnemonic(KeyEvent.VK_F);
			cbPerformReplacement.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					boolean isEnabled = cbPerformReplacement.isSelected();
					rbTabifySelectedLines.setEnabled(isEnabled);
					rbUntabifySelectedLines.setEnabled(isEnabled);
				}
			});
			cbPerformReplacement.setEnabled(isTextSelected(textComponent));
			bgTextReplacement.add(rbTabifySelectedLines);
			bgTextReplacement.add(rbUntabifySelectedLines);
			rbTabifySelectedLines.setMnemonic(KeyEvent.VK_T);
			rbUntabifySelectedLines.setMnemonic(KeyEvent.VK_U);
			rbTabifySelectedLines.setSelected(true);
			rbTabifySelectedLines.setEnabled(false);
			rbUntabifySelectedLines.setEnabled(false);
			pTextReplacement.setBorder(new CompoundBorder (new TitledBorder("Text replacement"), 
					new EmptyBorder(0, 5, 5, 5)));
			BoxLayout blTextReplacement = new BoxLayout(pTextReplacement, BoxLayout.Y_AXIS);
			pTextReplacement.setLayout(blTextReplacement);
			pTextReplacement.add(rbTabifySelectedLines);
			pTextReplacement.add(rbUntabifySelectedLines);
			
			// create left panel
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.anchor = GridBagConstraints.WEST;
			c.fill = GridBagConstraints.HORIZONTAL;
			panel.add(cbTransformText, c);
			c.gridy = 1;			
			panel.add(pTextTransformation, c);
			c.gridy = 2;
			panel.add(cbPerformReplacement, c);
			c.gridy = 3;
			panel.add(pTextReplacement, c);
			return panel;
		}
		
		private JPanel getRightPanel() {
			cbDeleteHorizontalWhiteSpace.setMnemonic(KeyEvent.VK_D);
			cbRemoveTrailingWhitespaces.setMnemonic(KeyEvent.VK_R);
			cbChangeLineIndent.setMnemonic(KeyEvent.VK_L);
			cbChangeLineIndent.addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					boolean isEnabled = cbChangeLineIndent.isSelected();
					rbIncreaseLineIndent.setEnabled(isEnabled);
					rbDecreaseLineIndent.setEnabled(isEnabled);
					tfChangeLineIndentBy.setEnabled(isEnabled);
				}
			});
			bgLineIndent.add(rbIncreaseLineIndent);
			bgLineIndent.add(rbDecreaseLineIndent);
			rbIncreaseLineIndent.setMnemonic(KeyEvent.VK_I);
			rbDecreaseLineIndent.setMnemonic(KeyEvent.VK_E);
			rbIncreaseLineIndent.setSelected(true);
			rbIncreaseLineIndent.setEnabled(false);
			rbDecreaseLineIndent.setEnabled(false);
			lChangeLineIndentBy.setDisplayedMnemonic(KeyEvent.VK_B);
			lChangeLineIndentBy.setLabelFor(tfChangeLineIndentBy);
			tfChangeLineIndentBy.setText("1");
			tfChangeLineIndentBy.setEnabled(false);
			pLineIndent.setBorder(new CompoundBorder(new TitledBorder("Line indent"), 
					new EmptyBorder(0, 5, 5, 5)));
			pLineIndent.setLayout(new GridBagLayout());
			GridBagConstraints cli = new GridBagConstraints();
			cli.gridx = 0;
			cli.gridy = 0;
			cli.weightx = 0.5;
			cli.gridwidth = 2;
			cli.anchor = GridBagConstraints.WEST;
			pLineIndent.add(rbIncreaseLineIndent, cli);
			cli.gridy = 1;
			pLineIndent.add(rbDecreaseLineIndent, cli);
			cli.gridx = 0;
			cli.gridy = 2;
			cli.gridwidth = 1;
			cli.insets = new Insets(0, 5, 0, 5);
			pLineIndent.add(lChangeLineIndentBy, cli);
			cli.gridx = 1;
			cli.insets = new Insets(0, 0, 0, 0);
			cli.weightx = 10;
			pLineIndent.add(tfChangeLineIndentBy, cli);
						
			// create right panel
			JPanel panel = new JPanel();
			panel.setLayout(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
			c.gridx = 0;
			c.gridy = 0;
			c.anchor = GridBagConstraints.WEST;
			c.fill = GridBagConstraints.HORIZONTAL;
			panel.add(cbDeleteHorizontalWhiteSpace, c);
			c.gridy = 1;
			panel.add(cbRemoveTrailingWhitespaces, c);
			c.gridy = 2;
			panel.add(cbChangeLineIndent, c);
			c.gridy = 3;
			panel.add(pLineIndent, c);
			
			return panel;
		}
		
		private JPanel getButtonsPanel() {
			getRootPane().setDefaultButton(bOk);
			bOk.setMnemonic(KeyEvent.VK_O);
			bCancel.setMnemonic(KeyEvent.VK_C);
			bOk.addActionListener(alOk);
			bCancel.addActionListener(alCancel);
			JPanel panel = new JPanel();
			BoxLayout layout = new BoxLayout(panel, BoxLayout.X_AXIS);
			panel.setLayout(layout);
			panel.add(Box.createHorizontalGlue());
			panel.add(bOk);
			panel.add(Box.createHorizontalStrut(5));
			panel.add(bCancel);
			panel.add(Box.createHorizontalGlue());
			return panel;
		}
	}	
	
	private static void establishBounds(JDialog dialog, Window owner) {
	      Dimension d1 = dialog.getSize();
	      Dimension d2 = owner.getSize();
	      Dimension ds = dialog.getToolkit().getScreenSize();
	      int x = Math.max((d2.width-d1.width)/2, 0);
	      int y = Math.max((d2.height-d1.height)/2, 0);
	      int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
	      int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
	      dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
	}
	
	
	
	public static void showComplexOperationDialog(Window owner, JTextComponent textComponent) {
		if (!(textComponent instanceof IPlainTextComponent)) {
			showMessageDialog(owner, "Text component not supports the complex operation", "Error message", ERROR_MESSAGE);
			return;
		}
		if (isTextEmpty(textComponent)) {
			showMessageDialog(owner, "Cannot perform complex operation because text is empty", "Error message", ERROR_MESSAGE);
			return;
		}
		ComplexOperationDialog dialog = new ComplexOperationChooser().new ComplexOperationDialog(owner, textComponent);
		establishBounds(dialog, owner);
		dialog.setVisible(true);
	}

}
