package neutrino.dialogs;

import neutrino.TextEditorWindow;
import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import static neutrino.TextEditorWindow.*;

/**
 * Dialog for changing window position and size on the start of program.
 * Also some options of window behaviour.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class WindowOptionsChooser {

    private class WindowOptionsDialog extends JDialog {

        private JCheckBox cbSaveWindowSize = new JCheckBox("Save window size");
        private JCheckBox cbConfirmExit = new JCheckBox("Confirm exit when closing window");
        private JCheckBox cbShowToolBar = new JCheckBox("Show tool bar");
        private JRadioButton rbCenterWindowOnScreen = new JRadioButton("Center window on screen");
        private JRadioButton rbSaveWindowLocation = new JRadioButton("Save window location");
        private JRadioButton rbRandomizeWindowLocation = new JRadioButton("Randomize window location");
        private ButtonGroup bgWindowLocation = new ButtonGroup();

        private JButton bOk = new JButton("Ok");
        private JButton bCancel = new JButton("Cancel");
        private JButton bDefaults = new JButton("Defaults");

        private TextEditorWindow window = null;

        private TextEditorWindow getWindow() {
            return this.window;
        }

        /**
         * Initialize dialog.
         */
        private void initialize() {
            cbConfirmExit.setSelected(getWindow().isConfirmExit());
            cbShowToolBar.setSelected(getWindow().isToolBarVisible());
            cbSaveWindowSize.setSelected(getWindow().isSaveWindowSize());
            if (getWindow().getTypeOfWindowLocation() == SAVE_WINDOW_LOCATION) {
                rbSaveWindowLocation.setSelected(true);
            } else if (getWindow().getTypeOfWindowLocation() == CENTER_WINDOW_ON_SCREEN) {
                rbCenterWindowOnScreen.setSelected(true);
            } else if (getWindow().getTypeOfWindowLocation() == RANDOMIZE_WINDOW_LOCATION) {
                rbRandomizeWindowLocation.setSelected(true);
            }
        }

        /**
         * Set default values.
         */
        private void defaults() {
            cbConfirmExit.setSelected(true);
            cbShowToolBar.setSelected(true);
            cbSaveWindowSize.setSelected(true);
            rbCenterWindowOnScreen.setSelected(true);
        }

        /**
         * Establish propgarm variables
         */
        private void establishValues() {
            getWindow().setConfirmingExit(cbConfirmExit.isSelected());
            getWindow().setSavingWindowSize(cbSaveWindowSize.isSelected());
            getWindow().setToolBarVisible(cbShowToolBar.isSelected());
            if (rbSaveWindowLocation.isSelected()) {
                getWindow().setTypeOfWindowLocation(SAVE_WINDOW_LOCATION);
            } else if (rbCenterWindowOnScreen.isSelected()) {
                getWindow().setTypeOfWindowLocation(CENTER_WINDOW_ON_SCREEN);
            } else if (rbRandomizeWindowLocation.isSelected()) {
                getWindow().setTypeOfWindowLocation(RANDOMIZE_WINDOW_LOCATION);
            }
        }

        public WindowOptionsDialog(TextEditorWindow owner) {
            super();
            this.window = owner;
            setTitle("Window options");

            JPanel panel = new JPanel();
            panel.setBorder(new EmptyBorder(5, 5, 5, 5));
            panel.setLayout(new FlowLayout());
            panel.add(getMainPanel());

            getContentPane().add(panel);
            getRootPane().setDefaultButton(bOk);
            pack();

            initialize();

            setResizable(false);
            setModal(true);
            setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        }

        private JPanel getMainPanel() {
            JPanel mainPanel = new JPanel();
            GridBagLayout mainLayout = new GridBagLayout();
            GridBagConstraints constraints = new GridBagConstraints();
            mainPanel.setLayout(mainLayout);
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTH;
            constraints.insets = new Insets(0, 5, 0, 0);
            mainPanel.add(getWindowPanel(), constraints);

            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTH;
            constraints.insets = new Insets(5, 10, 0, 0);
            mainPanel.add(getButtonsPanel(), constraints);
            return mainPanel;
        }

        private JPanel getWindowPanel() {
            cbConfirmExit.setMnemonic(KeyEvent.VK_N);
            cbShowToolBar.setMnemonic(KeyEvent.VK_B);
            cbSaveWindowSize.setMnemonic(KeyEvent.VK_V);
            rbCenterWindowOnScreen.setMnemonic(KeyEvent.VK_E);
            rbSaveWindowLocation.setMnemonic(KeyEvent.VK_S);
            rbRandomizeWindowLocation.setMnemonic(KeyEvent.VK_Z);
            bgWindowLocation.add(rbSaveWindowLocation);
            bgWindowLocation.add(rbCenterWindowOnScreen);
            bgWindowLocation.add(rbRandomizeWindowLocation);
            JPanel pLocation = new JPanel();
            BoxLayout blLocation = new BoxLayout(pLocation, BoxLayout.Y_AXIS);
            pLocation.setBorder(new TitledBorder("Location"));
            pLocation.setLayout(blLocation);
            pLocation.add(rbCenterWindowOnScreen);
            pLocation.add(Box.createHorizontalStrut(5));
            pLocation.add(rbSaveWindowLocation);
            pLocation.add(Box.createHorizontalStrut(5));
            pLocation.add(rbRandomizeWindowLocation);

            JPanel windowPanel = new JPanel();
            Border windowBorder = new TitledBorder("Window");
            windowPanel.setLayout(new GridBagLayout());
            windowPanel.setBorder(windowBorder);
            GridBagConstraints c = new GridBagConstraints();
            c.gridx = 0;
            c.gridy = 0;
            c.anchor = GridBagConstraints.WEST;
            c.fill = GridBagConstraints.HORIZONTAL;
            c.insets = new Insets(0, 0, 5, 0);
            windowPanel.add(cbConfirmExit, c);
            c.gridy = 1;
            windowPanel.add(cbShowToolBar, c);
            c.gridy = 2;
            windowPanel.add(cbSaveWindowSize, c);
            c.gridy = 3;
            c.insets = new Insets(0, 0, 0, 0);
            windowPanel.add(pLocation, c);

            return windowPanel;
        }

        private JPanel getButtonsPanel() {
            bOk.setToolTipText("Set options");
            bDefaults.setToolTipText("Select default values");
            bCancel.setToolTipText("Abort window options dialog");
            bOk.setMnemonic(KeyEvent.VK_O);
            bCancel.setMnemonic(KeyEvent.VK_C);
            bDefaults.setMnemonic(KeyEvent.VK_D);
            bOk.addActionListener(alOk);
            bCancel.addActionListener(alCancel);
            bDefaults.addActionListener(alDefaults);
            JPanel buttonsPanel = new JPanel();
            GridLayout buttonsLayout = new GridLayout(3, 1, 0, 5);
            buttonsPanel.setLayout(buttonsLayout);
            buttonsPanel.add(bOk);
            buttonsPanel.add(bCancel);
            buttonsPanel.add(bDefaults);
            return buttonsPanel;
        }

        private ActionListener alOk = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                establishValues();
                setVisible(false);
                dispose();
            }
        };

        private ActionListener alCancel = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setVisible(false);
                dispose();
            }
        };

        private ActionListener alDefaults = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                defaults();
            }
        };

    }


    private static void establishBounds(JDialog dialog, Window owner) {
        Dimension d1 = dialog.getSize();
        Dimension d2 = owner.getSize();
        Dimension ds = dialog.getToolkit().getScreenSize();
        int x = Math.max((d2.width-d1.width)/2, 0);
        int y = Math.max((d2.height-d1.height)/2, 0);
        int xshift = ((x + d1.width + owner.getX()) > ds.width) ? (((ds.width - d1.width) / 2) - x) : owner.getX();
        int yshift = ((y + d1.height + owner.getY()) > ds.height) ? (((ds.height - d1.height) / 2) - y) : owner.getY();
        dialog.setBounds(x + xshift, y + yshift, d1.width, d1.height);
    }

    public static void showWindowOptionsDialog(TextEditorWindow owner) {
        WindowOptionsChooser chooser = new WindowOptionsChooser();
        WindowOptionsDialog dialog = chooser.new WindowOptionsDialog(owner);
        establishBounds(dialog, owner);
        dialog.setVisible(true);
    }

}
