/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad;

import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import jnpad.config.Config;

/**
 * The Class JNPadBundle.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class JNPadBundle {
  /** The Constant BUNDLE_NAME. */
  private static final String              BUNDLE_NAME = "jnpad.i18n.JNPadBundle"; //$NON-NLS-1$

  /** The bundle. */
  private static Reference<ResourceBundle> ourBundle;

  /**
   * Gets the string.
   *
   * @param key the key
   * @param params the params
   * @return the string
   */
  public static String getString(final String key, final Object... params) {
    return getString(getBundle(), key, params);
  }

  /**
   * Gets the string.
   *
   * @param bundle the bundle
   * @param key the key
   * @param params the params
   * @return the string
   */
  public static String getString(final ResourceBundle bundle, final String key, final Object... params) {
    return getMessage(bundle, key, null, params);
  }

  /**
   * Gets the message.
   *
   * @param key the key
   * @param defaultValue the default value
   * @param params the params
   * @return the message
   */
  public static String getMessage(final String key, final String defaultValue, final Object... params) {
    return getMessage(getBundle(), key, defaultValue, params);
  }

  /**
   * Gets the message.
   *
   * @param bundle the bundle
   * @param key the key
   * @param defaultValue the default value
   * @param params the params
   * @return the message
   */
  public static String getMessage(final ResourceBundle bundle, final String key, final String defaultValue, final Object... params) {
    if (bundle == null)
      return defaultValue;

    String value;
    try {
      value = bundle.getString(key);
    }
    catch (MissingResourceException e) {
      if (defaultValue != null) {
        value = defaultValue;
      }
      else {
        value = '!' + key + '!';
      }
    }

    if (params != null && params.length > 0 && value.indexOf('{') >= 0) {
      return MessageFormat.format(value, params);
    }

    return value;
  }

  /**
   * Gets the bundle.
   *
   * @return the bundle
   */
  private static ResourceBundle getBundle() {
    ResourceBundle bundle = null;
    if (ourBundle != null) {
      bundle = ourBundle.get();
    }
    if (bundle == null) {
      Locale locale = Config.getLocale();
      bundle = (locale != null) ?
          ResourceBundle.getBundle(BUNDLE_NAME, locale) :
          ResourceBundle.getBundle(BUNDLE_NAME);
      ourBundle = new SoftReference<ResourceBundle>(bundle);
    }
    return bundle;
  }

  /**
   * Gets the cancel button text.
   *
   * @return the cancel button text
   */
  public static String getCancelButtonText() {
    return getString("button.cancel"); //$NON-NLS-1$
  }

  /**
   * Gets the ok button text.
   *
   * @return the ok button text
   */
  public static String getOkButtonText() {
    return getString("button.ok"); //$NON-NLS-1$
  }

  /**
   * Gets the activate button text.
   *
   * @return the activate button text
   */
  public static String getActivateButtonText() {
    return getString("button.activate"); //$NON-NLS-1$
  }

  /**
   * Gets the sort button text.
   *
   * @return the sort button text
   */
  public static String getSortButtonText() {
    return getString("button.sort"); //$NON-NLS-1$
  }

  /**
   * Gets the move button text.
   *
   * @return the move button text
   */
  public static String getMoveButtonText() {
    return getString("button.move"); //$NON-NLS-1$
  }
  
  /**
   * Gets the close button text.
   *
   * @return the close button text
   */
  public static String getCloseButtonText() {
    return getString("button.close"); //$NON-NLS-1$
  }

  /**
   * Gets the save button text.
   *
   * @return the save button text
   */
  public static String getSaveButtonText() {
    return getString("button.save"); //$NON-NLS-1$
  }

  /**
   * Gets the all button text.
   *
   * @return the all button text
   */
  public static String getAllButtonText() {
    return getString("button.all"); //$NON-NLS-1$
  }

  /**
   * Gets the none button text.
   *
   * @return the none button text
   */
  public static String getNoneButtonText() {
    return getString("button.none"); //$NON-NLS-1$
  }

  /**
   * Gets the error title.
   *
   * @return the error title
   */
  public static String getErrorTitle() {
    return getString("title.error"); //$NON-NLS-1$
  }

  /**
   * Gets the warning title.
   *
   * @return the warning title
   */
  public static String getWarningTitle() {
    return getString("title.warning"); //$NON-NLS-1$
  }

  /**
   * Gets the information title.
   *
   * @return the information title
   */
  public static String getInformationTitle() {
    return getString("title.information"); //$NON-NLS-1$
  }

  /**
   * Gets the ok option text.
   *
   * @return the ok option text
   */
  public static String getOkOptionText() {
    return getString("option.ok"); //$NON-NLS-1$
  }

  /**
   * Gets the details option text.
   *
   * @return the details option text
   */
  public static String getDetailsOptionText() {
    return getString("option.details"); //$NON-NLS-1$
  }

  /**
   * Gets the yes option text.
   *
   * @return the yes option text
   */
  public static String getYesOptionText() {
    return getString("option.yes"); //$NON-NLS-1$
  }

  /**
   * Gets the no option text.
   *
   * @return the no option text
   */
  public static String getNoOptionText() {
    return getString("option.no"); //$NON-NLS-1$
  }

  /**
   * Gets the cancel option text.
   *
   * @return the cancel option text
   */
  public static String getCancelOptionText() {
    return getString("option.cancel"); //$NON-NLS-1$
  }

}
