/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad;

import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.NoSuchElementException;

import jnpad.text.BufferSet;
import jnpad.text.Buffer;
import jnpad.text.Viewer;
import jnpad.ui.tab.CardTabbedPane;

/**
 * The Class JNPadCardPane.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class JNPadCardPane extends CardTabbedPane implements BufferSet {
  private JNPadCardViewer  viewer;

  /** UID */
  private static final long serialVersionUID = -1555865677176455496L;

  /**
   * Instantiates a new jNPad card pane.
   *
   * @param viewer the viewer
   */
  public JNPadCardPane(JNPadCardViewer viewer) {
    this.viewer = viewer;
  }

  /**
   * Gets the viewer.
   *
   * @return the viewer
   * @see jnpad.text.BufferSet#getViewer()
   */
  @Override
  public Viewer getViewer() {
    return viewer;
  }

  /**
   * Gets the selected buffer.
   *
   * @return the selected buffer
   * @see jnpad.text.BufferSet#getSelectedBuffer()
   */
  @Override
  public Buffer getSelectedBuffer() {
    return (Buffer) getSelectedComponent();
  }

  /**
   * Gets the buffer at.
   *
   * @param index the index
   * @return the buffer at
   * @see jnpad.text.BufferSet#getBufferAt(int)
   */
  @Override
  public Buffer getBufferAt(int index) {
    return (Buffer) getComponentAt(index);
  }
  
  // ViewSet
  
  /**
   * Gets the buffer count.
   *
   * @return the buffer count
   * @see jnpad.text.BufferSet#getBufferCount()
   */
  @Override
  public int getBufferCount() {
    return getTabCount();
  }
  
  /**
   * Sets the active line visible.
   *
   * @param b the new active line visible
   * @see jnpad.text.IView#setActiveLineVisible(boolean)
   */
  @Override
  public void setActiveLineVisible(boolean b) {
    for (int i = 0; i < getTabCount(); i++) {
      getBufferAt(i).setActiveLineVisible(b);
    }
  }

  /**
   * Sets the line numbers visible.
   *
   * @param b the new line numbers visible
   * @see jnpad.text.IView#setLineNumbersVisible(boolean)
   */
  @Override
  public void setLineNumbersVisible(boolean b) {
    for (int i = 0; i < getBufferCount(); i++) {
      getBufferAt(i).setLineNumbersVisible(b);
    }
  }

  /**
   * Sets the line wrap.
   *
   * @param b the new line wrap
   * @see jnpad.text.IView#setLineWrap(boolean)
   */
  @Override
  public void setLineWrap(boolean b) {
    for (int i = 0; i < getBufferCount(); i++) {
      getBufferAt(i).setLineWrap(b);
    }
  }

  /**
   * Sets the right margin line visible.
   *
   * @param b the new right margin line visible
   * @see jnpad.text.IView#setRightMarginLineVisible(boolean)
   */
  @Override
  public void setRightMarginLineVisible(boolean b) {
    for (int i = 0; i < getBufferCount(); i++) {
      getBufferAt(i).setRightMarginLineVisible(b);
    }
  }

  /**
   * Sets the mark strip visible.
   *
   * @param b the new mark strip visible
   * @see jnpad.text.IView#setMarkStripVisible(boolean)
   */
  @Override
  public void setMarkStripVisible(boolean b) {
    for (int i = 0; i < getBufferCount(); i++) {
      getBufferAt(i).setMarkStripVisible(b);
    }
  }

  /**
   * Sets the occurrences highlighter visible.
   *
   * @param b the new occurrences highlighter visible
   * @see jnpad.text.IView#setOccurrencesHighlighterVisible(boolean)
   */
  @Override
  public void setOccurrencesHighlighterVisible(boolean b) {
    for (int i = 0; i < getBufferCount(); i++) {
      getBufferAt(i).setOccurrencesHighlighterVisible(b);
    }
  }
  
  /**
   * Sets the bracket highlighter visible.
   *
   * @param b the new bracket highlighter visible
   * @see jnpad.text.IView#setBracketHighlighterVisible(boolean)
   * @since 0.3
   */
  @Override
  public void setBracketHighlighterVisible(boolean b) {
    for (int i = 0; i < getBufferCount(); i++) {
      getBufferAt(i).setBracketHighlighterVisible(b);
    }
  }

  /**
   * Iterator.
   *
   * @return Iterator
   * @see java.lang.Iterable#iterator()
   */
  @Override
  public Iterator<Buffer> iterator() {
    return new Itr();
  }
  
  /////////////////////////////////////////////////////////////////////////////
  /**
   * The Class Itr.
   */
  private class Itr implements Iterator<Buffer> {
    int pos     = 0;
    int lastRet = -1;

    /**
     * Checks for next.
     *
     * @return boolean
     * @see java.util.Iterator#hasNext()
     */
    @Override
    public boolean hasNext() {
      return pos < getTabCount();
    }

    /**
     * Next.
     *
     * @return the buffer
     * @see java.util.Iterator#next()
     */
    @Override
    public Buffer next() {
      try {
        Buffer next = getBufferAt(pos);
        lastRet = pos++;
        return next;
      }
      catch (IndexOutOfBoundsException e) {
        throw new NoSuchElementException();
      }
    }

    /**
     * Removes the.
     *
     * @see java.util.Iterator#remove()
     */
    @Override
    public void remove() {
      if (lastRet == -1)
        throw new IllegalStateException();
      try {
        removeTabAt(lastRet);
        if (lastRet < pos)
          pos--;
        lastRet = -1;
      }
      catch (IndexOutOfBoundsException e) {
        throw new ConcurrentModificationException();
      }
    }
  }
  /////////////////////////////////////////////////////////////////////////////
  
}
