/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JPopupMenu;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;

import jnpad.action.JNPadAction;
import jnpad.config.Config;
import jnpad.ui.JNPadCheckBoxMenuItem;
import jnpad.ui.JNPadMenuItem;
import jnpad.ui.layout.ModifiedFlowLayout;
import jnpad.ui.plaf.LAFUtils;
import jnpad.ui.toolbar.JNPadToolBar;
import jnpad.ui.toolbar.PopupButton2;
import jnpad.ui.toolbar.PopupToolBoxButton;
import jnpad.ui.toolbar.ToolBarButton;
import jnpad.ui.toolbar.ToolBarToggleButton;

/**
 * The Class JNPadToolBarHelper.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
final class JNPadToolBarHelper {
  JToolBar           toolBar;
  private boolean    setToolbarsEnabled = true;
  private JNPadFrame jNPad;
  
  /**
   * Instantiates a new jNPad tool bar helper.
   *
   * @param jNPad the jNPad's frame
   */
  JNPadToolBarHelper(JNPadFrame jNPad) {
    this.jNPad = jNPad;
    
    if (Config.isDefaultMode()) {
      if (Config.TOOLBAR_ROWS.getValue()) {
        toolBar = new JToolBar();
        toolBar.setLayout(new ModifiedFlowLayout(FlowLayout.LEADING, 0, 0));
      }
      else {
        toolBar = new JNPadToolBar();
      }
    }
    else {
      toolBar = new JToolBar();
    }
    
    toolBar.setFloatable(false);
    
    if (Config.isDefaultMode() || Config.isBasicMode()) {
      toolBar.addMouseListener(new MouseAdapter() {
        @Override
        public void mousePressed(final MouseEvent e) {
          if (e.isPopupTrigger()) {
            showToolBarPopupMenu(e);
          }
        }

        @Override
        public void mouseReleased(final MouseEvent e) {
          if (e.isPopupTrigger()) {
            showToolBarPopupMenu(e);
          }
        }
      });
    }
  }

  /**
   * Show tool bar popup menu.
   *
   * @param e the mouse event
   */
  private void showToolBarPopupMenu(final MouseEvent e) {
    final JNPadActionHelper actions = jNPad.actions;

    JPopupMenu popupMenu = new JPopupMenu();
    popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsFileAction));
    popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsEditRecantationAction));
    popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsEditClipboardAction));
    if (Config.isDefaultMode()) {
      popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsEditSelectionAction));
      popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsEditDeletionAction));
    }
    popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsSearchAction));
    if (Config.isDefaultMode()) {
      popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsViewAction));
      popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsFormatAction));
    }
    popupMenu.add(new JNPadCheckBoxMenuItem(actions.toolbarsWindowAction));
    popupMenu.addSeparator();
    popupMenu.add(new JNPadMenuItem(actions.toolbarsAllVisibleAction));
    popupMenu.add(new JNPadMenuItem(actions.toolbarsNoneVisibleAction));

    SwingUtilities.updateComponentTreeUI(popupMenu);
    popupMenu.pack();
    popupMenu.show(e.getComponent(), e.getX(), e.getY());
  }  

  /**
   * Sets the tool bar visible.
   *
   * @param b the new tool bar visible
   */
  void setToolBarVisible(boolean b) {
    toolBar.setVisible(b);
    JNPadAction.checkSelection(jNPad.actions.toolBarVisibleAction, b);
  }

  /**
   * Checks if is tool bar visible.
   *
   * @return true, if is tool bar visible
   */
  boolean isToolBarVisible() {
    return toolBar.isVisible();
  }

  /**
   * Sets the tool bars.
   */
  void setToolBars() {
    if (!setToolbarsEnabled)
      return;
    switch (Config.getMode()) {
      case MINIMALIST:
      case DISTRACTION_FREE:
        return;
      case BASIC:
        setToolBars_basic();
        break;
      case DEFAULT:
      default:
        setToolBars_default();
        break;
    }
  }
  
  /**
   * Sets the tool bars_basic.
   */
  private void setToolBars_basic() {
    final JNPadActionHelper actions = jNPad.actions;

    Dimension d = null;

    if (LAFUtils.isJNPadLAF()) {
      d = new Dimension(8, 30);
    }

    toolBar.removeAll();

    // Archivo
    if (Config.TOOLBARS_FILE_SELECTED.getValue()) {
      toolBar.add(new ToolBarButton(actions.newFileAction));
      toolBar.add(new ToolBarButton(actions.openFileAction));
      toolBar.add(new ToolBarButton(actions.closeFileAction));
      toolBar.add(new ToolBarButton(actions.saveFileAction));
      toolBar.add(new ToolBarButton(actions.saveAllFilesAction));
      addToolBarSeparator(d);
    }
    // Editar
    if (Config.TOOLBARS_EDIT_RECANTATION_SELECTED.getValue() ||
        Config.TOOLBARS_EDIT_CLIPBOARD_SELECTED.getValue()) {
      if (Config.TOOLBARS_EDIT_RECANTATION_SELECTED.getValue()) {
        toolBar.add(new ToolBarButton(actions.undoAction));
        toolBar.add(new ToolBarButton(actions.redoAction));
      }
      if (Config.TOOLBARS_EDIT_CLIPBOARD_SELECTED.getValue()) {
        toolBar.add(new ToolBarButton(actions.copyAction));
        toolBar.add(new ToolBarButton(actions.cutAction));
        toolBar.add(new ToolBarButton(actions.pasteAction));
      }
      addToolBarSeparator(d);
    }

    // Buscar
    if (Config.TOOLBARS_SEARCH_SELECTED.getValue()) {
      toolBar.add(new ToolBarButton(actions.findAction));
      toolBar.add(new ToolBarButton(actions.replaceAction));
      addToolBarSeparator(d);
    }

    // Ventana
    if (Config.TOOLBARS_WINDOWS_SELECTED.getValue()) {
      toolBar.add(new ToolBarButton(actions.previousBufferAction));
      toolBar.add(new ToolBarButton(actions.nextBufferAction));
    }
    
    try {
      toolBar.doLayout();
      toolBar.validate();
    }
    catch (Exception ex) {
      //ignored
    }
  }

  /**
   * Sets the tool bars_default.
   */
  private void setToolBars_default() {
    final JNPadActionHelper actions = jNPad.actions;

    Dimension d = null;

    if (LAFUtils.isJNPadLAF()) {
      d = new Dimension(8, 23);
    }
    else if (LAFUtils.isWindowsLAF() && Config.TOOLBAR_ROWS.getValue()) {
      d = new Dimension(8, 25);
    }

    toolBar.removeAll();

    // Archivo
    if (Config.TOOLBARS_FILE_SELECTED.getValue()) {
      toolBar.add(new ToolBarButton(actions.newFileAction));
      toolBar.add(new ToolBarButton(actions.openFileAction));
      toolBar.add(jNPad.menus.btRecentFiles);
      toolBar.add(new ToolBarButton(actions.closeFileAction));
      toolBar.add(new ToolBarButton(actions.closeAllAction));
      toolBar.add(new ToolBarButton(actions.saveFileAction));
      toolBar.add(new ToolBarButton(actions.saveFileAsAction));
      toolBar.add(new ToolBarButton(actions.saveAllFilesAction));
      //toolBar.add(new ToolBarButton(actions.printAction));
      addToolBarSeparator(d);
    }

    // Editar
    if (Config.TOOLBARS_EDIT_RECANTATION_SELECTED.getValue() ||
        Config.TOOLBARS_EDIT_CLIPBOARD_SELECTED.getValue() ||
        Config.TOOLBARS_EDIT_SELECTION_SELECTED.getValue() ||
        Config.TOOLBARS_EDIT_DELETION_SELECTED.getValue()) {
      if (Config.TOOLBARS_EDIT_RECANTATION_SELECTED.getValue()) {
        toolBar.add(new ToolBarButton(actions.undoAction));
        toolBar.add(new ToolBarButton(actions.redoAction));
      }
      if (Config.TOOLBARS_EDIT_CLIPBOARD_SELECTED.getValue()) {
        toolBar.add(new ToolBarButton(actions.copyAction));
        toolBar.add(new ToolBarButton(actions.cutAction));
        toolBar.add(new ToolBarButton(actions.pasteAction));
      }
      if (Config.TOOLBARS_EDIT_SELECTION_SELECTED.getValue()) {
        PopupButton2 btSelection = new PopupButton2();
        btSelection.setAction(actions.selectAllAction);
        btSelection.setDropDownToolTipText(JNPadBundle.getString("PopupButton.selection")); //$NON-NLS-1$
        JPopupMenu pmSelection = new JPopupMenu();
        pmSelection.add(new JNPadMenuItem(actions.selectAllAction));
        pmSelection.addSeparator();
        pmSelection.add(new JNPadMenuItem(actions.selectParagraphAction));
        pmSelection.add(new JNPadMenuItem(actions.selectionBeginParagraphAction));
        pmSelection.add(new JNPadMenuItem(actions.selectionEndParagraphAction));
        pmSelection.addSeparator();
        pmSelection.add(new JNPadMenuItem(actions.selectLineAction));
        pmSelection.add(new JNPadMenuItem(actions.selectionBeginLineAction));
        pmSelection.add(new JNPadMenuItem(actions.selectionEndLineAction));
        pmSelection.addSeparator();
        pmSelection.add(new JNPadMenuItem(actions.selectWordAction));
        btSelection.setPopupMenu(pmSelection);
        toolBar.add(btSelection);
      }
      if (Config.TOOLBARS_EDIT_DELETION_SELECTED.getValue()) {
        PopupButton2 btDeletion = new PopupButton2();
        btDeletion.setAction(actions.deleteAction);
        btDeletion.setDropDownToolTipText(JNPadBundle.getString("PopupButton.deletion")); //$NON-NLS-1$
        JPopupMenu pmDeletion = new JPopupMenu();
        pmDeletion.add(new JNPadMenuItem(actions.deleteAction));
        pmDeletion.addSeparator();
        pmDeletion.add(new JNPadMenuItem(actions.deleteParagraphAction));
        pmDeletion.add(new JNPadMenuItem(actions.deletionBeginParagraphAction));
        pmDeletion.add(new JNPadMenuItem(actions.deletionEndParagraphAction));
        pmDeletion.addSeparator();
        pmDeletion.add(new JNPadMenuItem(actions.deleteLineAction));
        pmDeletion.add(new JNPadMenuItem(actions.deletionBeginLineAction));
        pmDeletion.add(new JNPadMenuItem(actions.deletionEndLineAction));
        pmDeletion.addSeparator();
        pmDeletion.add(new JNPadMenuItem(actions.deleteWordAction));
        btDeletion.setPopupMenu(pmDeletion);
        toolBar.add(btDeletion);
      }
      addToolBarSeparator(d);
    }

    // Buscar
    if (Config.TOOLBARS_SEARCH_SELECTED.getValue()) {
      toolBar.add(new ToolBarButton(actions.findAction));
      toolBar.add(new ToolBarButton(actions.replaceAction));
      toolBar.add(new ToolBarButton(actions.nextOccurrenceAction));
      toolBar.add(new ToolBarButton(actions.previousOccurrenceAction));
      toolBar.add(new ToolBarButton(actions.goToLineAction));
      addToolBarSeparator(d);
    }

    // Ver
    if (Config.TOOLBARS_VIEW_SELECTED.getValue()) {
      toolBar.add(new ToolBarToggleButton(actions.lineWrapAction));
      toolBar.add(new ToolBarToggleButton(actions.lineNumbersAction));
      toolBar.add(new ToolBarToggleButton(actions.activeLineAction));
      toolBar.add(new ToolBarToggleButton(actions.rightMarginLineAction));
      toolBar.add(new ToolBarToggleButton(actions.markOccurrencesAction));
      addToolBarSeparator(d);
    }

    // Formato
    if (Config.TOOLBARS_FORMAT_SELECTED.getValue()) {
      PopupToolBoxButton btCharCaseChange = new PopupToolBoxButton(1, 0);
      btCharCaseChange.setDefault(new ToolBarButton(actions.toUpperCaseAction));
      btCharCaseChange.add(btCharCaseChange.getDefault());
      btCharCaseChange.add(new ToolBarButton(actions.toLowerCaseAction));
      btCharCaseChange.add(new ToolBarButton(actions.invertUpperLowerAction));
      btCharCaseChange.add(new ToolBarButton(actions.capitalizeAction));
      btCharCaseChange.add(new ToolBarButton(actions.toTitleAction));
      btCharCaseChange.setDescription(JNPadBundle.getString("PopupButton.charCaseChange")); //$NON-NLS-1$
      toolBar.add(btCharCaseChange);
      toolBar.add(new ToolBarToggleButton(actions.setReadOnlyAction));
      addToolBarSeparator(d);
    }

    // Ventana
    if (Config.TOOLBARS_WINDOWS_SELECTED.getValue()) {
      toolBar.add(new ToolBarButton(actions.previousBufferAction));
      toolBar.add(new ToolBarButton(actions.nextBufferAction));
      toolBar.add(new ToolBarButton(actions.splitHorizontallyAction));
      toolBar.add(new ToolBarButton(actions.splitVerticallyAction));
      toolBar.add(new ToolBarButton(actions.unsplitCurrentAction));
      toolBar.add(new ToolBarButton(actions.unsplitAction));
    }

    try {
      toolBar.doLayout();
      toolBar.revalidate();
      toolBar.validate();
    }
    catch (Exception ex) {
      //ignored
    }
  }

  /**
   * Adds the tool bar separator.
   *
   * @param d the dimension
   */
  private void addToolBarSeparator(Dimension d) {
    if (d != null)
      toolBar.addSeparator(d);
    else
      toolBar.addSeparator();
  }

  /**
   * View tool bars visible.
   *
   * @param b the boolean
   */
  void viewToolBarsVisible(boolean b) {
    switch (Config.getMode()) {
      case MINIMALIST:
      case DISTRACTION_FREE:
        return;
      case BASIC:
        viewToolBarsVisible_basic(b);
        break;
      case DEFAULT:
      default:
        viewToolBarsVisible_default(b);
        break;
    }
  }
  
  /**
   * View tool bars visible_default.
   *
   * @param b the boolean
   */
  private void viewToolBarsVisible_default(boolean b) {
    final JNPadActionHelper actions = jNPad.actions;

    setToolbarsEnabled = false;
    if (actions.toolbarsFileAction.isSelected() != b)
      actions.toolbarsFileAction.setSelected(b);
    if (actions.toolbarsEditRecantationAction.isSelected() != b)
      actions.toolbarsEditRecantationAction.setSelected(b);
    if (actions.toolbarsEditClipboardAction.isSelected() != b)
      actions.toolbarsEditClipboardAction.setSelected(b);
    if (actions.toolbarsEditSelectionAction.isSelected() != b)
      actions.toolbarsEditSelectionAction.setSelected(b);
    if (actions.toolbarsEditDeletionAction.isSelected() != b)
      actions.toolbarsEditDeletionAction.setSelected(b);
    if (actions.toolbarsSearchAction.isSelected() != b)
      actions.toolbarsSearchAction.setSelected(b);
    if (actions.toolbarsViewAction.isSelected() != b)
      actions.toolbarsViewAction.setSelected(b);
    if (actions.toolbarsFormatAction.isSelected() != b)
      actions.toolbarsFormatAction.setSelected(b);
    if (actions.toolbarsWindowAction.isSelected() != b)
      actions.toolbarsWindowAction.setSelected(b);
    setToolbarsEnabled = true;

    setToolBars();
  }

  /**
   * View tool bars visible_basic.
   *
   * @param b the boolean
   */
  private void viewToolBarsVisible_basic(boolean b) {
    final JNPadActionHelper actions = jNPad.actions;

    setToolbarsEnabled = false;
    if (actions.toolbarsFileAction.isSelected() != b)
      actions.toolbarsFileAction.setSelected(b);
    if (actions.toolbarsEditRecantationAction.isSelected() != b)
      actions.toolbarsEditRecantationAction.setSelected(b);
    if (actions.toolbarsEditClipboardAction.isSelected() != b)
      actions.toolbarsEditClipboardAction.setSelected(b);
    if (actions.toolbarsSearchAction.isSelected() != b)
      actions.toolbarsSearchAction.setSelected(b);
    if (actions.toolbarsWindowAction.isSelected() != b)
      actions.toolbarsWindowAction.setSelected(b);
    setToolbarsEnabled = true;

    setToolBars();
  }
  
}
