/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.action;

import static jnpad.action.JNPadActions.ACTION_NAME_ABOUT;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import jnpad.GUIUtilities;
import jnpad.JNPad;
import jnpad.JNPadBundle;
import jnpad.JNPadFrame;
import jnpad.action.JNPadActions.Group;
import jnpad.config.Accelerators;
import jnpad.ui.EscapableDialog;
import jnpad.ui.JNPadLabel;
import jnpad.ui.layout.VerticalFlowLayout;
import jnpad.ui.plaf.LAFUtils;
import jnpad.ui.status.StatusDisplayable.StatusType;
import jnpad.util.Version;

/**
 * The Class AboutAction.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class AboutAction extends JNPadAction {
  /** UID */
  private static final long serialVersionUID = 6360511321392656795L;

  /**
   * Instantiates a new <code>AboutAction</code>.
   *
   * @param jNPad the jNPad's frame
   */
  public AboutAction(JNPadFrame jNPad) {
    super(jNPad,
          ACTION_NAME_ABOUT,
          Group.HELP,
          Accelerators.ABOUT, "about.png"); //$NON-NLS-1$
  }

  /**
   * Perform action.
   *
   * @see jnpad.action.JNPadAction#performAction()
   */
  @Override
  public void performAction() {
    new AboutDialog(jNPad);
  }

}

////////////////////////////////////////////////////////////////////////////////
/**
 * The Class AboutDialog.
 */
class AboutDialog extends EscapableDialog {
  JPanel                      contentPane;
  JPanel                      pnBackground     = new JPanel();
  JPanel                      pnIcon           = new JPanel();
  JPanel                      pnLabels         = new JPanel();
  JPanel                      pnClose          = new JPanel();
  JButton                     btClose          = new JButton();
  JLabel                      lbIcon           = new JNPadLabel();
  JLabel                      lnProduct        = new JNPadLabel();

  JLabel                      lbVersion        = new JNPadLabel();
  JLabel                      lbCopyright      = new JNPadLabel();
  JLabel                      lbEmail          = new JNPadLabel();
  JLabel                      lbSourceForge    = new JNPadLabel();
  
  private JNPadFrame          jNPad;

  /** Logger */
  private static final Logger LOGGER           = Logger.getLogger(AboutDialog.class.getName());

  /** UID */
  private static final long   serialVersionUID = 2305689107733217463L;

  /**
   * Instantiates a new about dialog.
   *
   * @param jNPad the jNPad's frame
   */
  AboutDialog(JNPadFrame jNPad) {
    super(jNPad, ActionBundle.getString(ACTION_NAME_ABOUT.concat(".title")), true); //$NON-NLS-1$
    try {
      this.jNPad = jNPad;
      jbInit();
      pack();
      setLocationRelativeTo(jNPad);
      setVisible(true);
    }
    catch (Exception ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
      jNPad.setStatus(StatusType.ERROR, "Error - ".concat(getTitle())); //$NON-NLS-1$
    }
  }

  /**
   * Component initialization.
   *
   * @throws Exception the exception
   */
  private void jbInit() throws Exception {
    contentPane = (JPanel)this.getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(pnBackground, BorderLayout.NORTH);
    contentPane.add(pnClose, BorderLayout.SOUTH);

    final MouseAdapter hyperlinkMouseListener = new MouseAdapter() {
      @Override
      public void mousePressed(final MouseEvent e) {
        Object obj = e.getSource();
        if (obj == lbEmail) {
          open("mailto:rgsevero@gmail.com?subject=jNPad"); //$NON-NLS-1$
        }
        else if (obj == lbSourceForge) {
          open("http://sourceforge.net/projects/jnpad"); //$NON-NLS-1$
        }
      }

      @Override
      public void mouseEntered(final MouseEvent e) {
        e.getComponent().setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
      }

      @Override
      public void mouseExited(final MouseEvent e) {
        e.getComponent().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
      }
    };
    
    lnProduct.setIcon(GUIUtilities.loadIcon(ActionBundle.getString(ACTION_NAME_ABOUT.concat(".product")), false)); //$NON-NLS-1$
    lbVersion.setHorizontalAlignment(SwingConstants.TRAILING);
    lbVersion.setText(ActionBundle.getString(ACTION_NAME_ABOUT.concat(".version"), Version.getVersion())); //$NON-NLS-1$
    lbCopyright.setHorizontalAlignment(SwingConstants.TRAILING);
    lbCopyright.setText("Copyright  2014-2017  rgs"); //$NON-NLS-1$

    if(LAFUtils.isDarkLAF()) {
      lbSourceForge.setText("<html><a href=\"sourceforge.net/projects/jnpad\"><font color=\"#00CCFF\">http://sourceforge.net/projects/jnpad</font></a></html>"); //$NON-NLS-1$
      lbEmail.setText("<html><a href=\"mailto:" + JNPad.EMAIL + "\"><font color=\"#00CCFF\">" + JNPad.EMAIL + "</font></a></html>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    }
    else {
      lbSourceForge.setText("<html><a href=\"http://sourceforge.net/projects/jnpad\">http://sourceforge.net/projects/jnpad</a></html>"); //$NON-NLS-1$
      lbEmail.setText("<html><a href=\"mailto:" + JNPad.EMAIL + "\">" + JNPad.EMAIL + "</a></html>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    }

    lbEmail.setToolTipText(JNPad.EMAIL);
    lbEmail.setHorizontalAlignment(SwingConstants.TRAILING);
    lbEmail.addMouseListener(hyperlinkMouseListener);

    lbSourceForge.setIcon(GUIUtilities.loadIcon("sourceforge.png")); //$NON-NLS-1$
    lbSourceForge.setToolTipText("http://sourceforge.net/projects/jnpad"); //$NON-NLS-1$
    lbSourceForge.setHorizontalAlignment(SwingConstants.TRAILING);
    lbSourceForge.addMouseListener(hyperlinkMouseListener);
    
    lbIcon.setIcon(GUIUtilities.loadIcon("icon-jnpad48.png", false)); //$NON-NLS-1$
    pnIcon.setBorder(GUIUtilities.createEmptyBorder(5));
    pnIcon.add(lbIcon, null);

    pnLabels.setLayout(new VerticalFlowLayout());
    pnLabels.setBorder(GUIUtilities.createEmptyBorder(5));
    pnLabels.add(lnProduct, null);
    pnLabels.add(lbVersion, null);
    pnLabels.add(lbCopyright, null);
    pnLabels.add(lbEmail, null);
    pnLabels.add(lbSourceForge, null);

    pnBackground.setLayout(new BorderLayout());
    pnBackground.add(pnIcon, BorderLayout.WEST);
    pnBackground.add(pnLabels, BorderLayout.CENTER);

    pnClose.setLayout(new FlowLayout(FlowLayout.TRAILING));
    pnClose.add(btClose, null);

    GUIUtilities.setLocalizedText(btClose, JNPadBundle.getCloseButtonText());

    btClose.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        dispose();
      }
    });

    setResizable(false);

    setDefaultCloseOperation(DISPOSE_ON_CLOSE);
  }

  /**
   * Escape pressed.
   *
   * @see jnpad.ui.EscapableDialog#escapePressed()
   */
  @Override
  protected void escapePressed() {
    dispose();
  }
  
  /**
   * Open.
   *
   * @param str the string
   */
  private void open(String str) {
    try {
      GUIUtilities.openBrowser(str);
    }
    catch (Exception ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
      jNPad.setStatus(StatusType.ERROR, ActionBundle.getString(ACTION_NAME_ABOUT.concat(".notSupported"))); //$NON-NLS-1$
    }
  }
}
////////////////////////////////////////////////////////////////////////////////
