/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.action;

import java.awt.event.ItemEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JOptionPane;

import jnpad.JNPadBundle;
import jnpad.JNPadFrame;
import jnpad.action.JNPadActions.Group;
import jnpad.config.Config;
import jnpad.ui.MnemonicHelper;
import jnpad.ui.status.StatusDisplayable;
import jnpad.util.Utilities;

/**
 * The Class SelectKeymapAction.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class SelectKeymapAction extends JNPadAction {
  static final String                     NAME             = "select-keymap";              //$NON-NLS-1$

  private static Map<String, JNPadAction> map              = new HashMap<String, JNPadAction>();

  /** UID */
  private static final long               serialVersionUID = -9032300359975558889L;

  /**
   * Instantiates a new select keymap action.
   *
   * @param jNPad the jNPad's frame
   * @param file the file
   * @param selected the selected
   * @param mnemonicHelper the mnemonic helper
   */
  public SelectKeymapAction(JNPadFrame jNPad, File file, boolean selected, MnemonicHelper mnemonicHelper) {
    super(jNPad, NAME, Group.CONFIGURE);

    final String label = Utilities.getFileBaseName(file.getPath());
    final String name  = NAME + Utilities.SPACE_STRING + label;
    
    setLabel(label);
    setName(name);

    if (mnemonicHelper != null) {
      char mnemonic = mnemonicHelper.getMnemonic(label);
      if (mnemonic != 0) {
        setMnemonic(mnemonic);
      }
    }

    setDescription(ActionBundle.getString(NAME.concat(".description"), label)); //$NON-NLS-1$

    setSelected(selected);
    setStateAction(true);

    map.put(name, this);
  }

  /**
   * Handle state changed.
   *
   * @param e ItemEvent
   * @see jnpad.action.JNPadAction#handleStateChanged(java.awt.event.ItemEvent)
   */
  @Override
  public void handleStateChanged(ItemEvent e) {
    if (ItemEvent.SELECTED == e.getStateChange()) {
      Config.USE_CUSTOM_SHORTCUTS.setValue(!Config.JNPAD_KEYMAP.getDefault().equals(getLabel()));
      Config.JNPAD_KEYMAP.setValue(getLabel());

      String msg = ActionBundle.getString("select-keymap.message", getLabel()); //$NON-NLS-1$
      jNPad.setStatus(msg, StatusDisplayable.TIMEOUT_DEFAULT);

      String[] options = { JNPadBundle.getYesOptionText(), JNPadBundle.getNoOptionText() };
      int option = JOptionPane.showOptionDialog(jNPad,
                                                JNPadBundle.getString("JNPad.restart", msg), //$NON-NLS-1$
                                                JNPadBundle.getString("select-keymap.title"), //$NON-NLS-1$
                                                JOptionPane.DEFAULT_OPTION,
                                                JOptionPane.INFORMATION_MESSAGE,
                                                null,
                                                options,
                                                options[0]);
      if (option == 1 || option == JOptionPane.CLOSED_OPTION)
        return;
      
      jNPad.restart();
    }
  }

  /**
   * Gets the completions for prefix.
   *
   * @param prefix the prefix
   * @return the completions for prefix
   */
  static List<String> getCompletionsForPrefix(String prefix) {
    List<String> list = new ArrayList<String>();
    for (String key : map.keySet()) {
      JNPadAction action = map.get(key);
      if (key.startsWith(prefix) && action.isEnabled()) {
        list.add(key);
      }
    }
    Collections.sort(list);
    return list;
  }
  
  /**
   * Perform action.
   *
   * @param name the name
   * @return true, if successful
   */
  static boolean performAction(String name) {
    JNPadAction action = map.get(name);
    if (action != null && action.isEnabled()) {
      action.setSelected(!action.isSelected());
      return true;
    }
    return false;
  }
  
}
