/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.action;

import java.awt.event.ItemEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.JOptionPane;
import javax.swing.UIManager;

import jnpad.JNPadBundle;
import jnpad.JNPadFrame;
import jnpad.action.JNPadActions.Group;
import jnpad.config.Config;
import jnpad.ui.MnemonicHelper;
import jnpad.ui.status.StatusDisplayable;
import jnpad.util.Utilities;

/**
 * The Class SelectLAFAction.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class SelectLAFAction extends JNPadAction {
  private UIManager.LookAndFeelInfo       info;

  private String                          lafKey;

  static final String                     NAME             = "select-laf";              //$NON-NLS-1$

  private static Map<String, JNPadAction> map              = new HashMap<String, JNPadAction>();

  /** UID */
  private static final long               serialVersionUID = -7427981107412256683L;

  /**
   * Instantiates a new <code>SelectLAFAction</code>.
   *
   * @param jNPad the jNPad's frame
   * @param label the label
   * @param lafKey the laf key
   * @param selected the selected
   * @param mnemonicHelper the mnemonic helper
   */
  public SelectLAFAction(JNPadFrame jNPad, String label, String lafKey, boolean selected, MnemonicHelper mnemonicHelper) {
    super(jNPad, NAME, Group.CONFIGURE);
    this.lafKey = lafKey;

    setLabel(label);

    final String name  = NAME + Utilities.SPACE_STRING + label;
    setName(name);

    if (mnemonicHelper != null) {
      char mnemonic = mnemonicHelper.getMnemonic(label);
      if (mnemonic != 0) {
        setMnemonic(mnemonic);
      }
    }

    setDescription(ActionBundle.getString(NAME.concat(".description"), label)); //$NON-NLS-1$

    setSelected(selected);
    setStateAction(true);

    map.put(name, this);
  }

  /**
   * Instantiates a new <code>SelectLAFAction</code>.
   *
   * @param jNPad the jNPad's frame
   * @param info the info
   * @param selected the selected
   * @param mnemonicHelper the mnemonic helper
   */
  public SelectLAFAction(JNPadFrame jNPad, UIManager.LookAndFeelInfo info, boolean selected, MnemonicHelper mnemonicHelper) {
    super(jNPad, NAME, Group.CONFIGURE);
    this.info = info;

    final String label = info.getName();
    final String name  = NAME + Utilities.SPACE_STRING + label;

    setLabel(label);
    setName(name);

    if (mnemonicHelper != null) {
      char mnemonic = mnemonicHelper.getMnemonic(label);
      if (mnemonic != 0) {
        setMnemonic(mnemonic);
      }
    }

    setDescription(ActionBundle.getString(NAME.concat(".description"), label)); //$NON-NLS-1$

    setSelected(selected);
    setStateAction(true);

    map.put(name, this);
  }

  /**
   * Handle state changed.
   *
   * @param e the item event
   * @see jnpad.action.JNPadAction#handleStateChanged(java.awt.event.ItemEvent)
   */
  @Override
  public void handleStateChanged(final ItemEvent e) {
    if (ItemEvent.SELECTED == e.getStateChange()) {
      Config.JNPAD_LAF.setValue(info != null ? info.getClassName() : lafKey);
      Config.setAdjustPostLAFChangedEnabled(true);

      String msg = ActionBundle.getString("select-laf.message", getLabel()); //$NON-NLS-1$
      jNPad.setStatus(msg, StatusDisplayable.TIMEOUT_DEFAULT);

      String[] options = { JNPadBundle.getYesOptionText(), JNPadBundle.getNoOptionText() };
      int option = JOptionPane.showOptionDialog(jNPad,
                                                JNPadBundle.getString("JNPad.restart", msg), //$NON-NLS-1$
                                                JNPadBundle.getString("select-laf.title"), //$NON-NLS-1$
                                                JOptionPane.DEFAULT_OPTION,
                                                JOptionPane.INFORMATION_MESSAGE,
                                                null,
                                                options,
                                                options[0]);
      if (option == 1 || option == JOptionPane.CLOSED_OPTION)
        return;
      
      jNPad.restart();
    }
  }

  /**
   * Gets the completions for prefix.
   *
   * @param prefix the prefix
   * @return the completions for prefix
   */
  static List<String> getCompletionsForPrefix(String prefix) {
    List<String> list = new ArrayList<String>();
    for (String key : map.keySet()) {
      JNPadAction action = map.get(key);
      if (key.startsWith(prefix) && action.isEnabled()) {
        list.add(key);
      }
    }
    Collections.sort(list);
    return list;
  }
  
  /**
   * Perform action.
   *
   * @param name the name
   * @return true, if successful
   */
  static boolean performAction(String name) {
    JNPadAction action = map.get(name);
    if (action != null && action.isEnabled()) {
      action.setSelected(!action.isSelected());
      return true;
    }
    return false;
  }
  
}
