/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.config;

import java.awt.Color;
import java.util.StringTokenizer;

/**
 * The Class ColorProp.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class ColorProp extends Prop<Color> {
  /** UID */
  private static final long serialVersionUID = -6182268127019767746L;

  /**
   * Instantiates a new color prop.
   *
   * @param name the name
   * @param def the def
   */
  public ColorProp(String name, Color def) {
    super(name, def);
  }

  /**
   * Gets the default string.
   *
   * @return the default string
   * @see jnpad.config.Prop#getDefaultString()
   */
  @Override
  public String getDefaultString() {
    return format(defaultValue);
  }

  /**
   * Parses the.
   *
   * @param value the value
   * @return the color
   * @throws IllegalArgumentException the illegal argument exception
   */
  public static Color parse(String value) throws IllegalArgumentException {
    value = value.trim();

    if (value.charAt(0) == '#') {
      try {
        return Color.decode(value);
      }
      catch (NumberFormatException nf) {
        throw new IllegalArgumentException("Color invlido: " + value); //$NON-NLS-1$
      }
    }

    if ("red".equals(value))       return Color.RED;        //$NON-NLS-1$
    if ("green".equals(value))     return Color.GREEN;      //$NON-NLS-1$
    if ("blue".equals(value))      return Color.BLUE;       //$NON-NLS-1$
    if ("yellow".equals(value))    return Color.YELLOW;     //$NON-NLS-1$
    if ("orange".equals(value))    return Color.ORANGE;     //$NON-NLS-1$
    if ("white".equals(value))     return Color.WHITE;      //$NON-NLS-1$
    if ("lightGray".equals(value)) return Color.LIGHT_GRAY; //$NON-NLS-1$
    if ("gray".equals(value))      return Color.GRAY;       //$NON-NLS-1$
    if ("darkGray".equals(value))  return Color.DARK_GRAY;  //$NON-NLS-1$
    if ("black".equals(value))     return Color.BLACK;      //$NON-NLS-1$
    if ("cyan".equals(value))      return Color.CYAN;       //$NON-NLS-1$
    if ("magenta".equals(value))   return Color.MAGENTA;    //$NON-NLS-1$
    if ("pink".equals(value))      return Color.PINK;       //$NON-NLS-1$

    int r, g, b, a;
    try {
      StringTokenizer st = new StringTokenizer(value, ","); //$NON-NLS-1$
      r = IntProp.parse(st.nextToken().trim());
      g = IntProp.parse(st.nextToken().trim());
      b = IntProp.parse(st.nextToken().trim());
      a = st.hasMoreTokens() ? IntProp.parse(st.nextToken().trim()) : 255;
    }
    catch (Exception ex) {
      throw new IllegalArgumentException("Color invlido: " + value); //$NON-NLS-1$
    }

    return new Color(r, g, b, a);
  }

  /**
   * Format.
   *
   * @param value the value
   * @return the string
   */
  public static String format(Color value) {
    StringBuilder sb = new StringBuilder();
    sb.append(value.getRed()).append(","). //$NON-NLS-1$
       append(value.getGreen()).append(","). //$NON-NLS-1$
       append(value.getBlue()).append(","). //$NON-NLS-1$
       append(value.getAlpha());
    return sb.toString();
  }

  /**
   * Gets the value.
   *
   * @return the value
   * @see jnpad.config.Prop#getValue()
   */
  @Override
  public Color getValue() {
    return Config.getColorProp(this);
  }

  /**
   * Do set value.
   *
   * @param value the value
   * @see jnpad.config.Prop#doSetValue(java.lang.Object)
   */
  @Override
  protected void doSetValue(Color value) {
    Config.setColorProp(this, value);
  }
  
}
