/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.syntax;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;

import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.Segment;
import javax.swing.text.Utilities;

import jnpad.text.JNPadDocument;
import jnpad.text.syntax.CScheme.WordType;

/**
 * The Class CView.
 * 
 * @version 0.3
 * @since jNPad 0.1
 */
public class CView extends SyntaxView {
  CScheme scheme;

  /**
   * Instantiates a new c view.
   *
   * @param elem the elem
   * @param mini the mini
   */
  public CView(Element elem, boolean mini) {
    super(elem, mini);
    scheme = !mini ? (CScheme) ((JNPadDocument) getDocument()).getScheme() :
        (CScheme) ((JNPadDocument) getDocument()).getMiniScheme();
  }
  
  /**
   * Do draw unselected text.
   *
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawUnselectedText(java.awt.Graphics, int, int, int, int)
   */
  @Override
  int doDrawUnselectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    boolean inComment = isParagraphInComment(start);

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (Character.isJavaIdentifierStart(indexedChar) && !inComment) {
        String scannedIdentifier = scanIdentifier(content, wordIndex);
        int scannedIdentifierLength = scannedIdentifier.length();
        
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType(scannedIdentifier);
        g.setColor(scheme.getWordColor(type));
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;

        continue;
      }
      else if (Character.isDigit(indexedChar) && !inComment) {
        String scannedNumericLiteral = scanNumericLiteral(content, wordIndex);
        int scannedNumericLiteralLength = scannedNumericLiteral.length();

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedNumericLiteralLength, text);
        g.setColor(scheme.getNumberColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        
        wordIndex += scannedNumericLiteralLength;

        continue;
      }
      else if (isStringDelimiter(indexedChar) && !inComment) {
        int scannedStringLength = scanStringLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedStringLength, text);
        final Font savedFont = g.getFont();
        g.setColor(scheme.getStringColor());
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex += scannedStringLength;
        
        continue;
      }
      else if (isCharDelimiter(indexedChar) && !inComment) {
        int scannedCharLength = scanCharLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCharLength, text);
        g.setColor(scheme.getStringColor());
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);

        wordIndex += scannedCharLength;
        
        continue;
      }
      else if (isDirectiveStart(content, wordIndex) && !inComment &&
          (wordIndex < content.length() - 1)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex + 1);
        int scannedIdentifierLength = scannedIdentifier.length() + 1;

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType("#".concat(scannedIdentifier)); //$NON-NLS-1$
        g.setColor(scheme.getWordColor(type));
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;
        
        continue;
      }
      else if (isSingleLineCommentStart(content, wordIndex) && !inComment) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setColor(scheme.getCommentColor());
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isMultiLineCommentEnd(content, wordIndex) && inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 2, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        g.setColor(scheme.getCommentColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = false;
        wordIndex += 2;
        
        continue;
      }
      else if (isMultiLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanMultiLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        g.setColor(scheme.getCommentColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = true;
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isBracket(indexedChar) && !inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getBracketFont());
        g.setColor(scheme.getBracketColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;
        
        continue;
      }
      else if (isOperator(indexedChar) && !inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        g.setColor(scheme.getOperatorColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;
        
        continue;
      }
      
      final Font savedFont = g.getFont();
      Color color;
      if (inComment) {
        g.setFont(scheme.getCommentFont());
        color = scheme.getCommentColor();
      }
      else {
        color = scheme.getTextColor();
      }
      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      g.setColor(color);
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
      g.setFont(savedFont);
      
      wordIndex++;
    }

    return x;
  }
  
  /**
   * Do draw selected text.
   *
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawSelectedText(java.awt.Graphics, int, int, int, int)
   */
  @Override
  int doDrawSelectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    boolean inComment = isParagraphInComment(start);

    g.setColor(scheme.getSelectedTextColor());

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (Character.isJavaIdentifierStart(indexedChar) && !inComment) {
        String scannedIdentifier = scanIdentifier(content, wordIndex);
        int scannedIdentifierLength = scannedIdentifier.length();
        
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType(scannedIdentifier);
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;

        continue;
      }
      else if (isStringDelimiter(indexedChar) && !inComment) {
        int scannedStringLength = scanStringLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedStringLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex += scannedStringLength;
        
        continue;
      }
      else if (isCharDelimiter(indexedChar) && !inComment) {
        int scannedCharLength = scanCharLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCharLength, text);
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);

        wordIndex += scannedCharLength;
        
        continue;
      }
      else if (isDirectiveStart(content, wordIndex) && !inComment &&
          (wordIndex < content.length() - 1)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex + 1);
        int scannedIdentifierLength = scannedIdentifier.length() + 1;

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType("#".concat(scannedIdentifier)); //$NON-NLS-1$
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;
        
        continue;
      }
      else if (isSingleLineCommentStart(content, wordIndex) && !inComment) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isMultiLineCommentEnd(content, wordIndex) && inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 2, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = false;
        wordIndex += 2;
        
        continue;
      }
      else if (isMultiLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanMultiLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = true;
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isBracket(indexedChar) && !inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getBracketFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;

        continue;
      }
      else if (isOperator(indexedChar) && !inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;
        
        continue;
      }
      
      final Font savedFont = g.getFont();
      if (inComment) {
        g.setFont(scheme.getCommentFont());
      }
      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
      g.setFont(savedFont);
      
      wordIndex++;
    }

    return x;
  }
  
  /**
   * Checks if is paragraph in comment.
   *
   * @param startOffset the start offset
   * @return true, if is paragraph in comment
   */
  boolean isParagraphInComment(int startOffset) {
    if (startOffset > 0) {
      Element root = getDocument().getDefaultRootElement();
      Element previousElement = root.getElement(root.getElementIndex(startOffset - 1));
      if (previousElement.getAttributes().containsAttribute(JNPadDocument.MULTILINE_COMMENT, JNPadDocument.MULTILINE_COMMENT))
        return true;
    }
    return false;
  }

  /**
   * Checks if is single line comment start.
   *
   * @param s the s
   * @param index the index
   * @return true, if is single line comment start
   * @see jnpad.text.syntax.SyntaxView#isSingleLineCommentStart(javax.swing.text.Segment, int)
   */
  @Override
  boolean isSingleLineCommentStart(Segment s, int index) {
    return s.charAt(index) == '/' && index + 1 < s.length() && s.charAt(index + 1) == '/';
  }
  
  /**
   * Checks if is directive start.
   *
   * @param s the s
   * @param index the index
   * @return true, if is directive start
   */
  boolean isDirectiveStart(Segment s, int index) {
    return (s.charAt(index) == '#');
  }
  
  /**
   * Checks if is multi line comment start.
   *
   * @param s the s
   * @param index the index
   * @return true, if is multi line comment start
   * @see jnpad.text.syntax.SyntaxView#isMultiLineCommentStart(javax.swing.text.Segment, int)
   */
  @Override
  boolean isMultiLineCommentStart(Segment s, int index) {
    return s.charAt(index) == '/' && index + 1 < s.length() && s.charAt(index + 1) == '*';
  }

  /**
   * Checks if is multi line comment end.
   *
   * @param s the s
   * @param index the index
   * @return true, if is multi line comment end
   * @see jnpad.text.syntax.SyntaxView#isMultiLineCommentEnd(javax.swing.text.Segment, int)
   */
  @Override
  boolean isMultiLineCommentEnd(Segment s, int index) {
    return s.charAt(index) == '*' && index + 1 < s.length() && s.charAt(index + 1) == '/';
  }
  
  /**
   * Scan multi line comment.
   *
   * @param s the s
   * @param index the index
   * @return the int
   * @see jnpad.text.syntax.SyntaxView#scanMultiLineComment(javax.swing.text.Segment, int)
   */
  @Override
  int scanMultiLineComment(Segment s, int index) {
    int commentLength;
    boolean starFound = false;
    for (commentLength = 0; commentLength < (s.length() - index); commentLength++) {
      char commentChar = s.charAt(index + commentLength);
      if (starFound && commentChar == '/') {
        commentLength++;
        break;
      }
      starFound = false;
      if (commentChar == jnpad.util.Utilities.LF)
        break;
      else if (commentChar == '*')
        starFound = true;
    }
    return commentLength;
  }  

  /**
   * Gets the scheme.
   *
   * @return the scheme
   * @see jnpad.text.syntax.SyntaxView#getScheme()
   */
  @Override
  SyntaxScheme getScheme() {
    return scheme;
  }

}
