/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.plaf;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Rectangle;

import javax.swing.AbstractButton;
import javax.swing.ButtonModel;
import javax.swing.JComponent;
import javax.swing.JToggleButton;
import javax.swing.SwingUtilities;
import javax.swing.border.Border;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.basic.BasicHTML;
import javax.swing.plaf.metal.MetalButtonUI;
import javax.swing.text.View;

import jnpad.GUIUtilities;
import jnpad.util.Utilities;

/**
 * The Class JNPadToolBarButtonUI.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class JNPadToolBarButtonUI extends MetalButtonUI {
  private static final Border               EMPTY_BORDER = GUIUtilities.createEmptyBorder(5);

  private static Rectangle                  viewRect     = new Rectangle();
  private static Rectangle                  textRect     = new Rectangle();
  private static Rectangle                  iconRect     = new Rectangle();

  private Border                            oldBorder;
  private boolean                           oldRolloverEnabled;
  private boolean                           oldFocusable;

  private static Color                      selectedColor1;
  private static Color                      selectedColor2;
  private static Color                      borderColor1;
  private static Color                      borderColor2;
  
  /** The Constant INSTANCE. */
  private static final JNPadToolBarButtonUI INSTANCE     = new JNPadToolBarButtonUI();

  /**
   * 
   * @param c JComponent
   * @return ComponentUI
   */
  public static ComponentUI createUI(JComponent c) {
    return INSTANCE;
  }

  /**
   * 
   * @param b AbstractButton
   * @see javax.swing.plaf.metal.MetalButtonUI#installDefaults(javax.swing.AbstractButton)
   */
  @Override
  public void installDefaults(AbstractButton b) {
    super.installDefaults(b);

    // Save current...
    oldBorder = b.getBorder();
    oldRolloverEnabled = b.isRolloverEnabled();
    oldFocusable = b.isFocusable();

    b.setBorder(EMPTY_BORDER);
    b.setRolloverEnabled(true);
    b.setFocusable(false);
  }

  /**
   * 
   * @param b AbstractButton    
   * @see javax.swing.plaf.metal.MetalButtonUI#uninstallDefaults(javax.swing.AbstractButton)
   */
  @Override
  public void uninstallDefaults(AbstractButton b) {
    super.uninstallDefaults(b);

    // Restore old...
    b.setBorder(oldBorder);
    b.setRolloverEnabled(oldRolloverEnabled);
    b.setFocusable(oldFocusable);
  }
  

  /**
   * 
   * @param g Graphics
   * @param c JComponent
   */
  @Override
  public void paint(Graphics g, JComponent c) {
    AbstractButton b = (AbstractButton) c;
    //b.setRolloverEnabled(true);
    ButtonModel model = b.getModel();

    FontMetrics fm = g.getFontMetrics();

    Insets i = c.getInsets();

    viewRect.x = i.left;
    viewRect.y = i.top;
    viewRect.width = b.getWidth() - (i.right + viewRect.x);
    viewRect.height = b.getHeight() - (i.bottom + viewRect.y);

    textRect.x = textRect.y = textRect.width = textRect.height = 0;
    iconRect.x = iconRect.y = iconRect.width = iconRect.height = 0;

    Font f = c.getFont();
    g.setFont(f);

    // layout the text and icon
    String text = SwingUtilities.layoutCompoundLabel(
          c, fm, b.getText(), b.getIcon(),
          b.getVerticalAlignment(), b.getHorizontalAlignment(),
          b.getVerticalTextPosition(), b.getHorizontalTextPosition(),
          viewRect, iconRect, textRect,
          b.getText() == null ? 0 : defaultTextIconGap);

    clearTextShiftOffset();
    
    if (model.isArmed() && model.isPressed() || model.isSelected()) {
      g.setColor(b instanceof JToggleButton ? selectedColor2() : selectedColor1());
      g.fillRect(0, 0, c.getSize().width, c.getSize().height);
      doPaintIcon(g, b, iconRect);
      GUIUtilities.drawBorder(g, borderColor1(), 0, 0, c.getSize().width, c.getSize().height);
    }
    else if (model.isRollover()) {
      g.setColor(JNPadMenuUI.topLevelMenuBg());
      g.fillRect(0, 0, c.getSize().width, c.getSize().height);
      doPaintIcon(g, b, iconRect);
      GUIUtilities.drawBorder(g, borderColor2(), 0, 0, c.getSize().width, c.getSize().height);
    }
    else {
      doPaintIcon(g, b, iconRect);
    }

    if (Utilities.isNotEmptyString(text)) {
      View v = (View) c.getClientProperty(BasicHTML.propertyKey);
      if (v != null) {
        v.paint(g, textRect);
      }
      else {
        paintText(g, c, textRect, text);
      }
    }
    
    if (b.isFocusPainted() && b.hasFocus()) {
      // paint UI specific focus
      paintFocus(g, b, viewRect, textRect, iconRect);
    }
  }
  
  /**
   * 
   * @param g Graphics
   * @param b AbstractButton
   * @param iconRect Rectangle
   */
  private void doPaintIcon(Graphics g, AbstractButton b, Rectangle iconRect) {
    if (b.getIcon() != null) {
      paintIcon(g, b, iconRect);
    }
  }

  /**
   * Selected color1.
   *
   * @return the color
   */
  private static Color selectedColor1() {
    if (selectedColor1 == null) {
      selectedColor1 = LAFUtils.isDarkLAF() ? JNPadDarkMetalTheme.WHITE : JNPadLightMetalTheme.LIGHT_GRAY_1;
    }
    return selectedColor1;
  }

  /**
   * Selected color2.
   *
   * @return the color
   */
  private static Color selectedColor2() {
    if (selectedColor2 == null) {
      selectedColor2 = LAFUtils.isDarkLAF() ? JNPadDarkMetalTheme.WHITE : JNPadLightMetalTheme.LIGHT_GRAY_3;
    }
    return selectedColor2;
  }

  /**
   * Border color1.
   *
   * @return the color
   */
  private static Color borderColor1() {
    if (borderColor1 == null) {
      borderColor1 = LAFUtils.isDarkLAF() ? JNPadLightMetalTheme.DARK_GRAY : JNPadLightMetalTheme.GRAY;
    }
    return borderColor1;
  }
  
  /**
   * Border color2.
   *
   * @return the color
   */
  private static Color borderColor2() {
    if (borderColor2 == null) {
      borderColor2 = LAFUtils.isDarkLAF() ? JNPadLightMetalTheme.GRAY : JNPadLightMetalTheme.DARK_GRAY;
    }
    return borderColor2;
  }

}
