/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.plaf;

import java.awt.Color;
import java.awt.Font;
import java.awt.SystemColor;
import java.awt.Toolkit;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.plaf.metal.DefaultMetalTheme;
import javax.swing.plaf.metal.MetalLookAndFeel;

import jnpad.config.Config;
import jnpad.ui.Gray;
import jnpad.util.Platform;

/**
 * The Class LAFUtils.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class LAFUtils {
  // --- por defecto (Ocean) ---
  /** The Constant BLACK. */
  public static final Color  BLACK                          = new Gray(51);
  
  /** The Constant SECONDARY1. */
  private static final Color SECONDARY1                     = new Color(122, 138, 153);
  /** The Constant SECONDARY2. */
  private static final Color SECONDARY2                     = new Color(184, 207, 229);
  /** The Constant SECONDARY3. */
  private static final Color SECONDARY3                     = new Gray(238);
  
  /** The Constant BACKGROUND. */
  public static final Color  BACKGROUND                     = SECONDARY3;
  /** The Constant SELECTION_BACKGROUND. */
  public static final Color  SELECTION_BACKGROUND           = SECONDARY2;
  /** The Constant DISABLED_TEXT_COLOR. */
  public static final Color  DISABLED_TEXT_COLOR            = new Gray(153);
  /** The Constant PLAIN_FONT. */
  public static final Font   PLAIN_FONT                     = new Font("Dialog", Font.PLAIN, 12);                         //$NON-NLS-1$
  /** The Constant BOLD_FONT. */
  public static final Font   BOLD_FONT                      = new Font("Dialog", Font.BOLD, 12);                          //$NON-NLS-1$
  // ---
  
  /** The Constant METAL_LAF_CLASS_NAME. */
  public static final String METAL_LAF_CLASS_NAME           = "javax.swing.plaf.metal.MetalLookAndFeel";                  //$NON-NLS-1$
  /** The Constant MOTIF_LAF_CLASS_NAME. */
  public static final String MOTIF_LAF_CLASS_NAME           = "com.sun.java.swing.plaf.motif.MotifLookAndFeel";           //$NON-NLS-1$
  /** The Constant GTK_LAF_CLASS_NAME. */
  public static final String GTK_LAF_CLASS_NAME             = "com.sun.java.swing.plaf.gtk.GTKLookAndFeel";               //$NON-NLS-1$
  /** The Constant MAC_LAF_CLASS_NAME. */
  public static final String MAC_LAF_CLASS_NAME             = "com.sun.java.swing.plaf.mac.MacLookAndFeel";               //$NON-NLS-1$
  /** The Constant NIMBUS_LAF_CLASS_NAME. */
  public static final String NIMBUS_LAF_CLASS_NAME          = "com.sun.java.swing.plaf.nimbus.NimbusLookAndFeel";         //$NON-NLS-1$
  /** The Constant WINDOWS_LAF_CLASS_NAME. */
  public static final String WINDOWS_LAF_CLASS_NAME         = "com.sun.java.swing.plaf.windows.WindowsLookAndFeel";       //$NON-NLS-1$
  /** The Constant WINDOWS_CLASSIC_LAF_CLASS_NAME. */
  public static final String WINDOWS_CLASSIC_LAF_CLASS_NAME = "com.sun.java.swing.plaf.windows.WindowsClassicLookAndFeel"; //$NON-NLS-1$
  /** The Constant AQUA_LAF_CLASS_NAME. */
  public static final String AQUA_LAF_CLASS_NAME            = "apple.laf.AquaLookAndFeel";                                //$NON-NLS-1$
  /** The Constant AQUA_LAF_CLASS_NAME_6. */
  public static final String AQUA_LAF_CLASS_NAME_6          = "com.apple.laf.AquaLookAndFeel";                            //$NON-NLS-1$

  
  /** The Constant LAF_KEY_JNPAD_LIGHT. */
  public static final String LAF_KEY_JNPAD_LIGHT            = "jNPad_light";                                              //$NON-NLS-1$
  /** The Constant LAF_KEY_JNPAD_DARK. */
  public static final String LAF_KEY_JNPAD_DARK             = "jNPad_dark";                                               //$NON-NLS-1$
  /** The Constant LAF_KEY_OCEAN2. */
  public static final String LAF_KEY_OCEAN2                 = "Ocean2";                                                   //$NON-NLS-1$
  /** The Constant LAF_KEY_STEEL. */
  public static final String LAF_KEY_STEEL                  = "Steel";                                                    //$NON-NLS-1$
  /** The Constant LAF_KEY_NATIVE. */
  public static final String LAF_KEY_NATIVE                 = "System";                                                   //$NON-NLS-1$
  /** The Constant LAF_KEY_JAVA. */
  public static final String LAF_KEY_JAVA                   = "CrossPlatform";                                            //$NON-NLS-1$
  
  /** Clave del LAF usado actualmente. */
  private static String      _lafKey                        = LAF_KEY_JAVA;

  /** no instances */
  private LAFUtils() {
    super();
  }

  /**
   * Sets the look and feel.
   *
   * @param lafKey String
   * @throws Exception the exception
   */
  public static void setLookAndFeel(String lafKey) throws Exception {
    if (LAF_KEY_JNPAD_LIGHT.equals(lafKey)) {
      MetalLookAndFeel.setCurrentTheme(new JNPadLightMetalTheme());
      UIManager.setLookAndFeel(METAL_LAF_CLASS_NAME);
    }
    else if (LAF_KEY_JNPAD_DARK.equals(lafKey)) {
      MetalLookAndFeel.setCurrentTheme(new JNPadDarkMetalTheme());
      UIManager.setLookAndFeel(METAL_LAF_CLASS_NAME);
    }
    else if (LAF_KEY_STEEL.equals(lafKey)) {
      MetalLookAndFeel.setCurrentTheme(new DefaultMetalTheme());
      UIManager.setLookAndFeel(METAL_LAF_CLASS_NAME);
    }
    else if (LAF_KEY_OCEAN2.equals(lafKey)) {
      MetalLookAndFeel.setCurrentTheme(new OceanTheme2());
      UIManager.setLookAndFeel(METAL_LAF_CLASS_NAME);
    }
    else if (LAF_KEY_NATIVE.equals(lafKey)) {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
    }
    else if (LAF_KEY_JAVA.equals(lafKey)) {
      UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
    }
    else {
      if(METAL_LAF_CLASS_NAME.equals(lafKey)) {
        final boolean boldMetal = Config.JNPAD_LAF_BOLD_METAL.getValue();
        UIManager.put("swing.boldMetal", boldMetal); //$NON-NLS-1$
      }
      UIManager.setLookAndFeel(lafKey);
    }

    _lafKey = lafKey;
  }

  /**
   * Puede retornar la clase del LAF o uno de los siguientes valores:
   * <ul>
   * <li><code>LAF_KEY_JNPAD_LIGHT</code>
   * <li><code>LAF_KEY_JNPAD_DARK</code>
   * <li><code>LAF_KEY_STEEL</code>
   * <li><code>LAF_KEY_OCEAN2</code>
   * <li><code>LAF_KEY_NATIVE</code>
   * <li><code>LAF_KEY_JAVA</code>
   * </ul>.
   *
   * @return the current laf key
   */
  public static String getCurrentLAFKey() {return _lafKey;}
  
  /**
   * Checks if is jNPad laf.
   *
   * @return true, if is jNPad laf
   */
  public static boolean isJNPadLAF() {
    return LAF_KEY_JNPAD_LIGHT.equals(_lafKey) || LAF_KEY_JNPAD_DARK.equals(_lafKey);
  }

  /**
   * Checks if is dark laf.
   *
   * @return true, if is dark laf
   */
  public static boolean isDarkLAF() {
    return UIManager.getBoolean("jnpad.dark.theme"); //$NON-NLS-1$
  }
  
  /**
   * Checks if is metal laf.
   *
   * @return true, if is metal laf
   */
  public static boolean isMetalLAF() {
    return "Metal".equals(UIManager.getLookAndFeel().getID()); //$NON-NLS-1$
  }

  /**
   * Checks if is gtk laf.
   *
   * @return true, if is gtk laf
   */
  public static boolean isGTKLAF() {
    return "GTK".equals(UIManager.getLookAndFeel().getID()); //$NON-NLS-1$
  }

  /**
   * Checks if is motif laf.
   *
   * @return true, if is motif laf
   */
  public static boolean isMotifLAF() {
    return "Motif".equals(UIManager.getLookAndFeel().getID()); //$NON-NLS-1$
  }

  /**
   * Checks if is nimbus laf.
   *
   * @return true, if is nimbus laf
   */
  public static boolean isNimbusLAF() {
    return "Nimbus".equals(UIManager.getLookAndFeel().getID()); //$NON-NLS-1$
  }

  /**
   * Checks if is windows laf.
   *
   * @return true, if is windows laf
   */
  public static boolean isWindowsLAF() {
    return "Windows".equals(UIManager.getLookAndFeel().getID()); //$NON-NLS-1$
  }

  /**
   * Checks if is win laf on vista.
   *
   * @return true, if is win laf on vista
   * @since 0.3
   */
  public static boolean isWinLafOnVista() {
    return isWindowsLAF() && Platform.isWinVistaOrNewer;
  }
  
  /**
   * Checks if is windows xp laf.
   *
   * @return true, if is windows xp laf
   */
  public static boolean isWindowsXPLAF() {
    return isWindowsLAF() && isXPTheme();
  }

  /**
   * Checks if is xp theme.
   *
   * @return true, if is xp theme
   */
  public static boolean isXPTheme() {
    Boolean isXP = (Boolean) Toolkit.getDefaultToolkit().getDesktopProperty("win.xpstyle.themeActive"); //$NON-NLS-1$
    return (isXP != null) && isXP;
  }
  
  /**
   * Checks if is under aqua look and feel.
   *
   * @return true, if is under aqua look and feel
   */
  public static boolean isUnderAquaLookAndFeel() {
    return Platform.isMac && UIManager.getLookAndFeel().getName().contains("Mac OS X"); //$NON-NLS-1$
  }
  
  /**
   * Checks if is under native mac look and feel.
   *
   * @return true, if is under native mac look and feel
   */
  public static boolean isUnderNativeMacLookAndFeel() {
    return isUnderAquaLookAndFeel();
  }

  /**
   * Checks if is under gtk look and feel.
   *
   * @return true, if is under gtk look and feel
   * @since 0.3
   */
  public static boolean isUnderGTKLookAndFeel() {
    return UIManager.getLookAndFeel().getName().contains("GTK"); //$NON-NLS-1$
  }
  
  /**
   * Checks if is full row selection laf.
   *
   * @return true, if is full row selection laf
   */
  public static boolean isFullRowSelectionLAF() {
    return isNimbusLAF() || isUnderGTKLookAndFeel();
  }

  /**
   * Sets the default look and feel decorated.
   *
   * @param b the new default look and feel decorated
   */
  public static void setDefaultLookAndFeelDecorated(boolean b) {
    JDialog.setDefaultLookAndFeelDecorated(b);
    JFrame.setDefaultLookAndFeelDecorated(b);
    Toolkit.getDefaultToolkit().setDynamicLayout(b);
    if (b)
      System.setProperty("sun.awt.noerasebackground", "true"); //$NON-NLS-1$ //$NON-NLS-2$
    else
      System.setProperty("sun.awt.noerasebackground", "false");  //$NON-NLS-1$//$NON-NLS-2$
  }

  /**
   * Gets the separator shadow.
   *
   * @return the separator shadow
   */
  public static Color getSeparatorShadow() {
    Color c = UIManager.getColor("Separator.shadow"); //$NON-NLS-1$
    if (c == null)
      c = SystemColor.controlShadow;
    return c;
  }

  /**
   * Gets the separator highlight.
   *
   * @return the separator highlight
   */
  public static Color getSeparatorHighlight() {
    Color c = UIManager.getColor("Separator.highlight"); //$NON-NLS-1$
    if (c == null)
      c = SystemColor.controlHighlight;
    return c;
  }

  /**
   * Gets the table header background.
   *
   * @return the table header background
   */
  public static Color getTableHeaderBackground() {
    Color c = UIManager.getColor("TableHeader.background"); //$NON-NLS-1$
    if (c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the table header foreground.
   *
   * @return the table header foreground
   */
  public static Color getTableHeaderForeground() {
    Color c = UIManager.getColor("TableHeader.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the table header font.
   *
   * @return the table header font
   */
  public static Font getTableHeaderFont() {
    Font f = UIManager.getFont("TableHeader.font"); //$NON-NLS-1$
    if (f == null)
      f = PLAIN_FONT;
    return f;
  }

  /**
   * Gets the table header cell border.
   *
   * @return the table header cell border
   */
  public static Border getTableHeaderCellBorder() {
    return UIManager.getBorder("TableHeader.cellBorder"); //$NON-NLS-1$
  }

  /**
   * Gets the text area background.
   *
   * @return the text area background
   */
  public static Color getTextAreaBackground() {
    Color c = UIManager.getColor("TextArea.background"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the text area foreground.
   *
   * @return the text area foreground
   */
  public static Color getTextAreaForeground() {
    Color c = UIManager.getColor("TextArea.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the text area selection background.
   *
   * @return the text area selection background
   */
  public static Color getTextAreaSelectionBackground() {
    Color c = UIManager.getColor("TextArea.selectionBackground"); //$NON-NLS-1$
    if (c == null)
      c = SELECTION_BACKGROUND;
    return c;
  }

  /**
   * Gets the text area selection foreground.
   *
   * @return the text area selection foreground
   */
  public static Color getTextAreaSelectionForeground() {
    Color c = UIManager.getColor("TextArea.selectionForeground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the list background.
   *
   * @return the list background
   */
  public static Color getListBackground() {
    // Under GTK+ L&F "List.background" often has main panel color, which looks ugly
    Color c = isGTKLAF() ? getTreeTextBackground() : UIManager.getColor("List.background"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the list foreground.
   *
   * @return the list foreground
   */
  public static Color getListForeground() {
    Color c = UIManager.getColor("List.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the list font.
   *
   * @return the list font
   */
  public static Font getListFont() {
    Font f = UIManager.getFont("List.font"); //$NON-NLS-1$
    if (f == null)
      f = BOLD_FONT;
    return f;
  }

  /**
   * Gets the list selection background.
   *
   * @return the list selection background
   */
  public static Color getListSelectionBackground() {
    Color c = UIManager.getColor("List.selectionBackground"); //$NON-NLS-1$
    if (c == null)
      c = UIManager.getColor("List[Selected].textBackground"); //$NON-NLS-1$
    if (c == null)
      c = SELECTION_BACKGROUND;
    return c;
  }

  /**
   * Gets the list selection foreground.
   *
   * @return the list selection foreground
   */
  public static Color getListSelectionForeground() {
    Color c = UIManager.getColor("List.selectionForeground"); //$NON-NLS-1$
    if (c == null)
      c = UIManager.getColor("List[Selected].textForeground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the list focus cell highlight border.
   *
   * @return the list focus cell highlight border
   */
  public static Border getListFocusCellHighlightBorder() {
    return UIManager.getBorder("List.focusCellHighlightBorder"); //$NON-NLS-1$
  }

  /**
   * Gets the tree background.
   *
   * @return the tree background
   */
  public static Color getTreeBackground() {
    Color c = UIManager.getColor("Tree.background"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the tree foreground.
   *
   * @return the tree foreground
   */
  public static Color getTreeForeground() {
    Color c = UIManager.getColor("Tree.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the tree selection background.
   *
   * @return the tree selection background
   */
  public static Color getTreeSelectionBackground() {
    Color c = UIManager.getColor("Tree.selectionBackground"); //$NON-NLS-1$
    if (c == null)
      c = SELECTION_BACKGROUND;
    return c;
  }

  /**
   * Gets the tree selection foreground.
   *
   * @return the tree selection foreground
   */
  public static Color getTreeSelectionForeground() {
    Color c = UIManager.getColor("Tree.selectionForeground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the tree font.
   *
   * @return the tree font
   */
  public static Font getTreeFont() {
    Font f = UIManager.getFont("Tree.font"); //$NON-NLS-1$
    if (f == null)
      f = PLAIN_FONT;
    return f;
  }

  /**
   * Gets the tree text background.
   *
   * @return the tree text background
   */
  public static Color getTreeTextBackground() {
    Color c = UIManager.getColor("Tree.textBackground"); //$NON-NLS-1$
    if(c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the tree text foreground.
   *
   * @return the tree text foreground
   */
  public static Color getTreeTextForeground() {
    Color c = UIManager.getColor("Tree.textForeground"); //$NON-NLS-1$
    if(c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the panel background.
   *
   * @return the panel background
   */
  public static Color getPanelBackground() {
    Color c = UIManager.getColor("Panel.background"); //$NON-NLS-1$
    if(c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the text pane background.
   *
   * @return the text pane background
   */
  public static Color getTextPaneBackground() {
    Color c = UIManager.getColor("TextPane.background"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the text pane foreground.
   *
   * @return the text pane foreground
   */
  public static Color getTextPaneForeground() {
    Color c = UIManager.getColor("TextPane.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the control dk shadow.
   *
   * @return the control dk shadow
   */
  public static Color getControlDkShadow() {
    Color c = UIManager.getColor("controlDkShadow"); //$NON-NLS-1$
    if (c == null)
      c = SECONDARY1;
    return c;
  }

  /**
   * Gets the control text.
   *
   * @return the control text
   */
  public static Color getControlText() {
    Color c = UIManager.getColor("controlText"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the control highlight.
   *
   * @return the control highlight
   */
  public static Color getControlHighlight() {
    Color c = UIManager.getColor("controlHighlight"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the control shadow.
   *
   * @return the control shadow
   */
  public static Color getControlShadow() {
    Color c = UIManager.getColor("controlShadow"); //$NON-NLS-1$
    if (c == null)
      c = SECONDARY2;
    return c;
  }

  /**
   * Gets the control.
   *
   * @return the control
   */
  public static Color getControl() {
    Color c = UIManager.getColor("control"); //$NON-NLS-1$
    if (c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the control lt highlight.
   *
   * @return the control lt highlight
   */
  public static Color getControlLtHighlight() {
    Color c = UIManager.getColor("controlLtHighlight"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the tool bar shadow.
   *
   * @return the tool bar shadow
   */
  public static Color getToolBarShadow() {
    Color c = UIManager.getColor("ToolBar.shadow"); //$NON-NLS-1$
    if (c == null)
      c = SECONDARY2;
    return c;
  }

  /**
   * Gets the tool bar highlight.
   *
   * @return the tool bar highlight
   */
  public static Color getToolBarHighlight() {
    Color c = UIManager.getColor("ToolBar.highlight"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the label background.
   *
   * @return the label background
   */
  public static Color getLabelBackground() {
    Color c = UIManager.getColor("Label.background"); //$NON-NLS-1$
    if (c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the label foreground.
   *
   * @return the label foreground
   */
  public static Color getLabelForeground() {
    Color c = UIManager.getColor("Label.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the label font.
   *
   * @return the label font
   */
  public static Font getLabelFont() {
    Font f = UIManager.getFont("Label.font"); //$NON-NLS-1$
    if (f == null)
      f = BOLD_FONT;
    return f;
  }

  /**
   * Gets the text field inactive foreground.
   *
   * @return the text field inactive foreground
   */
  public static Color getTextFieldInactiveForeground() {
    Color c = UIManager.getColor("TextField.inactiveForeground"); //$NON-NLS-1$
    if (c == null)
      c = SECONDARY2;
    return c;
  }

  /**
   * Gets the text field inactive background.
   *
   * @return the text field inactive background
   */
  public static Color getTextFieldInactiveBackground() {
    Color c = UIManager.getColor("TextField.inactiveBackground"); //$NON-NLS-1$
    if (c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the text field foreground.
   *
   * @return the text field foreground
   */
  public static Color getTextFieldForeground() {
    Color c = UIManager.getColor("TextField.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the text field background.
   *
   * @return the text field background
   */
  public static Color getTextFieldBackground() {
    Color c = UIManager.getColor("TextField.background"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the text field font.
   *
   * @return the text field font
   */
  public static Font getTextFieldFont() {
    Font f = UIManager.getFont("TextField.font"); //$NON-NLS-1$
    if (f == null)
      f = PLAIN_FONT;
    return f;
  }

  /**
   * Gets the text field border.
   *
   * @return the text field border
   */
  public static Border getTextFieldBorder() {
    return UIManager.getBorder("TextField.border"); //$NON-NLS-1$
  }

  /**
   * Gets the progress bar font.
   *
   * @return the progress bar font
   */
  public static Font getProgressBarFont() {
    Font f = UIManager.getFont("ProgressBar.font"); //$NON-NLS-1$
    if (f == null)
      f = BOLD_FONT;
    return f;
  }

  /**
   * Gets the progress bar background.
   *
   * @return the progress bar background
   */
  public static Color getProgressBarBackground() {
    Color c = UIManager.getColor("ProgressBar.background"); //$NON-NLS-1$
    if (c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the button disabled text.
   *
   * @return the button disabled text
   */
  public static Color getButtonDisabledText() {
    Color c = UIManager.getColor("Button.disabledText"); //$NON-NLS-1$
    if (c == null)
      c = DISABLED_TEXT_COLOR;
    return c;
  }

  /**
   * Gets the button background.
   *
   * @return the button background
   */
  public static Color getButtonBackground() {
    Color c = UIManager.getColor("Button.background"); //$NON-NLS-1$
    if (c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the check box disabled text.
   *
   * @return the check box disabled text
   */
  public static Color getCheckBoxDisabledText() {
    Color c = UIManager.getColor("CheckBox.disabledText"); //$NON-NLS-1$
    if (c == null)
      c = DISABLED_TEXT_COLOR;
    return c;
  }

  /**
   * Gets the menu background.
   *
   * @return the menu background
   * @since 0.3
   */
  public static Color getMenuBackground() {
    Color c = UIManager.getColor("Menu.background"); //$NON-NLS-1$
    if (c == null)
      c = BACKGROUND;
    return c;
  }

  /**
   * Gets the menu foreground.
   *
   * @return the menu foreground
   * @since 0.3
   */
  public static Color getMenuForeground() {
    Color c = UIManager.getColor("Menu.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the menu selection background.
   *
   * @return the menu selection background
   * @since 0.3
   */
  public static Color getMenuSelectionBackground() {
    Color c;
    if (isNimbusLAF()) {
      c = UIManager.getColor("Table[Enabled+Selected].textBackground"); //$NON-NLS-1$
      if (c == null)
        c = UIManager.getColor("nimbusSelectionBackground"); //$NON-NLS-1$
    }
    else {
      c = UIManager.getColor("Menu.selectionBackground"); //$NON-NLS-1$
    }
    if (c == null)
      c = SELECTION_BACKGROUND;
    return c;
  }

  /**
   * Gets the menu selection foreground.
   *
   * @return the menu selection foreground
   * @since 0.3
   */
  public static Color getMenuSelectionForeground() {
    Color c = isNimbusLAF() ? 
        UIManager.getColor("Menu[Enabled+Selected].textForeground") : //$NON-NLS-1$
        UIManager.getColor("Menu.selectionForeground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }
  
  /**
   * Gets the table selection background.
   *
   * @return the table selection background
   */
  public static Color getTableSelectionBackground() {
    Color c;
    if (isNimbusLAF()) {
      c = UIManager.getColor("Table[Enabled+Selected].textBackground"); //$NON-NLS-1$
      if (c == null)
        c = UIManager.getColor("nimbusSelectionBackground"); //$NON-NLS-1$
    }
    else {
      c = UIManager.getColor("Table.selectionBackground"); //$NON-NLS-1$
    }
    if (c == null)
      c = SELECTION_BACKGROUND;
    return c;
  }

  /**
   * Gets the table selection foreground.
   *
   * @return the table selection foreground
   */
  public static Color getTableSelectionForeground() {
    Color c = isNimbusLAF() ? 
        UIManager.getColor("Table[Enabled+Selected].textForeground") : //$NON-NLS-1$
        UIManager.getColor("Table.selectionForeground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the table background.
   *
   * @return the table background
   */
  public static Color getTableBackground() {
    // Under GTK+ L&F "Table.background" often has main panel color, which looks ugly
    Color c = isGTKLAF() ? getTreeTextBackground() : UIManager.getColor("Table.background"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the table foreground.
   *
   * @return the table foreground
   */
  public static Color getTableForeground() {
    Color c = UIManager.getColor("Table.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the table font.
   *
   * @return the table font
   */
  public static Font getTableFont() {
    Font f = UIManager.getFont("Table.font"); //$NON-NLS-1$
    if (f == null)
      f = BOLD_FONT;
    return f;
  }

  /**
   * Gets the table focus cell background.
   *
   * @return the table focus cell background
   */
  public static Color getTableFocusCellBackground() {
    Color c = UIManager.getColor("Table.focusCellBackground"); //$NON-NLS-1$
    if (c == null)
      c = Color.WHITE;
    return c;
  }

  /**
   * Gets the table focus cell foreground.
   *
   * @return the table focus cell foreground
   */
  public static Color getTableFocusCellForeground() {
    Color c = UIManager.getColor("Table.focusCellForeground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the table focus cell highlight border.
   *
   * @return the table focus cell highlight border
   */
  public static Border getTableFocusCellHighlightBorder() {
    return UIManager.getBorder("Table.focusCellHighlightBorder"); //$NON-NLS-1$
  }

  /**
   * Gets the tool tip background.
   *
   * @return the tool tip background
   */
  public static Color getToolTipBackground() {
    Color c = UIManager.getColor("ToolTip.background"); //$NON-NLS-1$
    if (c == null)
      c = SELECTION_BACKGROUND;
    return c;
  }

  /**
   * Gets the tool tip foreground.
   *
   * @return the tool tip foreground
   */
  public static Color getToolTipForeground() {
    Color c = UIManager.getColor("ToolTip.foreground"); //$NON-NLS-1$
    if (c == null)
      c = BLACK;
    return c;
  }

  /**
   * Gets the tool tip border.
   *
   * @return the tool tip border
   */
  public static Border getToolTipBorder() {
    return UIManager.getBorder("ToolTip.border"); //$NON-NLS-1$
  }

}
