/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.toolbar;

import java.awt.Insets;
import java.awt.event.MouseEvent;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.SwingConstants;

import jnpad.GUIUtilities;
import jnpad.action.JNPadAction;
import jnpad.config.Config;
import jnpad.ui.plaf.LAFUtils;
import jnpad.ui.status.StatusDisplayable;

/**
 * The Class ToolBarButton.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class ToolBarButton extends JButton {
  private String            _description;
  private boolean           _descriptionSet;
  private StatusDisplayable _displayable;

  /** UID */
  private static final long serialVersionUID = 2329693941581222137L;

  /**
   * Instantiates a new tool bar button.
   */
  public ToolBarButton() {
    this(null, null);
  }

  /**
   * Instantiates a new tool bar button.
   *
   * @param icon Icon
   */
  public ToolBarButton(Icon icon) {
    this(null, icon);
  }

  /**
   * Instantiates a new tool bar button.
   *
   * @param text String
   */
  public ToolBarButton(String text) {
    this(text, null);
  }

  /**
   * Instantiates a new tool bar button.
   *
   * @param a Action
   */
  public ToolBarButton(Action a) {
    this();
    setAction(a);
  }
  
  /**
   * Instantiates a new tool bar button.
   *
   * @param text String
   * @param icon Icon
   */
  public ToolBarButton(String text, Icon icon) {
    super(text, icon);
    setRequestFocusEnabled(false);
    setMargin(LAFUtils.isNimbusLAF() ? new Insets(1, 1, 1, 1) : new Insets(2, 2, 2, 2));
  }

  /**
   * Sets the action.
   *
   * @param a Action
   * @see javax.swing.AbstractButton#setAction(javax.swing.Action)
   */
  @Override
  public void setAction(Action a) {
    if (a instanceof JNPadAction) {
      if (!Config.isBasicMode()) {
        setHideActionText(true); // putClientProperty("hideActionText", Boolean.TRUE);
      }
      super.setAction(a);
      setDescription(((JNPadAction) a).getDescription());
      setToolTipText(((JNPadAction) a).getButtonToolTipText());
      setMnemonic('\0');
      if (Config.isBasicMode()) {
        setHorizontalTextPosition(SwingConstants.CENTER);
        setVerticalTextPosition(SwingConstants.BOTTOM);
        setText(((JNPadAction) a).getButtonText());
      }
    }
    else {
      super.setAction(a);
    }
  }
  
  /**
   * Sets the description.
   *
   * @param description String
   */
  public void setDescription(String description) {
    _description = description;
  }

  /**
   * Gets the description.
   *
   * @return the description
   */
  public String getDescription() {
    return _description;
  }

  /**
   * Sets the status displayable.
   *
   * @param displayable the new status displayable
   */
  public void setStatusDisplayable(StatusDisplayable displayable) {
    _displayable = displayable;
  }

  /**
   * Gets the status displayable.
   *
   * @return the status displayable
   */
  public StatusDisplayable getStatusDisplayable() {
    return _displayable;
  }

  /**
   * Gets the status displayable.
   *
   * @param e the MouseEvent
   * @return the status displayable
   */
  private StatusDisplayable getStatusDisplayable(MouseEvent e) {
    if (_displayable == null) {
      _displayable = GUIUtilities.getJNPadFrame(e);
    }
    return _displayable;
  }

  /**
   * Process mouse event.
   *
   * @param e the MouseEvent
   * @see javax.swing.JComponent#processMouseEvent(java.awt.event.MouseEvent)
   */
  @Override
  protected void processMouseEvent(MouseEvent e) {
    if (_description != null) {
      switch (e.getID()) {
        case MouseEvent.MOUSE_ENTERED:
          getStatusDisplayable(e).setStatus(_description);
          _descriptionSet = true;
          break;
        case MouseEvent.MOUSE_EXITED:
        case MouseEvent.MOUSE_RELEASED:
          if (_descriptionSet) {
            getStatusDisplayable(e).clearStatus();
            _descriptionSet = false;
          }
          break;
        default: //Keep FindBugs happy
          break;
      }
    }
    super.processMouseEvent(e);
  }

}
