import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.InetAddress;
import java.net.URL;

/**
 * Protocol independent download core.
 * This class is entry point to download engine itself.
 *
 * @author Radim Kolar
 * @since 0.24
 */
public final class downloadcore {

	public final static byte DOWNLOAD_UNKNOWN=0;
	public final static byte DOWNLOAD_OK=1;
	public final static byte DOWNLOAD_ERR=2;
	public final static byte DOWNLOAD_FATAL=3;
	public final static byte DOWNLOAD_REDIRECT=4;
	public final static byte DOWNLOAD_BUSY=5;
	public final static byte DOWNLOAD_RESUME=6;

	/* how to process resume requests */
	/** unknown resume request state - should not happen */
	public final static int RESUME_UNKNOWN=0;
	/** send Pragma: no-cache on resume requests */
	public final static int RESUME_NOCACHE=1;
	/** use direct connection (bypass proxy) for resume requests */
	public final static int RESUME_DIRECT=2;
	/** no special resume request handling needed */
	public final static int RESUME_NORMAL=3;

	/**
	 * Reroutes download request to appropriate factory and proxy. This is high
	 * level function and its called from main loop.
	 *
	 * @param URL URL to be downloaded
	 * @param qf queued item to operate on
	 *
	 * @return DOWNLOAD_XXX status code
	 * @throws java.net.MalformedURLException if URL is invalid
	 * @throws java.io.IOException if IO error occurred during download
	 */
	public static final int downloadCore(String URL,qfile qf)
	                 throws java.net.MalformedURLException, java.io.IOException
	{
		int skipbytes=(int)new File(dmachine.tmp_dir,qf.getLocalName()).length();
		URL url=new URL(URL);
		/* do we need to send request via FTP-proxy ? */
		if(url.getProtocol().equalsIgnoreCase("ftp"))
			if(dmachine.ftp_proxydefined)
			{
				boolean nocache=false;
				if(skipbytes>0)
				{
					if(dmachine.ftp_resume==RESUME_NOCACHE)
						nocache=true;
					else if(dmachine.ftp_resume==RESUME_DIRECT)
						return GenericTransferLogic(InetAddress.getByName(url.getHost()),url.getPort()>0? url.getPort():21, false,new ftpfactory(URL,qf,true));
				}
				/* no resume, going to ftp proxy */
				return GenericTransferLogic(dmachine.ftp_proxyserver,dmachine.ftp_proxyport,nocache,new httpfactory(URL,qf,false));
			}
			else
			{
				/* no ftp proxy defined - direct ftp connection */
				return GenericTransferLogic(InetAddress.getByName(url.getHost()),url.getPort()>0? url.getPort():21, false, new ftpfactory(URL,qf,true));
			}

		else
			if(!dmachine.http_proxydefined && url.getProtocol().equalsIgnoreCase("http"))
				/* HTTP direct connection, no http proxy defined */
				return
				GenericTransferLogic(InetAddress.getByName(url.getHost()),url.getPort()>0? url.getPort():80, false,new httpfactory(URL,qf,true));
			else
				if(url.getProtocol().equalsIgnoreCase("fsp"))
					/* FSP protocol connection */
					return GenericTransferLogic(InetAddress.getByName(url.getHost()),url.getPort()>0? url.getPort():21, false,new fspfactory(URL,qf,true));
				else
				{
					/* handle HTTP and possibly others download types via http_proxy */
					if(!dmachine.http_proxydefined)
						/* no luck here, unknown download protocol */
						return DOWNLOAD_FATAL;
					boolean nocache=false;
					if(skipbytes>0)
					{
						if(dmachine.http_resume==RESUME_NOCACHE)
							nocache=true;
						else
							if(dmachine.http_resume==RESUME_DIRECT)
								return GenericTransferLogic(InetAddress.getByName(url.getHost()),url.getPort()>0? url.getPort():80, false,new httpfactory(URL,qf,true));
					}
					return GenericTransferLogic(dmachine.http_proxyserver,dmachine.http_proxyport,nocache,new httpfactory(URL,qf,false));
				}
	}

	/**
	 * Protocol independent resume handling
	 * @param server server for connecting to
	 * @param port for connecting
	 * @param nocache Pragma: no-cache needed
	 * @param df download factory for handling request
	 * @return DOWNLOAD_XX codes
    * @throws MalformedURLException if URL processing fails
    * @throws IOException if I/O problem occurred
	 */
	protected final static int GenericTransferLogic(InetAddress server,int port,boolean nocache,downloadfactory df) throws MalformedURLException, IOException
	{
		df.processRequest(server,port,nocache);
		long skipbytes=df.havebytes; // nasty hack for now
		switch(df.rc)
		{
			case DOWNLOAD_FATAL:
			case DOWNLOAD_ERR:
			case DOWNLOAD_BUSY:
				df.close();
				return df.rc;
		}

		/* skipbytes = bytes to skip from our request */
		if(df.rc!=DOWNLOAD_OK && df.rc!=DOWNLOAD_RESUME)
		{
			System.out.println("[INTERNAL ERROR] rc="+df.rc+" returned from processRequest for "+df.url);
			df.close();return df.rc;
		}
		/* BROKEN CODE!!! check if ctsize < skipbytes */
		if(df.rc==DOWNLOAD_OK && df.contentSize<skipbytes && df.contentSize>0)
		{
			// move into attic directory
			new File(dmachine.attic_dir).mkdirs();
			new File(dmachine.attic_dir,df.qf.getLocalName()).delete();
			new File(dmachine.tmp_dir,df.qf.getLocalName()).renameTo
			(new File(dmachine.attic_dir,df.qf.getLocalName()));

			// file on the server is smaller than ours, download it again
			dmachine.log_err("Temporary file "+df.qf.getLocalName()+" for "+df.url+" is bigger than expected. Moving it into Attic directory.");
			skipbytes=0;
			df.havebytes=0;
		}

		if(df.rc==DOWNLOAD_RESUME)
		{
			System.out.println("[INFO] Continuing download of "+df.qf.getLocalName()+" on pos="+df.havebytes+".");
			skipbytes=0;
		}
		else
			skipbytes=df.havebytes;

		/* skip firsts bytes if needed */
		if(skipbytes>0)
			if(DataTransfer(df,skipbytes)==DOWNLOAD_ERR)
				return DOWNLOAD_ERR;

		return DataTransfer(df,0);
	}

	/**
	 * Transfers data from opened stream by download factory into file,
	 * optionally skipping (not saving to disk) initial bytes.
	 * You must call this function twice, first for data skipping and
	 * second for data loading. if this fuction is called with skipbytes > 0
	 * then data only skip phase will be done.
	 * @param df download factory used for downloading file
	 * @param skipbytes how much bytes to skip from input stream
	 */

	private final static int DataTransfer(downloadfactory df, long skipbytes) throws IOException
	{
		long lastrep,ct,start;
		DataOutputStream dos;
		byte b[]=new byte[4096];
		int rb;  // how many bytes was read in last turn
		long total=0;  // how many bytes we have read
		dos=null;
		start=System.currentTimeMillis()-1100L;
		lastrep=start-dmachine.reptime+10000L;  // first report at 10 sec.

		if(skipbytes>0) {
			System.err.println("[INFO] Resume on "+df.url+" is not supported by server.");
		} else
		{
			//open tmp outputfile
			dos=new DataOutputStream(new java.io.BufferedOutputStream(
					new java.io.FileOutputStream(dmachine.tmp_dir+File.separator+df.qf.getLocalName(),true),4096));
		}

		while( (dos==null && skipbytes>0) || dos!=null)
		{
			try
			{
				if(dos!=null || skipbytes>=4096)
					rb=df.datain.read(b);
				else
					rb=df.datain.read(b,0,(int)skipbytes);
			}
			catch (IOException ioe)
			{
				if(dos!=null)
					dos.close();
				throw ioe;
			}

			if(rb==-1) break; /* konec dat! */
			total+=rb;
			if(skipbytes>0) skipbytes-=rb;
			if(dos!=null) dos.write(b,0,rb);
			ct=System.currentTimeMillis();
			if(ct-lastrep>dmachine.reptime)
			{
				long CPS=total/( (ct-start)/1000L);

				System.err.println(
						(skipbytes>0?"[SKIP] ":"[LOAD] ")+df.qf.getName()+" "+(df.havebytes-skipbytes+total)+"/"+(df.fileSize)+" B ("+(int)(100*(float)(df.havebytes-skipbytes+total)/(float)(df.fileSize))+"%), ETA="+util.timeToString(ct-start)+" of "+util.timeToString(1000L*df.contentSize/CPS)+
						", BPS="+CPS);
				lastrep=ct;
			}
		}

		if(dos!=null) {
			dos.close();
			df.close();
		}

		if(skipbytes>0)
		{
			dmachine.log_err("Connection closed on "+df.url);
			return DOWNLOAD_ERR;
		}

		if(dos==null) return DOWNLOAD_OK;

		if(total<df.contentSize) {
			dmachine.log_err("Connection closed on "+df.url+" Got "+total+"B, expected "+df.contentSize+"B.");
			return DOWNLOAD_ERR;
		}
		// hotovo
		dmachine.log_ok("*** Download of "+df.qf.getName()+" ("+new File(dmachine.tmp_dir,df.qf.getLocalName()).length()+" B) was SUCCESSFULL ***");

		// move downloaded file to right directory
		if(!handleDownloadedFile(df,df.qf))
			return DOWNLOAD_FATAL;

		/* we are done! */
		return DOWNLOAD_OK;
	}

	/**
	 *  Moves downloaded file into appropriate directory for downloaded files.
	 *  @return false if there is problem moving downloaded file
	 */
	private final static boolean handleDownloadedFile(downloadfactory df, qfile qf)
	{
		int i;
		/* default savedir */
		File savedir=new File(dmachine.download_dir);
		File tmpfile = new File(dmachine.tmp_dir,qf.getLocalName());
		if (dmachine.preservemodtime && df.lastModified>0)
			tmpfile.setLastModified(df.lastModified);
		/* find save directory */
		i=qfile.isInRegexpArray(df.url.toString(),qfile.saveto,0);
		if(i>-1) savedir=qfile.saveto_dir[i];

		if(!tmpfile.renameTo (new File(savedir,qf.getLocalName()))
		)
		{
			// rename after download failed.
			dmachine.log_err("Can not rename file "+qf.getLocalName()+" from "+dmachine.tmp_dir+" to "+savedir);
			if(i==-1)
			{
				// no fallback possible
				return false;
			}
			// retry to default savedir
			if(!tmpfile.renameTo(new File(dmachine.download_dir,qf.getLocalName())))
			{
				dmachine.log_fatal("Can not move downloaded file "+qf.getLocalName()+" from temporary directory "+dmachine.tmp_dir+" to download directory "+dmachine.download_dir);
				return false;
			}
			else
			{
				dmachine.log_ok("File "+qf.getName()+" moved to normal download directory instead.");
			}
		}

		if(!qf.getName().equals(qf.getLocalName()))
		{
			dmachine.log_ok("*** File "+qf.getName()+" saved as "+qf.getLocalName()+" ***");
			String autonm=qfile.genAutoName(qf.getName());
			if(!qf.getLocalName().equals(autonm)) qf.touch();
		}
		return true;
	}

}
