/**
 * $Id: BookmarkInterface.java,v 1.2 2001/09/21 02:48:24 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.net.*;
import java.util.Vector;

import javax.swing.*;
import javax.swing.table.*;

import redlight.hotline.*;
import redlight.utils.DebuggerOutput;

/**
 * This class renders the column with the number of users.
 */
class BookmarkTableCellRenderer extends JLabel 
    implements TableCellRenderer {
    static Color black = new Color(0), white = new Color(0xffffff),
	grey = new Color(0xeeeedd);

    public Component getTableCellRendererComponent(JTable table,
						   Object value,
						   boolean isSelected,
						   boolean hasFocus,
						   int row,
						   int column) {
	setHorizontalAlignment(SwingConstants.CENTER);
	setForeground(black);
	setText(value.toString());
	return this;
    }

    public void paint(Graphics g) {

	setBackground(grey);
	g.clearRect(0, 0, getWidth(), getHeight());
	super.paint(g);

    }

}


class BookmarkTableModel extends AbstractTableModel {
    BookmarkInterface fti;
    BookmarkTableModel(BookmarkInterface f) {
	fti = f;
    }
    public int getColumnCount() { return 1; }
    public int getRowCount() { return fti.getRowCount(); }
    public Object getValueAt(int row, int col) {	
	return fti.getValueAt(row, col);
    }
    public Class getColumnClass(int col) {
	return String.class;
    }
    public String getColumnName(int col) {
	if(col == 0) return "Name";
	return "error: col = "+col;
    }
}


public class BookmarkInterface 
    implements Child, 
	       KeyListener, 
	       ActionListener, 
	       WindowListener, 
	       MouseListener, 
	       Runnable {
    Machine rlm;
    Parent parent;
    Options rlo;
    Container contentPane;
    JTextField server, port, login, password;
    JButton connect;
    JButton reload;
    JFrame f;
    JTable list;
    JPopupMenu popupMenu;
    JScrollPane scrollPane;
    JMenuItem connectToBookmarks, deleteBookmarks, editBookmarks;
    Thread getter;
    BookmarkTableModel bookmarkTableModel;
    ServerStatusChecker serverChecker;
    Vector bookmarks;
    short[] statuses;
    
    public BookmarkInterface(Parent p, Options options) {
	parent = p;
	rlo = options;
	serverChecker = null;
	
	Font guiFont = ((Font) Main.rlo.getProperty("Font.gui"));

	f = new JFrame("Bookmarks");
	contentPane = f.getContentPane();
	GridBagLayout gbl = new GridBagLayout();
	GridBagConstraints gbc = new GridBagConstraints();
	reload = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.REFRESH_ICON]));
	bookmarkTableModel = new BookmarkTableModel(this);
	list = new JTable(bookmarkTableModel);
	scrollPane = new JScrollPane(list);
	popupMenu = new JPopupMenu();
	connectToBookmarks = new JMenuItem("Connect to selected bookmarks");
	deleteBookmarks = new JMenuItem("Delete selected bookmarks");
	editBookmarks = new JMenuItem("Edit selected bookmarks");

	contentPane.setLayout(gbl);
	reload.setFont(guiFont);
        reload.setActionCommand("Reload");
        reload.setMnemonic('R');
        reload.setToolTipText("Reload the list of bookmarks.");
	connectToBookmarks.setFont(guiFont);
	deleteBookmarks.setFont(guiFont);
	editBookmarks.setFont(guiFont);
	list.getColumnModel().getColumn(0).setPreferredWidth(1000);
	list.setShowVerticalLines(false);
	list.addMouseListener(this);
	list.setDefaultRenderer(Number.class, new BookmarkTableCellRenderer());
	list.setFont(((Font) Main.rlo.getProperty("Font.list")));
        list.registerKeyboardAction(this, "Activate", 
                                    KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, 
                                                           0), 
                                    JComponent.WHEN_IN_FOCUSED_WINDOW);

	reload.addActionListener(this);
	f.addWindowListener(this);
	connectToBookmarks.addActionListener(this);
	editBookmarks.addActionListener(this);
	deleteBookmarks.addActionListener(this);

	gbc.weightx = 1;
	gbc.insets = new Insets(4, 4, 4, 4);
	gbc.fill = GridBagConstraints.NONE;
	gbc.anchor = GridBagConstraints.WEST;
	contentPane.add(reload, gbc);
       	gbc.gridx = 0; gbc.gridy = 1;
	gbc.weightx = 1; gbc.weighty = 1;
	gbc.fill = GridBagConstraints.BOTH;
	contentPane.add(scrollPane, gbc);
	popupMenu.add(connectToBookmarks);
	popupMenu.add(editBookmarks);
	popupMenu.addSeparator();
	popupMenu.add(deleteBookmarks);

	f.pack();

	f.setLocation(((Point) Main.rlo.getProperty("WindowLocation.Bookmarks")));
	f.setSize((Dimension) Main.rlo.getProperty("WindowDimension.Bookmarks"));

	show();

	bookmarks = new Vector();
        SwingUtilities.invokeLater(this);

    }

    public void run() {

	reload.setEnabled(false);
	list.removeAll();
	bookmarks.removeAllElements();
	File dir = ((File) Main.rlo.getProperty("Path.Bookmarks"));

	if(dir.exists()) {

	    String[] files = dir.list();

	    for(int i=0; i<files.length; i++) {

		HLBookmarkFile bmf = null;
		File f = new File(dir, files[i]);

		if(f.exists()) {

		    bmf = new HLBookmarkFile(f);

		    try {

			bmf.load();
			bookmarks.addElement(bmf);

		    } catch (IOException e) {}

		}

	    }

	}

	bookmarkTableModel.fireTableDataChanged();

        reload.setEnabled(true);
        
	statuses = new short[bookmarks.size()];
	serverChecker = new ServerStatusChecker(bookmarks, statuses, this);

        list.repaint();
            
        parent.displayPropertyChanged("bookmarkentries", "reload");

    }

    public int getColumnCount() { 

        return 1; 

    }

    public int getRowCount() { 

	if(bookmarks != null)
		return bookmarks.size();
       
	return 1;

    }

    public Object getValueAt(int row, int col) { 

	return ((HLBookmarkFile) bookmarks.elementAt(row)).getFile().getName();

    }

    public void show() {
	f.show();

        if(list.getSelectionModel().isSelectionEmpty())
            if(list.getRowCount() > 0)
                list.getSelectionModel().setSelectionInterval(0, 0);

        list.requestFocus();
    }

    void itemActivated(int i) {

	HLBookmarkFile bmf = (HLBookmarkFile) bookmarks.elementAt(i);
	String address = bmf.get("address");
	int port = 5500;

	if(address.indexOf(":") != -1) {

	    port = new Integer(address.substring(address.indexOf(":") + 1)).intValue(); 
	    address = address.substring(0, address.indexOf(":"));

	}

	new ConnectionInterface(address, 
                                port, 
                                bmf.get("login"), 
                                bmf.get("password"), 
                                bmf.getFile().getName());

    }

    void itemEdit(int i) {

	HLBookmarkFile bmf = (HLBookmarkFile) bookmarks.elementAt(i);
	String address = bmf.get("address");
	int port = 5500;

	if(address.indexOf(":") != -1) {

	    port = new Integer(address.substring(address.indexOf(":") + 1)).intValue(); 
	    address = address.substring(0, address.indexOf(":"));

	}

	new ConnectInterface(parent, 
				 address, 
				 new Integer(port).toString(), 
				 bmf.get("login"), 
				 bmf.get("password"), 
                                 bmf.getFile().getName(),
				 true);

    }

    /**
     * Populates a container with the contents of the bookmarks
     * directory. The container must be initialized prior to entry.
     */
    static JMenuItem empty = new JMenuItem("No bookmarks.");
    static public void populateContainerWithBookmarks(Container container,
                                                      ActionListener l) {

        int oldCount = container.getComponentCount();

        /* Add files from bookmark directory to menu. */
        
        File dir = ((File) Main.rlo.getProperty("Path.Bookmarks"));
        
        if(dir.exists()) {
            
            String[] files = dir.list();
            
            for(int i=0; i < files.length; i++) {
                
                HLBookmarkFile bmf = null;
                File f = new File(dir, files[i]);
                
                if(f.exists()) {
                    
                    bmf = new HLBookmarkFile(f);
                    
                    try {
                        
                        bmf.load();
                        BookmarkMenuItem bookmark = new BookmarkMenuItem(bmf);
                        bookmark.setActionCommand("ActivateBookmark");
                        bookmark.addActionListener(l);
                        container.add(bookmark);
                        
                    } catch (IOException e) {}
                    
                }
                
            }
            
        }

        if(container.getComponentCount() == oldCount) {

            /* No bookmarks. */

            empty.setEnabled(false);
            container.add(empty);

        }
        
    }
    
    /**
     * Following method implements ActionListener.
     */
    public void actionPerformed(ActionEvent e) {

	if(e.getActionCommand().equals("Reload")) {

            SwingUtilities.invokeLater(this);

	} else if(e.getSource() == deleteBookmarks) {

	    int[] indices = list.getSelectedRows();
	    HLBookmarkFile bmf;
	    File file;

	    for(int i=0; i<indices.length; i++) {

		bmf = (HLBookmarkFile) bookmarks.elementAt(indices[i]);
		file = new File(((File) Main.rlo.getProperty("Path.Bookmarks")), 
				bmf.getFile().getName());
		file.delete();

	    }

            list.setRowSelectionInterval(0, 0);

	    parent.displayPropertyChanged("bookmarkentries", "");
            SwingUtilities.invokeLater(this);

	} else if(e.getSource() == editBookmarks) {

	    int[] indices = list.getSelectedRows();

	    for(int i=0; i<indices.length; i++) 
		itemEdit(indices[i]);
	   
	} else if(e.getSource() == connectToBookmarks) {

	    int[] indices = list.getSelectedRows();

	    for(int i=0; i<indices.length; i++) 
		itemActivated(indices[i]);

	} else if(e.getActionCommand().equals("Activate")) {

	    int selected = list.getSelectedRow();
	    if(selected != -1)
                itemActivated(selected);

        }

    }

    /**
     * Following methods implement Child.
     */
    public void close() {

	Main.rlo.setProperty("WindowLocation.Bookmarks", f.getLocation());
	Main.rlo.setProperty("WindowDimension.Bookmarks", f.getSize());
	f.setVisible(false);

    }

    public void displayPropertyChanged(String what, Object property) {

	if(what.equals("listfont")) { 

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        list.setFont(((Font) Main.rlo.getProperty("Font.list")));
                        list.repaint();

                    }

                });

	} else if(what.equals("bookmarkstatus")) {

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        list.repaint();

                    }

                });

	} else if(what.equals("bookmarkentries")) {

            if(!property.toString().equals("reload"))
                SwingUtilities.invokeLater(this);

	}

    }

    /**
     * Following methods implement KeyListener.
     */
    public void keyPressed(KeyEvent e) {}
    public void keyReleased(KeyEvent e) {}
    public void keyTyped(KeyEvent e) {
	boolean changed = true;
	if(KeyEvent.VK_ENTER == e.getKeyChar()) {
	}
    }

    /**
     * Following methods implement WindowListener.
     */
    public void windowActivated(WindowEvent e) {}
    public void windowDeactivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) { }
    public void windowClosing(WindowEvent e) {
	close();
    }
    public void windowIconified(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}
	
    /**
     * Following methods implement MouseListener.
     */
    public void mouseClicked(MouseEvent e) {

	if(e.getClickCount() == 2 && (e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK)
	    if(list.getSelectedRow() != -1)
		itemActivated(list.getSelectedRow());

    }

    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {

        mouseReleased(e);

    }

    public void mouseReleased(MouseEvent e) {

	if(e.isPopupTrigger()) {

	    deleteBookmarks.setEnabled(list.getSelectedRows().length == 0 ? 
                                       false : true);
			       
	    editBookmarks.setEnabled(list.getSelectedRows().length == 0 ? 
                                     false : true);
			       
	    popupMenu.show(list, e.getX(), e.getY());

	}

    }

}

class ServerStatusChecker extends Thread {
    Vector bookmarkFiles;
    short[] statuses;
    BookmarkInterface fbi;
    Thread[] scanners;
    boolean[] scannerBusy;

    ServerStatusChecker(Vector bmfs, short[] st, BookmarkInterface f) {
	bookmarkFiles = bmfs;
	statuses = st;
	scanners = new Thread[4];
	scannerBusy = new boolean[scanners.length];
	for(int i=0; i<scannerBusy.length; i++)
	    scannerBusy[i] = false;
	fbi = f;
    }

    BookmarkInterface getBookmarkInterface() {
	return fbi;
    }
    public void run() {
	DebuggerOutput.debug("ServerStatusChecker start...");
	for(int i=0; i<bookmarkFiles.size(); i++) {
	    HLBookmarkFile bmf = (HLBookmarkFile) bookmarkFiles.elementAt(i);
	    String address = bmf.get("address");
	    int port;
	    if(address.indexOf(":") != -1) {
		port = new Integer(address.substring(address.indexOf(":") + 1)).intValue(); 
		address = address.substring(0, address.indexOf(":"));
	    }
	    else 
		port = 5500;
	    
	    boolean slotFound = false;
	    try {
	    do {
		for(int j=0; j<scannerBusy.length; j++) 
		    if(scannerBusy[j] == false) {
			install(j, new Thread(new PortScanner(this, statuses, i, address, port, j)));
			slotFound = true;
			break;
		    }
		if(slotFound) break;
		//		DebuggerOutput.debug("waiting for socket slot to come up..");
		waitForRelease();
	    } while(!slotFound);
	    } catch (InterruptedException ex) {
	    }
	}
	boolean stillBusy;
	do {
	    stillBusy = false;
	    for(int j=0; j<scannerBusy.length; j++) {
		if(scannerBusy[j] == true) {
		    stillBusy = true;
		    break;
		}
	    }
	    if(stillBusy) {
		//		DebuggerOutput.debug("waiting for last slots to be released...");
		try {
		    waitForRelease();
		} catch(InterruptedException e) {
		}
	    }
	} while(stillBusy);
	DebuggerOutput.debug("ServerStatusChecker done.");
    }
    
    public synchronized void waitForRelease() throws InterruptedException {
	wait();
    }
    public synchronized void install(int slot, Thread s) {
	scannerBusy[slot] = true;
	//	DebuggerOutput.debug("installed socket slot: "+slot); 
	scanners[slot] = s;
	scanners[slot].start();
    }
    public synchronized void release(int slot) {
	fbi.displayPropertyChanged("bookmarkstatus", statuses);
	scannerBusy[slot] = false;
	//	DebuggerOutput.debug("released slot "+slot);
	notify();
    }

    public synchronized void stopAllScanners() {
	//	super.stop();
	//	for(int j=0; j<scannerBusy.length; j++)
	//   if(scannerBusy[j] == true) scanners[j].stop();
    }
}

class PortScanner implements Runnable {
    short[] statuses;
    String address;
    int port, index, slot;
    ServerStatusChecker serverChecker;

    PortScanner(ServerStatusChecker ssc, short[] st, int idx, String a, int p, int sl) {
	serverChecker = ssc;
	address = a;
	port = p;
	statuses = st;
	index = idx;
	slot = sl;
    }

    public void run() {
	try {
	    Socket s = new Socket(address, port);
	    Thread.currentThread().sleep(500); // wait before close to avoid ghost users
	    s.close();
	    statuses[index] = 2;
	} catch(Exception e) {
	    statuses[index] = 1;
	}
	serverChecker.release(slot);
    }


}

class BookmarkMenuItem extends JMenuItem {
    
    HLBookmarkFile bmf;
    
    BookmarkMenuItem(HLBookmarkFile bmf) {
        
        super(bmf.getFile().getName());
        this.bmf = bmf;
        
    }

    public HLBookmarkFile getBookmarkFile() {

        return bmf;

    }

}





