/**
 * $Id: Machine.java,v 1.10 2001/10/09 02:02:03 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.awt.Color;
import java.net.UnknownHostException;
import java.io.*;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.net.SocketException;
import java.lang.reflect.InvocationTargetException;
import javax.swing.DefaultListModel;
import javax.swing.SwingUtilities;

import redlight.hotline.*;
import redlight.utils.TextUtils;
import redlight.utils.DebuggerOutput;
import redlight.utils.LegacyAudioClip;
import redlight.utils.AppAudioClip;
import redlight.utils.Logger;
import redlight.utils.FilenameUtils;
import redlight.utils.QueueThread;
import redlight.script.*;

public class Machine extends HLClientAdapter implements Scriptable {

    boolean audioMute = false, connected = false;
    int port;
    Interface rli;
    ColorScheme colorScheme;
    Color fgColor, bgColor;
    HLClient hlc = null;
    ScriptBroker scriptBroker;
    Hashtable chatWindows, 
	listModels, 
	userDomains, 
	domainStates, 
	domainLogWriters,
	taskListeners;
    String server, login, password, serverName;
    QueueThread transferQueue;
    KillThread killThread;

    class KillThread extends Thread {

        Machine rlm;
        Integer lock = new Integer(0);
        boolean running = false;

        KillThread(Machine rlm) {

            this.rlm = rlm;

        }

        public void disconnect() {

            synchronized(lock) {

                if(running)
                    return;

            }

            synchronizedDisconnect();

        }

        private synchronized void synchronizedDisconnect() {

            DebuggerOutput.debug("KillThread.disconnect: notifying kill thread");
            
            notify();

        }
        
        public synchronized void run() {
            
            try {
               
                wait();
                
            } catch(InterruptedException e) {}
            
            synchronized(lock) {

                running = true;

            }

            DebuggerOutput.debug("KillThread.run: kill thread woken up.");
            
            if(rlm.hlc != null)
                rlm.hlc.removeHLClientListener(rlm);
            
            DebuggerOutput.debug("KillThread.run: calling disconnectSuccess on interface");
        
            rlm.rli.disconnectSuccess();
            
            try {
                
                /* When hlc is null, that means create() failed, and we
                   never got into connect(), so cleanup is different. */
                
                if(rlm.hlc != null) {
                    
                    /* Make sure we don't get any events while we're taking
                       down the interface. */
                    
                    DebuggerOutput.debug("KillThread.run: closing HLClient");
                    rlm.hlc.close(); 
                    
                }
                
            } catch(IOException e) {
                
                DebuggerOutput.debug("KillThread.run: IOException: " + e);
                
            }
            
            rlm.scriptBroker.removeTarget(rlm);
            Main.removeMachine(rlm);
            rlm.destroyDomain("Public");
            
            rlm.transferQueue.stop();
            rlm.taskListeners.clear();
        
            DebuggerOutput.debug("KillThread.run: finished");
        
        }
    
    }

    public Machine(Interface rli) {

	this.rli = rli;

	server = "redlight.wox.org"; 
	login = "guest"; 
	password = ""; 
	port = 5500;

	scriptBroker = new ScriptBroker();
	taskListeners = new Hashtable();		
	userDomains = new Hashtable();
	listModels = new Hashtable();
	chatWindows = new Hashtable();
	domainStates = new Hashtable();
	domainLogWriters = new Hashtable();
        transferQueue = new QueueThread();
        transferQueue.sleepTime = 5000;
        killThread = new KillThread(this);
        killThread.start();

    }
	
    public void create() throws IOException {

        scriptBroker.addTarget(this);
        createDomain("Public");
        Main.addMachine(this);

	hlc = new HLClient(server, 
                           port, 
                           login, 
                           password, 
                           Main.rlo.getStringProperty("User.Nick"), 
                           (short) Main.rlo.getIntegerProperty("User.IconNumber"));
        
    }

    public void disconnect() {

        killThread.disconnect();

    }

    public synchronized void connect() {

	HLProtocol.UserListComponent[] uc;

	try {

	    hlc.addHLClientListener(this);
            DebuggerOutput.debug("Machine.connect: connecting ...");
	    hlc.connect();
            DebuggerOutput.debug("Machine.connect: logging in ...");
	    hlc.login();
            DebuggerOutput.debug("Machine.connect: getting user list ...");
	    uc = hlc.getUserList();

            /* synchronized(userDomains) to ensure that no
               handleUserChange() events are processed while we are
               the users from the user list. */

            synchronized(userDomains) {

                addUsersToDomain(uc);

            }

	    rli.connectSuccess();
            connected = true;
            
	} catch (UnknownHostException e) {

	    rli.error("Could not locate the server " + 
				getServerName());

	} catch (SocketException e) {		

            DebuggerOutput.stackTrace(e);
	    rli.error("The connnection to " + 
				getServerName() + " was refused.");

        } catch (AgreementRejectedException e) {

	} catch (FailedLoginException e) {
	    
	    rli.error(e.getMessage());

        } catch (HLTaskErrorException e) {

            /* Task errors are handled by handleTaskError(). */

	} catch (Exception e) {

            DebuggerOutput.stackTrace(e);
	    rli.error("An unknown error occurred while trying to connect to " + getServerName() + ".");

	} finally {

	    if(!connected)
                disconnect();

	}

    }

    void randomBGColor() {

	int it = 0;
	bgColor = new Color((int) (Math.random() * (double) 0xffffff));
	float[] hsb = Color.RGBtoHSB(bgColor.getRed(), 
				     bgColor.getBlue(), 
				     bgColor.getGreen(), 
				     null);
	while(hsb[2] < .8 && hsb[1] < .5 && it++ < 100)
	    bgColor = bgColor.brighter();	

    }
	
    public ScriptBroker getScriptBroker() {

	return scriptBroker;

    }

    public void addTaskListener(int id, HLClientListener h) {

	taskListeners.put(new Integer(id), h);

    }

    public String getServerName() {

	if(serverName == null) 
	    return server + ":" + port;
	return serverName;

    }

    public String getServer() {

	return server;

    }

    public int getPort() {

	return port;

    }

    public String getLogin() {

	return login;

    }

    public String getPassword() {

	return password;

    }

    public HLClient getHLC() {

	return hlc;

    }

    public Interface getInterface() {

	return rli;

    }

    public void setServerName(String s) {

	serverName = s;

    }

    public void setServer(String s) {

	server = s;
	readColorScheme();

    }

    public void setPort(int p) {

	port = p;

    }

    public void setLogin(String s) {

	login = s;

    }

    public void setPassword(String s) {

	password = s;

    }

    public void writeChat(String msg) {

	writeChat("Public", msg);

    }

    public void writeChat(Object domain, String msg) {

	if(chatWindows.containsKey(domain))
	    ((ChatWindow) chatWindows.get(domain)).chat(msg);

    }

    public boolean isShowJoinLeaveEnabled() {

	return isShowJoinLeaveEnabled("Public");

    }

    public boolean isShowJoinLeaveEnabled(Object domain) {

	return ((Boolean) 
		getState(domain, 
			 StateObject.SHOW_JOIN_LEAVE)).
	    booleanValue();

    }

    public boolean isLogEnabled() {

	return isLogEnabled("Public");

    }

    public boolean isLogEnabled(Object domain) {

	return ((Boolean) 
		getState(domain, 
			 StateObject.LOG)).
	    booleanValue();

    }

    public Object getState(int what) {
	
	return getState("Public", what);

    }
    
    public Object getState(Object domain, int what) {

	if(domainStates.containsKey(domain)) {
	    Hashtable state = (Hashtable) domainStates.get(domain);
	    return state.get(new Integer(what));
	}
	
	return null;
    }

    public void setState(int what, Object state) {
	
	setState("Public", what, state);
    }

    public void setState(Object domain, int w, Object s) {

	StateObject state = new StateObject(domain, w, s);

	try {

	    getScriptBroker().
		executeScript(new ScriptObject(Scripting.STATE_CHANGED, 
					       state));

	} catch (UnknownMessageException _e) {}

    }

    public DefaultListModel getUserListModel() {

	return getUserListModel("Public");

    }

    public DefaultListModel getUserListModel(Object domain) {

        DebuggerOutput.debug("Machine.getUserListModel: getting user list for domain " + domain);
	return (DefaultListModel) listModels.get(domain);

    }

    Hashtable getUserHashtable(Object domain) {
	
	return (Hashtable) userDomains.get(domain);

    }
	
    void modifyUser(int sock,
		    String nick,
		    int icon,
		    int color) {

	User newuser = new User();
	newuser.sock = sock;
	newuser.nick = nick;
	newuser.icon = icon;
	newuser.color = color;

	Enumeration en = listModels.keys();

	while(en.hasMoreElements()) {

	    Object domain = en.nextElement();

	    if(isKnownUser(domain, sock)) {

		DefaultListModel ulm = getUserListModel(domain);
		Hashtable uh = getUserHashtable(domain);

		User user = getUser(domain, sock);
		if(!user.nick.equals(nick))
		    handleChat(domain,
			       "\n <<< " + user.nick + 
			       " is now known as " + 
			       nick + " >>>");

		int index = ulm.indexOf(user);
		ulm.setElementAt(newuser, index);
		uh.put(new Integer(sock), newuser); 

	    }

	}

    }

    /**
     * Never returns null.
     */
    public User getUserOrDummy(int sock) {

        User user = getUser(sock);

        if(user == null)
            user = new User(sock, -1, -1, 0, "???");

        return user;

    }

    public User getUser(int sock) {

	return getUser("Public", sock);

    }

    public User getUser(Object domain,
			int sock) {

	Hashtable uh = getUserHashtable(domain);

	return (User) uh.get(new Integer(sock));

    }
    
    public boolean isKnownUser(int sock) {

	return isKnownUser("Public", sock);

    }

    public boolean isKnownUser(Object domain,
			       int sock) {

	Hashtable uh = getUserHashtable(domain);
	
	return uh.containsKey(new Integer(sock));

    }

    Hashtable createDomainStates() {

	Hashtable s = new Hashtable();

	s.put(new Integer(StateObject.LOG), 
	      new Boolean(Main.rlo.getBooleanProperty("Toggle.Log")));
	s.put(new Integer(StateObject.SHOW_JOIN_LEAVE), 
	      new Boolean(Main.rlo.getBooleanProperty("Toggle.JoinLeave")));

	return s;

    }

    public void createDomain(Object domain) {

        DebuggerOutput.debug("Machine.createDomain: creating domain " + domain);

	domainStates.put(domain, createDomainStates());
	userDomains.put(domain, new Hashtable());
	listModels.put(domain, new UserListModel());
	domainLogWriters.put(domain, new Logger());
	if(isLogEnabled(domain)) 
	    openLogWriter(domain);

    }

    public void destroyDomain(Object domain) {

	closeLogWriter(domain);
	domainLogWriters.remove(domain);
	domainStates.remove(domain);
	userDomains.remove(domain);
	listModels.remove(domain);

    }

    public boolean isKnownDomain(Object domain) {

	return listModels.containsKey(domain);

    }

    void addUsersToDomain(HLProtocol.UserListComponent[] uh) {

	addUsersToDomain("Public", uh);

    }

    public void addUsersToDomain(Object domain, HLProtocol.UserListComponent[] uc) {

	for(int i = uc.length - 1; i >= 0; i--) {
	    if(uc[i] != null) 
		addUserToDomain(domain,
				uc[i].sock,
				uc[i].nick,
				uc[i].icon,
				uc[i].clr);
				
	}

    }

    void addUserToDomain(Object domain,
			 int sock,
			 String nick,
			 int icon,
			 int color) {

        DebuggerOutput.debug("addUserToDomain: calling handleUserChange");
        
	handleUserChange(domain,
			 sock, 
			 nick,
			 icon, 
			 color,
			 true);

    }

    public User[] getUsers() {

	return getUsers("Public");

    }

    public User[] getUsers(Object domain) {

	DefaultListModel ulm = getUserListModel(domain);
	Hashtable uh = getUserHashtable(domain);

	User[] u = new User[ulm.getSize()];

	for(int i=0; i<ulm.getSize(); i++) {
	    u[i] = (User) ulm.getElementAt(i);
	}

	return u;

    }

    void notifyJoinLeave(Object domain, String msg) {

	if(isShowJoinLeaveEnabled(domain))
	    handleChat(domain, msg);

    }

    boolean openLogWriter(Object domain) {

	DebuggerOutput.debug("opening logfile for " + domain);

	Logger logger = getLogWriter(domain);
	File logPath = new File(Main.CONFIGURATION_DIRECTORY, "Log");
	String kind = domain.equals("Public") ? 
	    "" : "-privchat-" + Integer.
	    toHexString(Math.abs(((Integer) domain).intValue()));
	String logFile = FilenameUtils.
	    qualify(getServerName() + kind);
	if(!logger.
	   openLog(new File(logPath, logFile),
		   "Server: " + getServerName() + " [" + getServer() + 
		   ":" + getPort() + "]",
		   true)) {
	    getInterface().error("Error opening logfile: " + 
				 logger.getError().getMessage());
	    setState(domain, 
	    	     StateObject.LOG,
		     new Boolean(false));
	    return false;
	}

	return true;
    }

    void closeLogWriter(Object domain) {

	if(isLogEnabled(domain)) {
	    DebuggerOutput.debug("closing logfile for " + domain);	    
	    getLogWriter(domain).closeLog();
	}
	
    }

    Logger getLogWriter(Object domain) {

	return (Logger) domainLogWriters.get(domain);

    }

    public boolean isAudioSupported() {

	return AppAudioClip.isAudioSupported();

    }

    public void playAudio(String s) {

	if(Main.rlo != null && !audioMute) 
	    Main.rlo.playAudioClip(s);

    }

    public void muteAudio(boolean t) {

	audioMute = t;

    }

    public void registerChat(ChatWindow c) {

	registerChat("Public", c);

    }

    public void unregisterChat(ChatWindow c) {

	unregisterChat("Public", c);

    }

    public void registerChat(Object domain, ChatWindow c) {

	chatWindows.put(domain, c);

    }

    public void unregisterChat(Object domain, ChatWindow c) {

	chatWindows.remove(domain);

    }

    ScriptResult processStateChange(StateObject state) {

	int status = ScriptResult.RESULT_NO_ERROR;

	if(domainStates.containsKey(state.domain)) {

	    Hashtable states = (Hashtable) domainStates.get(state.domain);
	    states.put(new Integer(state.what), state.state);

	    switch(state.what) {
	    case StateObject.LOG:
		boolean open = ((Boolean) state.state).booleanValue();

		if(open) 

		    status = openLogWriter(state.domain) ?
			ScriptResult.RESULT_NO_ERROR :
			ScriptResult.RESULT_ERROR;

		else
		    
		    closeLogWriter(state.domain);

		break;
	    }
	}

	return new ScriptResult(this, status);

    }

    void readColorScheme() {

	if(Main.rlo != null) {
	    colorScheme = Main.rlo.getColorScheme(getServer() + "-" + 
						     getPort());
	    if(colorScheme != null) {
		fgColor = colorScheme.getSchemeColor("foreground");
		if(fgColor == null) fgColor = new Color(0);
		bgColor = colorScheme.getSchemeColor("background");
		if(bgColor == null) randomBGColor();
		return;
	    }

	}

	fgColor = new Color(0);

	if(Main.rlo != null) {

	    if(!Main.rlo.getBooleanProperty("Toggle.Color")) {
		bgColor = new Color(0xffffff);
		return;
	    }

	}
				
	randomBGColor();

    }
	
    public ColorScheme getColorScheme() {

	if(colorScheme != null)

	    return colorScheme;

	else {

	    ColorScheme cs = new ColorScheme();
	    cs.setSchemeColor("foreground", fgColor);
	    cs.setSchemeColor("background", bgColor);
	    return cs;

	}

    }
	
    public Color getSchemeColor(String part) {

	if(colorScheme != null)

	    return colorScheme.getSchemeColor(part);

	else {

	    if(part.equals("foreground")) return fgColor;
	    if(part.equals("background") && Main.rlo.getBooleanProperty("Toggle.Color")) 
		return bgColor;
	    if(part.equals("background") && !Main.rlo.getBooleanProperty("Toggle.Color")) 
		return new Color(0xffffff);

	}
	return new Color(0xffffff);

    }
	
    public void setSchemeColor(String part, Color color) {

	if(colorScheme == null) {

	    colorScheme = new ColorScheme();
	    colorScheme.setSchemeColor("foreground", fgColor);
	    colorScheme.setSchemeColor("background", bgColor);

	}

	colorScheme.setSchemeColor(part, color);
	Main.rlo.setColorScheme(getServer() + "-" + getPort(), 
				   colorScheme);

    }
	
    private String filter(String msg) {

	byte[] b = msg.getBytes();
	StringBuffer sb = new StringBuffer();
	for(int i=0; i<b.length; i++)
	    if(!Character.isISOControl((char) b[i])) sb.append((char) b[i]);
	return sb.toString();

    }

    public String toString() {

	return getServer() + ":" + 
	    getPort() + " [" + 
	    getLogin() + "/" + 
	    getPassword() + "]";

    }

    public boolean addTransfer(TransferInterface ti) {

        return transferQueue.add(ti);

    }

    public void removeTransfer(TransferInterface ti) {

        transferQueue.remove(ti);

    }

    /**
     * Following methods implement Scriptable.
     */

    public void gotTarget(ScriptBroker sb) {}

    public void lostTarget(ScriptBroker sb) {}

    public long getKnownMessages() {

	return Scripting.STATE_CHANGED;

    }

    public ScriptResult executeScript(ScriptObject s) 
	throws UnknownMessageException {
	
	int value = (int) s.getType();

	switch(value) {
	    
	case Scripting.STATE_CHANGED:
	    StateObject state = (StateObject) s.getUserObject();
	    return processStateChange(state);
	    
	}

	throw new UnknownMessageException();

    }

    /**
     * Following methods extend HLClientAdapter.
     */
    public void handleDisconnect(String msg) {

        DebuggerOutput.debug("Machine.handleDisconnect: error = " + msg);

	rli.error("The connection with the server has unexpectedly closed.");

        DebuggerOutput.debug("Machine.handleDisconnect: disconnecting ...");
        
        disconnect();
        
        DebuggerOutput.debug("Machine.handleDisconnect: finished");
        
    }
    
    public void handleTransferQueuePosition(int ref, int position) {

        for(Enumeration en = taskListeners.keys(); en.hasMoreElements(); )
            ((HLClientListener) taskListeners.get(en.nextElement())).handleTransferQueuePosition(ref, position);

    }

    public void handleTaskComplete(int id) {

        DebuggerOutput.debug("Machine.handleTaskComplete: id = " + id);

        Integer k = new Integer(id);

	if(taskListeners.containsKey(k)) {

	    HLClientListener h = 
		(HLClientListener) taskListeners.get(k);

            taskListeners.remove(k);

	    h.handleTaskComplete(id);

	}

    }

    public void handleTaskError(int id, String err) {

        DebuggerOutput.debug("Machine.handleTaskError: id = " + id + ", err = " + err);

        /* Task id's that represent file transfers are handled by the
           object that manages the file transfer UI. This is how the
           download retry system works, by giving download tasks a
           second chance. */

        Integer k = new Integer(id);

	if(taskListeners.containsKey(k)) {

	    HLClientListener h = 
		(HLClientListener) taskListeners.get(k);

            taskListeners.remove(k);

	    h.handleTaskError(id, err);

	} else {

	    rli.error(err);
            
        }

    }

    public void handleUserChange(int sock, 
				 String nick, 
				 int icon, 
				 int color) {

	handleUserChange(sock, nick, icon, color, false);

    }


    public void handleUserChange(int sock, 
				 String nick, 
				 int icon, 
				 int color, 
				 boolean init) {

	handleUserChange("Public",
			 sock,
			 nick,
			 icon,
			 color,
			 init);

    }

    
    public void handleUserChange(Object domain,
                                 int sock, 
                                 String nick, 
                                 int icon, 
                                 int color, 
                                 boolean init) {

        synchronized(userDomains) {

            DefaultListModel ulm = getUserListModel(domain);
            Hashtable uh = getUserHashtable(domain);
            
            if(isKnownUser(domain, sock)) {
                
                DebuggerOutput.debug("Machine.handleUserChange[" + sock + "]: user change");
                
                modifyUser(sock,
                           nick,
                           icon,
                           color);
                
            } else {
                
                DebuggerOutput.debug("Machine.handleUserChange[" + sock + "]: user join");
                
                User user = new User();
                user.sock = sock;
                user.icon = icon;
                user.color = color;
                user.nick = nick;

                if(init)
                    ulm.insertElementAt(user, 0);
                else
                    ulm.addElement(user);

                uh.put(new Integer(sock), user);
                
                if(!init)
                    playAudio("userjoin");
                
                notifyJoinLeave(domain,
                                "\n <<< " + nick + " joins >>>");
            }
            
        }

    }

    public void handleUserLeave(int sock) {

	handleUserLeave("Public", sock);

    }

    public void handleUserLeave(Object domain,
					     int sock) {

	DefaultListModel ulm = getUserListModel(domain);
	Hashtable uh = getUserHashtable(domain);

	User user = (User) uh.get(new Integer(sock));

	if(user != null) {

	    ulm.removeElement(user);
	    uh.remove(new Integer(sock));
	    playAudio("userleave");

	    notifyJoinLeave(domain,
			    "\n <<< " + user.nick + " leaves >>>");
	}

    }
    
    public void handlePrivateChatCreate(final int pcref, 
					final int sock, 
					final String nick, 
					final int icon, 
					final int color) {
	
        DebuggerOutput.debug("handlePrivateChatCreate: calling addUserToDomain");

        final Machine machine = this;

        createDomain(new Integer(pcref));
	addUserToDomain(new Integer(pcref),
			sock,
			nick,
			icon,
			color);

        try {
            
            SwingUtilities.invokeAndWait(new Runnable() {
                    
                    public void run() {
                        
                        new PrivChatInterface(machine, 
                                                  pcref, 
                                                  sock, 
                                                  nick, 
                                                  icon, 
                                                  color);
                        
                    }
                    
                });

        } catch(InterruptedException e) {
        } catch(InvocationTargetException e) {}

    }

    public void handlePrivateChatUserChange(int pcref, 
					    int sock, 
					    String nick, 
					    int icon, 
					    int color) {

	if(isKnownDomain(new Integer(pcref))) 
	    handleUserChange(new Integer(pcref),
			     sock,
			     nick,
			     icon,
			     color,
			     false);

    }

    public void handlePrivateChatUserLeave(int pcref,
					   int sock) {

	if(isKnownDomain(new Integer(pcref))) 
	    handleUserLeave(new Integer(pcref),
			    sock);

    }

    public void handlePrivateChat(int ref, 
                                  String msg) {

	handleChat(new Integer(ref), msg);

    }

    public void handleChat(String msg) {
	
	handleChat("Public", msg);

    }

    public void handleChat(Object domain, String msg) {
        
	writeChat(domain, msg.replace('\r', '\n'));

	if(isLogEnabled(domain)) {

	    Logger logger = getLogWriter(domain);

	    if(!logger.writeLog(msg.trim())) {
		getInterface().error("Error writing to logfile: " + 
				     logger.getError().getMessage());
		setState(domain,
			 StateObject.LOG,
			 new Boolean(false));
	    }

	}

	playAudio("receivechat");

    }


    public void handlePrivateChatInvite(int pcref, 
                                        int sock, 
                                        String nick) {
        
        new PrivChatInviteInterface(this, 
                                    pcref, 
                                    sock, 
                                    nick);
    }
    
    public void handleMessage(int sock, 
                              String nick, 
                              String message) {
        
        User user = getUser(sock);
        
        if(user == null)
            user = new User(sock, -1, -1, 0, nick);
        
        new MessageReceive(this, user, message);
        
    }
    
    public boolean handleAgreement(String agreement) {
        
        if(connected)
            return AgreementInterface.handleAgreement(this, agreement);
        
        return true;
        
    }
    
    public void handleAdministratorMessage(final String msg) {
        
        final Machine rlm = this;

        if(connected) {
            
            SwingUtilities.invokeLater(new Runnable() {
                    
                    public void run() {
                        
                        AdminMessageInterface.handleAdministratorMessage(rlm, msg);
                        
                    }
                    
                });
            
        }
        
    }
    
}

