/**
 * $Id: Options.java,v 1.11 2001/10/09 06:57:56 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.zip.*;
import java.net.URL;

import redlight.hotline.*;
import redlight.utils.AppAudioClip;
import redlight.utils.LegacyAudioClip;
import redlight.utils.Queueable;
import redlight.utils.DebuggerOutput;

/**
 * This class encapsulates the programs settings and provides methods
 * to read / write these settings.
 */
public class Options {

    transient LegacyAudioClip[] audioClips;
    transient LegacyAudioClip ac;
    static public final int CHAT_ICON = 0;
    static public final int NEWS_ICON = 1;
    static public final int FILES_ICON = 2;
    static public final int MESSAGE_ICON = 3;
    static public final int STOP_ICON = 4;
    static public final int DISCONNECT_ICON = 5;
    static public final int USER_ICON = 6;
    static public final int UPLOAD_ICON = 7;
    static public final int DOWNLOAD_ICON = 8;
    static public final int REFRESH_ICON = 9;
    static public final int TRASH_ICON = 10;
    static public final int FOLDER_ICON = 11;
    static public final int FIND_ICON = 12;
    static public final int INFO_ICON = 13;
    static public final int SPACER_ICON = 14;
    static public final int NEWUSER_ICON = 15;
    static public final int OPENUSER_ICON = 16;
    
    String[] audioClipNames;
    Vector trackers;
    Hashtable imageFiles, imageSets, properties, helperApps;

    public Options() {

	properties = new Hashtable();

        setProperty("Home", new Connection("awacs.dhs.org", 
                                           5500,
                                           "guest",
                                           "",
                                           "Red Light"));

        setupDefaults();

	trackers = new Vector();
	trackers.addElement(getStringProperty("Tracker.Default"));

	imageSets = new Hashtable();
	imageFiles = new Hashtable();
        helperApps = new Hashtable();

    }

    void setupDefaults() {

	Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
	Point topLeft = new Point(4, 64);

        if(getProperty("MacFileMethod") == null)
            setProperty("MacFileMethod", redlight.macfiles.MacFile.isSupportedMacFileMethod("Native") ? "Native" : "Split");
        
	if(getProperty("BorderInsets") == null)
            setProperty("BorderInsets", new Insets(4, 4, 4, 4));
        if(getProperty("Toggle.LiveResizing") == null)
            setBooleanProperty("Toggle.LiveResizing", true);
        if(getProperty("Toggle.QueueDownloads") == null)
            setBooleanProperty("Toggle.QueueDownloads", true);
        if(getProperty("Toggle.Color") == null)
            setBooleanProperty("Toggle.Color", false);
        if(getProperty("Toggle.JoinLeave") == null)
            setBooleanProperty("Toggle.JoinLeave", true);
        if(getProperty("Toggle.Log") == null)
            setBooleanProperty("Toggle.Log", false);
        if(getProperty("Toggle.SwallowConnections") == null)
            setBooleanProperty("Toggle.SwallowConnections", true);
        if(getProperty("Toggle.RememberWindowPositions") == null)
            setBooleanProperty("Toggle.RememberWindowPositions", true);
        if(getProperty("Toggle.LoadNews") == null)
            setBooleanProperty("Toggle.LoadNews", true);
        if(getProperty("Toggle.ShowIcons") == null)
            setBooleanProperty("Toggle.ShowIcons", true);
        if(getProperty("Auto.DownloadRetries") == null)
            setIntegerProperty("Auto.DownloadRetries", 3);
        if(getProperty("Auto.StallTimeout") == null)
            setIntegerProperty("Auto.StallTimeout", 180);
        if(getProperty("Auto.MaxFolderErrors") == null)
            setIntegerProperty("Auto.MaxFolderErrors", 10);
        if(getProperty("Path.Downloads") == null)
            setProperty("Path.Downloads", 
                        new File(Main.CONFIGURATION_DIRECTORY, "Downloads"));
        if(getProperty("Path.ServerHome") == null)
            setProperty("Path.ServerHome", 
                        new File(Main.CONFIGURATION_DIRECTORY, "Downloads"));
        if(getProperty("Path.Bookmarks") == null)
            setProperty("Path.Bookmarks", 
                        new File(Main.CONFIGURATION_DIRECTORY, "Bookmarks"));
        if(getProperty("Path.ColorSchemes") == null)
            setProperty("Path.ColorSchemes", 
                        new File(Main.CONFIGURATION_DIRECTORY, "ColorSchemes"));
        if(getProperty("Path.Sounds") == null)
            setProperty("Path.Sounds",
                        new File(Main.CONFIGURATION_DIRECTORY, "Sounds"));
        if(getProperty("User.IconNumber") == null)
            setIntegerProperty("User.IconNumber", 2003);
        if(getProperty("User.Nick") == null)
            setStringProperty("User.Nick", "Red Light");
        if(getProperty("Tracker.Default") == null)
            setStringProperty("Tracker.Default", "tracked.group.org");
        if(getProperty("WindowLocation.Tracker") == null)
            setProperty("WindowLocation.Tracker", topLeft);
        if(getProperty("WindowLocation.Bookmarks") == null)
            setProperty("WindowLocation.Bookmarks", topLeft);
        if(getProperty("WindowLocation.Main") == null)
            setProperty("WindowLocation.Main", new Point((screenSize.width - 600) / 2, (screenSize.height - 420) / 2));
        if(getProperty("WindowDimension.Tracker") == null)
            setProperty("WindowDimension.Tracker", new Dimension(450, 300));
        if(getProperty("WindowDimension.Bookmarks") == null)
            setProperty("WindowDimension.Bookmarks", new Dimension(250, 300));
        if(getProperty("WindowDimension.Main") == null)
            setProperty("WindowDimension.Main", new Dimension(600, 420));
        
        if(getProperty("Font.text") == null)
            setProperty("Font.text", new Font("Monospaced", Font.PLAIN, 12));
        if(getProperty("Font.list") == null)
            setProperty("Font.list", new Font("SansSerif", Font.PLAIN, 12));
        if(getProperty("Font.info") == null)
            setProperty("Font.info", new Font("SansSerif", Font.PLAIN, 10));
        if(getProperty("Font.small") == null)
            setProperty("Font.small", new Font("Dialog", Font.PLAIN, 10));
        if(getProperty("Font.gui") == null)
            setProperty("Font.gui", new Font("Dialog", Font.PLAIN, 12));

        if(getProperty("ServerName") == null)
            setProperty("ServerName", "Your server name");
        if(getProperty("ServerDescription") == null)
            setProperty("ServerDescription", "What you provide");
        if(getProperty("ServerTracker1") == null)
            setProperty("ServerTracker1", "");
        if(getProperty("ServerTracker2") == null)
            setProperty("ServerTracker2", "");
        if(getProperty("ServerTracker3") == null)
            setProperty("ServerTracker3", "");
        if(getProperty("ServerTracker4") == null)
            setProperty("ServerTracker4", "");

	trackers = new Vector();
	trackers.addElement(getStringProperty("Tracker.Default"));
        
	imageSets = new Hashtable();
	imageFiles = new Hashtable();
        helperApps = new Hashtable();

    }

    public boolean getBooleanProperty(String property) {

        Object value = properties.get(property);

        if(value == null)
            return false;

        if(!(value instanceof Boolean))
            throw new IllegalArgumentException(property + " is not a boolean property");

        return ((Boolean) value).booleanValue();

    }

    public void setBooleanProperty(String property, boolean value) {

        properties.put(property, new Boolean(value));

    }

    public int getIntegerProperty(String property) {

        Object value = properties.get(property);

        if(value == null)
            return 0;

        if(!(value instanceof Integer))
            throw new IllegalArgumentException(property + " is not an integer property");

        return ((Integer) value).intValue();

    }

    public void setIntegerProperty(String property, int value) {

        properties.put(property, new Integer(value));

    }

    public String getStringProperty(String property) {

        Object value = properties.get(property);

        if(value == null)
            return null;

        if(!(value instanceof String))
            return value.toString();

        return (String) value;
        
    }

    public void setStringProperty(String property, String value) {

        properties.put(property, value);

    }

    public Object getProperty(String key) {

	return properties.get(key);

    }

    public void setProperty(String key, Object value) {

        if(value == null)
            properties.remove(key);
        else
            properties.put(key, value);

    }

    public void addTracker(String s) {

	trackers.addElement(s);

    }

    public void removeTracker(String s) {

	trackers.removeElement(s);

    }

    public String[] getTrackers() {

	String[] s = new String[trackers.size()];
	for(int i=0; i<trackers.size(); i++)
	    s[i] = (String) trackers.elementAt(i);
	return s;

    }

    public String getHelperApp(URL url) {

        String helper = getHelperForType(url.getProtocol(), true);

        if(helper == null)
            return getHelperForType("http", false);

        return helper;

    }

    private String getHelperForType(String type, boolean ask) {

        String app = (String) helperApps.get(type);
        
        if(app == null && ask) {
            
            FileDialog fd = new FileDialog(Main.getActiveShell().f, "Select a helper app for type " + type);
            fd.show();

            if(fd.getFile() != null && fd.getDirectory() != null)
                helperApps.put(type, new File(fd.getDirectory(), fd.getFile().toString()).toString());

        }

        return (String) helperApps.get(type);

    }

    public void registerImageSet(String name, String[] images) {

	if(name != null && images != null) {

	    imageFiles.put(name, images);
	    loadImageSet(name);

	}

    }

    private void loadImageSet(String set) {

	String[] files = (String []) imageFiles.get(set);
	Image[] images = new Image[files.length];
	Frame frame = new Frame();
	URL imageURL;
	Toolkit tk = Toolkit.getDefaultToolkit();

	try {

	    for(int i=0; i < images.length; i++) {

		imageURL = getClass().getResource(files[i]);

                if(imageURL == null) {

                    DebuggerOutput.debug("Cannot load image " + files[i] + "; have to abort.");
                    System.exit(1);

                }

                images[i] = tk.createImage((java.awt.image.ImageProducer)
					   imageURL.getContent());
 		imageSets.put(set, images);

	    }

	} catch (java.io.IOException ex) {

	    DebuggerOutput.debug("loadImageSet: hmmm, error loading image set "+set+": "+ex);
	    imageSets.put(set, null);
	
	}

    }

    private void loadAllImageSets() {

	Enumeration e = imageFiles.keys();

	while(e.hasMoreElements())
	    loadImageSet((String) e.nextElement());

    }

    public Image[] getImageSet(String set) {

	return (Image []) imageSets.get(set);

    }

    public void registerAudioClips(String[] clips) {

        if(clips != null) {

	    audioClipNames = clips;
            loadAudioClips();

	}

    }

    private void loadAudioClips() {

        audioClipNames = null;

	if(audioClipNames != null) {

	    audioClips = new LegacyAudioClip[audioClipNames.length];

	    for(int i=0; i < audioClips.length; i++)
		audioClips[i] = new AppAudioClip(getClass().getResource(audioClipNames[i]));

	}

    }

    public synchronized void playAudioClip(String clipName) {

	if(audioClipNames != null) {

	    for(int i=0; i < audioClipNames.length; i++) {

		if(audioClipNames[i].equals("snd/" + clipName + ".au")) {

		    if(audioClips != null) {

			ac = audioClips[i];
                        ac.play();
                        ac.stop();

		    }

		}

	    }

	}

    }

    public ColorScheme getColorScheme(String s) {

        File colorSchemePath = (File) getProperty("Path.ColorSchemes");

	String[] schemes = colorSchemePath.list();

	if(schemes != null) {

            String match = s.toUpperCase();

            for(int i=0; i<schemes.length; i++) {

                if(match.equals(schemes[i].toUpperCase())) {
                    
                    File schemeFile = new File(colorSchemePath, schemes[i]);
                    return parseSchemeFile(schemeFile);

                }

            }

	}

	return null;

    }

    public void setColorScheme(String s, ColorScheme cs) {

        File colorSchemePath = (File) getProperty("Path.ColorSchemes");

	writeSchemeFile(new File(colorSchemePath, s), cs);

    }
	
    private ColorScheme parseSchemeFile(File f) {

	ColorScheme scheme = null;
	ColorScheme newScheme = new ColorScheme();
	LineNumberReader lnr = null;

	try {

	    FileReader fr = new FileReader(f);
	    lnr = new LineNumberReader(fr);

	    while(lnr.ready()) {

		String line = lnr.readLine();
		StringTokenizer st = new StringTokenizer(line, "=");

		while(st.hasMoreTokens()) {

		    String schemePart = st.nextToken().trim();
		    Color schemeColor = new Color(Integer.parseInt(st.nextToken().trim(), 16));
		    newScheme.setSchemeColor(schemePart, schemeColor);

		}

	    }

	    scheme = newScheme;

	} catch (NumberFormatException e) {

	} catch (IOException e) {

	} finally {

	    try {

		lnr.close();

	    } catch (IOException e) {}

	}

	return scheme;

    }
	
    private void writeSchemeFile(File f, ColorScheme cs) {

        File colorSchemePath = (File) getProperty("Path.ColorSchemes");

	PrintWriter pw = null;

	try {

	    FileWriter fw = new FileWriter(f);
	    pw = new PrintWriter(fw);
	    Enumeration e = cs.getSchemeParts();

	    while(e.hasMoreElements()) {

		String schemePart = (String) e.nextElement();
		pw.println(schemePart + " = " + Integer.toHexString( cs.getSchemeColor(schemePart).getRGB()).substring(2) );

	    }

	    if(pw != null)
                pw.close();

	} catch (IOException ex) {
	}

    }
	
    public void store(File f) throws IOException {

	FileOutputStream fos = new FileOutputStream(f);
	GZIPOutputStream gzos = new GZIPOutputStream(fos);
	ObjectOutputStream out = new ObjectOutputStream(gzos);
	out.writeObject(properties);
	out.writeObject(helperApps);
        out.writeObject(trackers);
	out.flush();
	out.close();

    }

    public void load(File f) {

        Hashtable backupProperties = properties;
        Hashtable backupHelperApps = helperApps;
        Vector backupTrackers = trackers;

        try {

            FileInputStream fis = new FileInputStream(f);
            GZIPInputStream gzis = new GZIPInputStream(fis);
            ObjectInputStream in = new ObjectInputStream(gzis);
            properties = (Hashtable) in.readObject();
            helperApps = (Hashtable) in.readObject();
            trackers = (Vector) in.readObject();
            in.close();

        } catch(Exception e) {

            DebuggerOutput.stackTrace(e);
            
            properties = backupProperties;
            helperApps = backupHelperApps;
            trackers = backupTrackers;

        }            

        /* Give default values to stuff that was not given. */

        setupDefaults();

    }

}
