/**
 * $Id: HLClientDispatcher.java,v 1.16 2001/10/08 22:03:28 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.net.*;
import java.io.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Vector;
import java.util.Calendar;
import java.util.Enumeration;

import redlight.macfiles.*;
import redlight.utils.*;
import redlight.hotline.*;

/**
 * This class is used by the HLClient class for the 
 * handling of incoming events.
 */
public class HLClientDispatcher {
    HLProtocol hlp;
    HLClient hlc;
    boolean agreementAccepted = true, agreementAborted = false, agreementReceived = false;

    HLClientDispatcher(HLClient h) {

	hlc = h;
	hlp = new HLProtocol();

    }

    /**
     * Main dispatcher.
     * @param packet the packet to process.
     */
    public void dispatch(HLProtocol.Packet packet)  {
        
        try {
            
            switch(packet.header.id) {
                
            case HLProtocol.HTLS_HDR_TASK:
                dispatchTask(packet);
                break;
            case HLProtocol.HTLS_HDR_CHAT:
                dispatchChat(packet);
                break;
            case HLProtocol.HTLS_HDR_MSG:
                dispatchMessage(packet);
                break;
            case HLProtocol.HTLS_HDR_USER_CHANGE:
                dispatchUserChange(packet);
                break;
            case HLProtocol.HTLS_HDR_USER_LEAVE:
                dispatchUserLeave(packet);
                break;
            case HLProtocol.HTLS_HDR_NEWS_POST:
                dispatchNewsPost(packet);
                break;
            case HLProtocol.HTLS_HDR_CHAT_SUBJECT:
                dispatchPrivateChatSubject(packet);
                break;
            case HLProtocol.HTLS_HDR_CHAT_INVITE:
                dispatchPrivateChatInvite(packet);
                break;
            case HLProtocol.HTLS_HDR_CHAT_USER_CHANGE:
                dispatchPrivateChatUserChange(packet);
                break;
            case HLProtocol.HTLS_HDR_CHAT_USER_LEAVE:
                dispatchPrivateChatUserLeave(packet);
                break;
            case HLProtocol.HTLS_HDR_AGREEMENT:
                dispatchAgreement(packet);
                break;
            case HLProtocol.HTLS_HDR_QUEUEPOS:
                dispatchTransferQueuePos(packet);
                break;
            default:
                DebuggerOutput.debug("dispatch: unknown header type: 0x" + Integer.toHexString(packet.header.id));
                    
            }
                
        } catch(IOException e) {
            
            DebuggerOutput.stackTrace(e);
                
        } catch(HLTaskNotFoundException e) {
                
            DebuggerOutput.stackTrace(e);
                
        }            
        
    }

    /* Private chat */

    void dispatchPrivateChatUserChange(HLProtocol.Packet packet) throws IOException {

	int icon = 0, sock = 0, color = 0, pcref = 0;
	byte[] nick = new byte[0];

        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
	    switch(dh.type) {
                
	    case HLProtocol.HTLS_DATA_SOCKET:
		sock = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_ICON:
		icon = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_NICK:
		nick = dh.data;
		break;
	    case HLProtocol.HTLS_DATA_COLOUR:
		color = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_CHAT_REF:
		pcref = ToArrayConverters.byteArrayToInt(dh.data);
		break;
                
	    }
            
	}
        
        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handlePrivateChatUserChange(pcref, sock, new String(nick), icon, color);
        
    }
	
    void dispatchPrivateChatUserLeave(HLProtocol.Packet packet) throws IOException {

	int sock = 0, pcref = 0;

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLS_DATA_CHAT_REF:
		pcref = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_SOCKET:
		sock = ToArrayConverters.byteArrayToInt(dh.data);
		break;

	    }

	}

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
		handlePrivateChatUserLeave(pcref, sock);

    }

    void dispatchPrivateChatSubject(HLProtocol.Packet packet) throws IOException {

	int pcref = 0;
	byte[] subject = new byte[0];

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLS_DATA_CHAT_REF:
		pcref = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_CHAT_SUBJECT:
		subject = dh.data;
		break;

	    }

	}

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handlePrivateChatSubject(pcref, new String(subject));

    }

    void dispatchPrivateChatInvite(HLProtocol.Packet packet) throws IOException {

	int sock = 0, pcref = 0;
	byte[] nick = new byte[0];

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLS_DATA_SOCKET:
		sock = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_CHAT_REF:
		pcref = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_NICK:
		nick = dh.data;
		break;

	    }

	}

	if(nick != null) 
            for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
                ((HLClientListener) en.nextElement()).
		    handlePrivateChatInvite(pcref, sock, new String(nick));

    }
	
    Object dispatchPrivateChatCreate(HLProtocol.Packet packet) throws IOException {

	int sock = 0, pcref = 0, icon = 0, color = 0;
	byte[] nick = new byte[0];

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLS_DATA_SOCKET:
		sock = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_ICON:
		icon = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_NICK:
		nick = dh.data;
		break;
	    case HLProtocol.HTLS_DATA_COLOUR:
		color = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLS_DATA_CHAT_REF:
		pcref = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    default:
		DebuggerOutput.debug("dispatchPrivateChatCreate: unknown dh type: " + dh.type);
		break;

	    }

	}

        DebuggerOutput.debug("calling listener for private chat create");
        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
		handlePrivateChatCreate(pcref, sock, new String(nick), icon, color);

        return null;

    }

    Object dispatchPrivateChatJoin(HLProtocol.Packet packet) throws IOException {

	Vector userV = new Vector();

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLS_DATA_USER_LIST:
		userV.addElement((HLProtocol.UserListComponent) dh);
		break;

	    default:
		DebuggerOutput.debug("dispatchPrivateChatJoin: unknown dh type: " + dh.type);
		break;

	    }

	}

	HLProtocol.UserListComponent[] users = 
	    new HLProtocol.UserListComponent[userV.size()];
	for(int i=0; i<userV.size(); i++) 
	    users[i] = (HLProtocol.UserListComponent) userV.elementAt(i);
        userV.removeAllElements();

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
		handlePrivateChatJoin(packet.header.trans, users);

        return users;

    }	

    /* Chat, messagse */

    void dispatchChat(HLProtocol.Packet packet) throws IOException {

	byte[] msg = new byte[0];
	int pcref = -1;

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLC_DATA_CHAT_REF:
		pcref = ToArrayConverters.byteArrayToInt(dh.data);
		break;

	    case HLProtocol.HTLC_DATA_CHAT:
		msg = dh.data;
		break;

	    }

	}

	if(msg != null) {

	    if(pcref == -1) 
                for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
                    ((HLClientListener) en.nextElement()).
			handleChat(new String(msg));
	    else 
                for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
                    ((HLClientListener) en.nextElement()).
			handlePrivateChat(pcref, new String(msg));

	}

    }

    void dispatchMessage(HLProtocol.Packet packet) throws IOException {

	byte[] msg = new byte[0]; 
	byte[] nick = null;
	int sock = 0;

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLC_DATA_SOCKET:
		sock = ToArrayConverters.byteArrayToInt(dh.data);
		break;
	    case HLProtocol.HTLC_DATA_MSG:
		msg = dh.data;
		break;
	    case HLProtocol.HTLC_DATA_NICK:
		nick = dh.data;
		break;
            default:
                DebuggerOutput.debug("dispatchMessage: unknown dh.type: 0x" + Integer.toHexString(dh.type));
                break;

	    }

	}

	if(nick == null)

	    for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
		((HLClientListener) en.nextElement()).
		    handleAdministratorMessage(new String(msg));
	
	else

	    for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
		((HLClientListener) en.nextElement()).
		    handleMessage(sock, new String(nick), new String(msg));

    }

    /* Files */

    void dispatchTransferQueuePos(HLProtocol.Packet packet) throws IOException {

        int ref = -1;
        int queuepos = -1;
        
        while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_HTXF_REF:
                ref = ToArrayConverters.byteArrayToInt(dh.data);
                break;
            case HLProtocol.HTLS_DATA_HTXF_QUEUEPOS:
                queuepos = ToArrayConverters.byteArrayToInt(dh.data);
                break;

            }

        }
        
        if(ref != -1 && queuepos != -1) 
	    for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
		((HLClientListener) en.nextElement()).
                    handleTransferQueuePosition(ref, queuepos);



    }

    Object dispatchFileGet(HLProtocol.Packet packet) throws IOException {

        long size = 0, fileSize = 0;
        int ref = 0;
        int queuepos = -1;

        while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_HTXF_SIZE:
                size = ToArrayConverters.byteArrayToInt(dh.data);
                break;
            case HLProtocol.HTLS_DATA_FILE_SIZE:
                fileSize = ToArrayConverters.byteArrayToInt(dh.data);
                break;
            case HLProtocol.HTLS_DATA_HTXF_REF:
                ref = ToArrayConverters.byteArrayToInt(dh.data);
                break;
            case HLProtocol.HTLS_DATA_HTXF_QUEUEPOS:
                queuepos = ToArrayConverters.byteArrayToInt(dh.data);
                break;

            }

        }

        HLProtocol hlp = new HLProtocol();
        Socket xfsocket = null;
        Transferrer transferrer = null;
        PipedOutputStream outputPipe = null;
        Transfer xf = hlc.getTransfer(packet.header.trans);

        if(xf == null)
            throw new IOException("Invalid transfer ID " + packet.header.trans);

        try { 


            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: creating Transferrer");
            transferrer = new Transferrer(packet.header.trans, ref, xf.meter);


            if(xf.forkOrFile == HLProtocol.DL_DATA_FORK) {

                /* Get just the data fork, no headers, and stream that
                   data to a pipe. */

                DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: creating output pipe");
                outputPipe = new PipedOutputStream();
                xf.inputPipe.connect(outputPipe);

            }

            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: starting Meter");

            xf.meter.startMeter(transferrer, ref, xf.name, size);

            if(queuepos != -1) {

                /* If we got a queue parameter, then tell the Meter
                   we're about to start, and immediately notify the
                   event handler about the queue position. */

                DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: got queuepos = " + queuepos + ", calling handleTransferQueuePosition");
                for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
                    ((HLClientListener) en.nextElement()).
                        handleTransferQueuePosition(ref, queuepos);

            }

            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: opening transfer socket");

            xfsocket = new Socket(hlc.getHost(), hlc.getPort() + 1);
            xfsocket.setSoTimeout(500);
            xfsocket.setSoLinger(true, 5);
            xfsocket.setReceiveBufferSize(HLProtocol.SOCK_RECEIVE_BUF_SIZE);

            DataOutputStream xfoutput = 
                new DataOutputStream(new BufferedOutputStream(xfsocket.getOutputStream()));
            DataInputStream xfinput = 
                new DataInputStream(new BufferedInputStream(new InterruptableInputStream(xfsocket.getInputStream())));

            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: setting up Transferrer");

            transferrer.setDataInputStream(xfinput);
            transferrer.setDataOutputStream(xfoutput);

            /* Send the file transfer header and receive the file
               transfer info in reply. */

            HLProtocol.FileTransferHeader fth =
                hlp.new FileTransferHeader(ref, 0);
            
            fth.write(xfoutput);

            HLProtocol.FileTransferInfo fti = null;

            if(xf.forkOrFile != HLProtocol.DL_DATA_FORK) {

                fti = hlp.new FileTransferInfo(xfinput);
                
                /* Initialize the local file. */
                
                if(xf.local.getDataSize() == 0)
                    xf.local.writeHeader(fti.fileName, 
                                         fti.fileType,
                                         fti.fileCreator,
                                         fti.fileComment,
                                         fti.creationDate,
                                         fti.modificationDate,
                                         fti.finderFlags);

            }
                
            byte[] l = null;

            if(xf.forkOrFile != HLProtocol.DL_DATA_FORK) {
                
                /* Get the data fork size. Maybe should check whether
                   magic matches something (e.g. "DATA"). This reads the
                   file size as a 64 bit long integer. */
                
                l = new byte[8];
                xfinput.skipBytes(8); 
                xfinput.readFully(l, 0, 8); 
                size = ToArrayConverters.byteArrayToLong(l);

                /* Resume. */

                xf.local.getDataFork().seek(xf.local.getDataFork().size());
                
            }

            /* Transfer the data fork. */

            DataOutput output;

            /* Setup the output to go to a file for downloads or
               to a pipe for viewing. */

            if(xf.forkOrFile != HLProtocol.DL_DATA_FORK) {

                output = xf.local.getDataFork().getDataOutput();
                
            } else {
                
                output = new DataOutputStream(outputPipe);
                
            }

            try {

                DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: transferring data fork ...");
                transferrer.transferDataFork(xfinput, output, size);

            } catch (InterruptedException e) {

                throw e;

            } catch (IOException e) {

                throw e;

            } finally {

                if(xf.forkOrFile != HLProtocol.DL_DATA_FORK) {

                    DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: closing data fork.");
                    xf.local.setDataSize(xf.local.getDataSize() + transferrer.getDataBytesWritten());
                    xf.local.getDataFork().close();                        

                }
                    
            }

            if(xf.forkOrFile != HLProtocol.DL_DATA_FORK) {
                
                size = 0;
                
                if(fti.numberOfBlocks == 3) {
                    
                    /* Should check for MACR. */
                    
                    xfinput.skipBytes(8);
                    xfinput.readFully(l, 0, 8);
                    size = ToArrayConverters.byteArrayToLong(l);
                    
                }
                
                /* Resume. */

                xf.local.getResourceFork().seek(xf.local.getResourceFork().size());
                
                /* Transfer the resource fork. */
                
                try {
                    
                    DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: transferring resource fork");
                    transferrer.
                        transferResourceFork(xfinput, 
                                             xf.local.getResourceFork().
                                             getDataOutput(), size);
                    
                } catch (InterruptedException e) {
                    
                    throw e;
                    
                } catch (IOException e) {
                    
                    throw e;

                } finally {
                 
                    DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: closing resource fork");
                    xf.local.setResourceSize(xf.local.getResourceSize() + transferrer.getResourceBytesWritten());
                    xf.local.getResourceFork().close();

                }
                
                /* Tell the Meter that the transmission is over. */

            }

            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: stopping meter");
            xf.meter.stopMeter(ref);

            if(xf.forkOrFile != HLProtocol.DL_DATA_FORK) {

                /* Set MacOS attributes on file if possible. */

                MacFileUtils.setFileTypeAndCreator(xf.local.getFile(), fti.fileType.getBytes(), fti.fileCreator.getBytes());

            }

        } catch(InterruptedException e) {

            /* This doesn't need handling, it is thrown when the
               Transferrer is interrupted by it's Meter. */

            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: caught InterruptedException");

        } catch(IOException e) {

            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: caught IOException: " + e.toString());
            DebuggerOutput.stackTrace(e);

            if(!transferrer.interrupted)
                xf.meter.stopMeterWithError(ref, e);

        } finally {

            DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: in final block");

            try {

                if(outputPipe != null) {

                    outputPipe.flush();
                    outputPipe.close();
                    
                }

                if(xf.forkOrFile != HLProtocol.DL_DATA_FORK)
                    if(xf.local != null)
                        xf.local.cleanup();

                if(xfsocket != null) {

                    DebuggerOutput.debug("dispatchFileGet[" + packet.header.trans + "]: closing socket");
                    xfsocket.close();

                }

            } catch(IOException e) { }

        }

        return null;

    }

    Object dispatchFilePut(HLProtocol.Packet packet) throws IOException { 
        
        int size = 0, ref = 0;
        HLProtocol.ResumeTransferComponent rflt = hlp.new ResumeTransferComponent();

        while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_HTXF_REF:
                ref = ToArrayConverters.byteArrayToInt(dh.data);
                break;
            case HLProtocol.HTLS_DATA_HTXF_RFLT:
                rflt = (HLProtocol.ResumeTransferComponent) dh;
                break;
            default:
                DebuggerOutput.debug("dispatchFilePut: unknown dh.type: " + dh.type);
                break;
                
            }

        }

        Transferrer transferrer = null;
        DataOutputStream xfoutput = null;
        Transfer xf = hlc.getTransfer(packet.header.trans);

        if(xf == null)
            throw new IOException("Invalid transaction ID " + packet.header.trans);
        
        Socket xfsocket = null;
        long rflt_data_size = ((Long) rflt.chunks.get("DATA")).longValue();
        long rflt_macr_size = ((Long) rflt.chunks.get("MACR")).longValue();
        long dataRemaining = xf.local_data_size - rflt_data_size;
        long rsrcRemaining = xf.local_rsrc_size - rflt_macr_size;

        try {
            
            /* Tell the Transferrer we're starting the
               transmission. */
            
            transferrer = new Transferrer(packet.header.trans, ref, xf.meter);
            xf.meter.startMeter(transferrer,
                                ref,
                                xf.local.getFile().getName(), 
                                dataRemaining + rsrcRemaining);

            if(dataRemaining < 0 || rsrcRemaining < 0) 
                throw new IOException("Attempt to resume beyond end of file");
                    
            xfsocket = new Socket(hlc.getHost(), hlc.getPort() + 1);
            xfsocket.setSoTimeout(2500);
            xfsocket.setSoLinger(true, 5);
            xfsocket.setSendBufferSize(HLProtocol.SOCK_SEND_BUF_SIZE);
            
            xfoutput = new DataOutputStream(xfsocket.getOutputStream());
            DataInputStream xfinput = 
                new DataInputStream(new InterruptableInputStream(xfsocket.getInputStream()));

            transferrer.setDataInputStream(xfinput);
            transferrer.setDataOutputStream(xfoutput);
            
            /* Construct the FILP file transfer info. 
               Changes here mean changes in HLClient.java. */

            HLProtocol.FileTransferInfo fti = 
                hlp.new FileTransferInfo(xf.local.getFile().getName(),
                                         xf.local.getType(),
                                         xf.local.getCreator(),
                                         xf.local.getComment(),
                                         xf.local.getCreationDate(),
                                         xf.local.getModificationDate(),
                                         xf.local.getFinderFlags());

            /* Construct the HTXF file upload header 
               (16 = size of HTXF header). */
            
            HLProtocol.FileTransferHeader fth = 
                hlp.new FileTransferHeader(ref, 16 + 
                                           (long) (dataRemaining +
                                                   rsrcRemaining + 
                                                   fti.data.length));

            /* Write the HTXF and FILP headers. */

            fth.write(xfoutput);
            fti.write(xfoutput);
            xfoutput.flush();

	    /* Write the DATA block. Use a long for the file size. */

            xfoutput.write(new String("DATA").getBytes());
            xfoutput.writeInt(0);
            xfoutput.writeLong(dataRemaining);
            xfoutput.flush();

            /* Transmit the data fork. */

            if(dataRemaining > 0) {

                xf.local.getDataFork().seek(rflt_data_size);
                transferrer.transferDataFork(xf.local.getDataFork().getDataInput(), xfoutput, dataRemaining);

            }

	    /* Write the MACR block. */

            xfoutput.write(new String("MACR").getBytes());
            xfoutput.writeInt(0);
            xfoutput.writeLong(rsrcRemaining);
            xfoutput.flush();

            /* Transmit the resource fork. */

            if(rsrcRemaining > 0) {

                xf.local.getResourceFork().seek(rflt_macr_size);
                transferrer.transferResourceFork(xf.local.getResourceFork().getDataInput(), xfoutput, rsrcRemaining);
            
            }

            xfoutput.flush();

            if(!HLProtocol.asynchronousFileTransfers) {

                try {
                    
                    /* Wait for the server to close the connection. */
                    
                    xfinput.read();
                    
                } catch(IOException e) {}
                
            }

            /* Tell the Meter that the transmission 
               has ended. */
            
            xf.meter.stopMeter(ref);
            
        } catch(InterruptedException e) {
            
            /* This doesn't need handling, it is thrown when the
               Transferrer is interrupted by it's Meter. */
            
        } catch(IOException e) {
            
            xf.meter.stopMeterWithError(ref, e);
            
        } finally {
            
            try {
                
                if(xfoutput != null) {

                    xfoutput.close();

                }

                if(xfsocket != null) {

                    xfsocket.close();

                }
                
            } catch(IOException e) {}
            
        }
        
        return null;

    }

    Object dispatchFileList(HLProtocol.Packet packet) throws IOException {

        Vector fileV = new Vector();

        while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_FILE_LIST:
                fileV.insertElementAt((HLProtocol.FileListComponent) dh, 0);
                break;

            }

        }

        HLProtocol.FileListComponent[] files = 
            new HLProtocol.FileListComponent[fileV.size()];	

        for(int i=0; i<fileV.size(); i++)
            files[i] = (HLProtocol.FileListComponent) fileV.elementAt(i);
        fileV.removeAllElements();

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handleFileList(packet.header.trans, files);

	return files;

    }

    /* Task dispatch */
    
    void dispatchTask(HLProtocol.Packet packet) throws IOException, HLTaskNotFoundException {

        HLTask task = (HLTask) hlc.getTask(packet.header.trans);
        Object resultValue = null;

        try {
            
            if(packet.header.isError == 1) {
                
                /* In case of an error. */
                
                String error = new String("Unspecified error.");
                
                while(packet.hasMoreComponents()) {
                    
                    HLProtocol.DataComponent dh = packet.nextComponent();
                    
                    switch(dh.type) {
                        
                    case HLProtocol.HTLS_DATA_TASKERROR: 
                        error = new String(dh.data);
                        break;

                    }

                }

                resultValue = error;

            } else {

                switch(task.type) {
                    
                case HLProtocol.HTLC_HDR_LOGIN:
                    dispatchLogin(packet);
                    break;
                case HLProtocol.HTLC_HDR_DIR_LIST:
                    resultValue = dispatchFileList(packet);
                    break;
                case HLProtocol.HTLC_HDR_USER_GETINFO:
                    resultValue = dispatchUserInfo(packet);
                    break;
                case HLProtocol.HTLC_HDR_USER_LIST:
                    resultValue = dispatchUserList(packet);
                    break;
                case HLProtocol.HTLC_HDR_NEWS_GET:
                    resultValue = dispatchNews(packet);
                    break;
                case HLProtocol.HTLC_HDR_FILE_GET:
                    resultValue = dispatchFileGet(packet);
                    break;
                case HLProtocol.HTLC_HDR_FILE_PUT:
                    resultValue = dispatchFilePut(packet);
                    break;
                case HLProtocol.HTLC_HDR_PRIVCHAT_CREATE:
                    resultValue = dispatchPrivateChatCreate(packet);
                    break;
                case HLProtocol.HTLC_HDR_PRIVCHAT_JOIN:
                    resultValue = dispatchPrivateChatJoin(packet);
                    break;
                case HLProtocol.HTLC_HDR_ACCOUNT_READ:
                    resultValue = dispatchAccountInfo(packet);
                    break;
                case HLProtocol.HTLC_HDR_FILE_GETINFO:
                    resultValue = dispatchFileInfo(packet);
                    break;
                default:
                    DebuggerOutput.debug("dispatchTask[" + packet.header.trans + "] task [" + packet.header.trans + "] undispatched: " + task);
                    break;
                    
                }
                
            }
            
        } finally {

            /* Notify threads waiting on completion of this task. */

            if(packet.header.isError == 1) 
                task.setError(new HLTaskErrorException((String) resultValue));
            else 
                task.setData(resultValue);

            /* Either we dispose this task or someone else does it
               (i.e. HLClient.waitFor()). */

            if(task.disposeWhenReceived) {

                DebuggerOutput.debug("dispatchTask: task[" + packet.header.trans + "]: disposing task ...");
                hlc.disposeTask(packet.header.trans);

            }

            if(packet.header.isError == 1) {

                for(Enumeration en = hlc.hle.elements(); 
                    en.hasMoreElements(); )
                    ((HLClientListener) en.nextElement()).
                        handleTaskError(packet.header.trans, (String) resultValue);
            
            } else {
                
                for(Enumeration en = hlc.hle.elements(); 
                    en.hasMoreElements(); )
                    ((HLClientListener) en.nextElement()).
                        handleTaskComplete(packet.header.trans);
            
            }

            DebuggerOutput.debug("dispatchTask: task[" + packet.header.trans + "]: dispatcher done.");

        }

    }

    Object dispatchFileInfo(HLProtocol.Packet packet) throws IOException {

        byte[] icon = new byte[0];
        byte[] type = new byte[0];
        byte[] creator = new byte[0];
        byte[] name = new byte[0];
        byte[] comment = new byte[0];
        long size = 0;
        Calendar created = Calendar.getInstance();
        Calendar modified = Calendar.getInstance();

        while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_FILE_ICON:
                icon = dh.data;
                break;
            case HLProtocol.HTLS_DATA_FILE_TYPE:
                type = dh.data;
                break;
            case HLProtocol.HTLS_DATA_FILE_CREATOR:
                creator = dh.data;
                break;
            case HLProtocol.HTLS_DATA_FILE_NAME:
                name = dh.data;
                break;
            case HLProtocol.HTLS_DATA_FILE_COMMENT:
                comment = dh.data;
                break;
            case HLProtocol.HTLS_DATA_FILE_DATE_CREATED:
                created = ((HLProtocol.DateComponent) dh).date;
                break;
            case HLProtocol.HTLS_DATA_FILE_DATE_MODIFIED:
                modified = ((HLProtocol.DateComponent) dh).date;
                break;
            case HLProtocol.HTLS_DATA_FILE_SIZE:
                size = ToArrayConverters.byteArrayToLong(dh.data);
                break;

            }

        }

        HLProtocol.FileInfo info = hlp.new FileInfo();
        info.icon = new String(icon);
        info.type = new String(type);
        info.creator = new String(creator);
        info.name = new String(name);
        info.comment = new String(comment);
        info.size = size;
        info.created = created;
        info.modified = modified;

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handleFileInfo(packet.header.trans, info);       

        return info;

    }

    /* Accounts */

    Object dispatchAccountInfo(HLProtocol.Packet packet) throws IOException {

        byte[] nick = new byte[0];
        byte[] password = new byte[0];
        byte[] login = new byte[0];
        byte[] access2 = new byte[0];
        long access = 0;

        while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {

                switch(dh.type) {
                    
                case HLProtocol.HTLS_DATA_NICK:
                    nick = dh.data;
                    break;
                case HLProtocol.HTLS_DATA_LOGIN:
                    login = HLProtocol.invert(dh.data);
                    break;
                case HLProtocol.HTLS_DATA_PASSWORD:
                    password = HLProtocol.invert(dh.data);
                    break;
                case HLProtocol.HTLS_DATA_PRIVILEGES:
                    access = ToArrayConverters.byteArrayToLong(ToArrayConverters.swapByteArray(dh.data));
                    break;
                    
                }

            }

        }
	
        HLProtocol.AccountInfo account = hlp.new AccountInfo(new String(login), new String(nick), new String(password), access, null);

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handleAccountInfo(packet.header.trans, account);

        return account;

    }
    
    /* Users */

    Object dispatchUserList(HLProtocol.Packet packet) throws IOException {

        Vector userV = new Vector();

        while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_USER_LIST:
                userV.insertElementAt((HLProtocol.UserListComponent) dh, 0);
                break;

            }

        }

        HLProtocol.UserListComponent[] users = 
            new HLProtocol.UserListComponent[userV.size()];	

        for(int i=0; i<userV.size(); i++) 
            users[i] = (HLProtocol.UserListComponent) userV.elementAt(i);
	userV.removeAllElements();

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handleUserList(packet.header.trans, users);

        return users;

    }
    
    Object dispatchUserInfo(HLProtocol.Packet packet) throws IOException {

        String userinfo = new String();

        while(packet.hasMoreComponents()) {

            HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_USER_INFO:
                userinfo = new String(dh.data);
                break;

            }

        }

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handleUserInfo(packet.header.trans, userinfo);

        return userinfo;

    }

    void dispatchUserChange(HLProtocol.Packet packet) throws IOException {

        int icon = 0, sock = 0, color = 0;
        byte[] nick = new byte[0];

        while(packet.hasMoreComponents()) {

            HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_SOCKET:
                sock = ToArrayConverters.byteArrayToInt(dh.data);
                break;
            case HLProtocol.HTLS_DATA_ICON:
                icon = ToArrayConverters.byteArrayToInt(dh.data);
                break;
            case HLProtocol.HTLS_DATA_NICK:
                nick = dh.data;
                break;
            case HLProtocol.HTLS_DATA_COLOUR:
                color = ToArrayConverters.byteArrayToInt(dh.data);
                break;

            }

        }

        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handleUserChange(sock, new String(nick), icon, color);

    }

    void dispatchUserLeave(HLProtocol.Packet packet) throws IOException {
	
        int sock = 0;
	
        while(packet.hasMoreComponents()) {

            HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_SOCKET:
                sock = ToArrayConverters.byteArrayToInt(dh.data);
                break;

            }

        }
	
        for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
            ((HLClientListener) en.nextElement()).
                handleUserLeave(sock);

    }

    /* News post, get */

    void dispatchNewsPost(HLProtocol.Packet packet) throws IOException {

        String post = null;

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();

	    switch(dh.type) {

	    case HLProtocol.HTLC_DATA_NEWS_POST:
                post = new String(dh.data);
		break;
                
	    }

	}

        if(post != null)
            for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
                ((HLClientListener) en.nextElement()).
                    handleNewsPost(post);

    }

    Object dispatchNews(HLProtocol.Packet packet) throws IOException {

        String news = null;

	while(packet.hasMoreComponents()) {

	    HLProtocol.DataComponent dh = packet.nextComponent();
            
	    switch(dh.type) {

	    case HLProtocol.HTLS_DATA_NEWS:
                news = new String(dh.data);
		break;

	    }

	}

        if(news != null)
            for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); )
                ((HLClientListener) en.nextElement()).
                    handleNews(packet.header.trans, news);
        
        return news;

    }
    
    /* Agreement, login */

    synchronized void waitForAgreement() throws InterruptedException {

        if(agreementAborted)
            throw new InterruptedException("agreement aborted");

        DebuggerOutput.debug("HLClientDispatcher: waiting for agreement ...");
        while(!agreementReceived && !agreementAborted)
            wait();
        DebuggerOutput.debug("HLClientDispatcher: got agreement.");

        if(agreementAborted)
            throw new InterruptedException("agreement aborted");

    }

    synchronized void notifyAgreementAborted() {

        agreementAborted = true;
        notify();

    }

    synchronized void notifyAgreementReceived() {

        if(!agreementAborted) {
            
            agreementReceived = true;
            notify();

        }

    }

    void dispatchAgreement(HLProtocol.Packet packet) throws IOException {

        try {

            while(packet.hasMoreComponents()) {
                
                HLProtocol.DataComponent dh = packet.nextComponent();
                
                switch(dh.type) {
                    
                case HLProtocol.HTLS_DATA_AGREEMENT:
                    String agreement = new String(dh.data);
                    
                    for(Enumeration en = hlc.hle.elements(); en.hasMoreElements(); ) {
                        
                        boolean ok = ((HLClientListener) en.nextElement()).
                            handleAgreement(agreement);
                        
                        if(!ok)
                            agreementAccepted = false;
                        
                    }
                    
                    break;
                    
                case HLProtocol.HTLS_DATA_AUTO_AGREE:
                    agreementAccepted = true;
                    int autoAgree = ToArrayConverters.byteArrayToInt(dh.data);
                    break;
                    
                }

            }

        } finally {
            
            notifyAgreementReceived();
            
        }
        
    }        
    
    void dispatchLogin(HLProtocol.Packet packet) throws IOException {

        int version = 0;

        while(packet.hasMoreComponents()) {

            HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {

            case HLProtocol.HTLS_DATA_VERSION:
                version = ToArrayConverters.byteArrayToInt(dh.data);
                break;

            default:
                DebuggerOutput.debug("dispatchLogin: unknown dh.type: " + dh.type);
                break;

            }

        }

        hlc.setServerVersion(version);

    }

}
