/**
 * $Id: HLClientListener.java,v 1.2 2001/07/22 07:12:45 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

/**
 * This interface defines methods that are invoked whenever the
 * corresponding event is received from the server. 
 */
public interface HLClientListener {

    /**
     * Called when the queue position has changed for the file
     * transfer identified by "ref".
     * @param ref a unique reference to the file transfer.
     * @param position the new position in the queue.
     */
    public void handleTransferQueuePosition(int ref, int position);

    /**
     * Called when requested file info has been received.
     * @param id the transaction id of the requesting task
     * @param fi the file info
     */
    public void handleFileInfo(int id, HLProtocol.FileInfo fi);

    /**
     * Called when requested account info has been received.
     * @param id the transaction id of the requesting task
     * @param fi the account info
     */
    public void handleAccountInfo(int id, HLProtocol.AccountInfo ac);

    /**
     * Called when a request to join a private chat has been 
     * successfull.
     * @param id the transaction id of the requesting task
     * @param uh the list of users on the private chat
     */
    public void handlePrivateChatJoin(int id, HLProtocol.UserListComponent[] uh);

    /**
     * Called when a request to create a private chat with 
     * another user has been successfull.
     * @param pcref the reference for the private chat
     * @param sock the socket of the user 
     * @param nick the nickname of the user
     * @param icon the icon of the user
     * @param color the "color" of the user
     */
    public void handlePrivateChatCreate(int pcref, int sock, String nick, int icon, int color);

    /**
     * Called when the subject of the private chat changes.
     * @param pcref the reference for the private chat
     * @param subject the new subject for the private chat
     */
    public void handlePrivateChatSubject(int pcref, String subject);

    /**
     * Called when a user leaves a private chat.
     * @param pcref the reference for the private chat
     * @param sock the socket of the user that has left
     */
    public void handlePrivateChatUserLeave(int pcref, int sock);

    /**
     * Called when a user changes nick, color or icon.
     * @param pcref the reference for the private chat
     * @param sock the socket of the user
     * @param nick the (possibly new) nickname of the user
     * @param icon the (possibly new) icon of the user
     * @param color the (possibly new) "color" of the user
     */
    public void handlePrivateChatUserChange(int pcref, int sock, String nick, int icon, int color);

    /**
     * Called when text arrives from a private chat.
     * @param pcref the reference for the private chat
     * @param msg the text
     */
    public void handlePrivateChat(int pcref, String msg);

    /**
     * Called when a user invites us to a private chat.
     * @param pcref the reference for the private chat
     * @param sock the socket of the user inviting us to a private chat
     * @param nick the nickname of the user inviting us to a private chat
     */
    public void handlePrivateChatInvite(int pcref, int sock, String nick);

    /**
     * Called when text arrives from the chat.
     * @param msg the text
     */
    public void handleChat(String msg);

    /**
     * Called when a message arrives from the administrator.
     * @param msg the message
     */
    public void handleAdministratorMessage(String msg);

    /**
     * Called when a private message arrives from another user.
     * @param sock the socket of the sender
     * @param nick the nickname of the sender
     * @param msg the message
     */
    public void handleMessage(int sock, String nick, String msg);

    /**
     * Called when a request for a list of files has been
     * successfull.
     * @param id the transaction id of the requesting task
     * @param fh a list of files
     */
    public void handleFileList(int id, HLProtocol.FileListComponent[] fh);

    /**
     * Called when a request for a list of users has been
     * successfull.
     * @param id the transaction id of the requesting task
     * @param uh list of users
     */
    public void handleUserList(int id, HLProtocol.UserListComponent[] uh);

    /**
     * Called when a user changes nick, color or icon.
     * @param sock the socket of the user
     * @param nick the (possibly new) nickname of the user
     * @param icon the (possibly new) icon of the user
     * @param color the (possibly new) "color" of the user
     */
    public void handleUserChange(int sock, String nick, int icon, int color);

    /**
     * Called when a user leaves.
     * @param sock the socket of the user that has left
     */
    public void handleUserLeave(int sock);

    /**
     * Called when a request for user info has been successfull.
     * @param id the transaction id of the requesting task
     * @param info the requested user info
     */
    public void handleUserInfo(int id, String info);

    /**
     * Called when a request for getting news has been 
     * successfull.
     * @param id the transaction id of the requesting task
     * @param news the requested news
     */
    public void handleNews(int id, String news);

    /**
     * Called when a news post arrives.
     * @param msg the news post
     */
    public void handleNewsPost(String msg);

    /**
     * The handler for this event must display the agreement to the
     * user and return true if the agreement was accepted, or false
     * otherwise. What happens after that depends on the server
     * version: <p>
     * 
     * 1. On >= 1.5 servers, the {@link
     * redlight.hotline.HLClient#login} method blocks until
     * all agreement handlers have been run. If any of the agreement
     * handlers returns false, or if there are no agreement handlers,
     * the {@link redlight.hotline.HLClient#login} method
     * wakes up and receives a {@link AgreementRejectedException}.
     * In turn, {@link redlight.hotline.HLClient#login}
     * propagates the exception up to the caller. <p>
     * 
     * 2. On < 1.5 servers, the {@link
     * redlight.hotline.HLClient#login} method will not block
     * and basically ignore the agreement handlers altogether, so that
     * it becomes the responsibility of the application to check
     * the server version and disconnect when the agreement is rejected.
     * @param msg the agreement 
     */
    public boolean handleAgreement(String msg);

    /**
     * Called when the task with the given transaction id
     * completes successfully. Every task ultimately ends up
     * either here or in {@link #handleTaskError}.
     * @param id the transaction id of the task that has completed
     */
    public void handleTaskComplete(int id);

    /**
     * Called when the task with the given transaction id
     * does not complete successfully. Every task ultimately ends up
     * either here or in {@link #handleTaskComplete}.
     * @param id the transaction id of the failed task
     * @param error the error that occurred
     */
    public void handleTaskError(int id, String error);

    /**
     * Called when the connection disappears,
     * either unexpectedly, or because the server went off-line
     * (administrator shutdown). <p>
     * Implementations must perform the following actions: <p>
     *
     * 1. The {@link redlight.utils.Meter} objects associated with
     *    pending transfers must invoke the interrupt() method
     *    on their {@link redlight.utils.MeterSource} objects (see {@link 
     *    redlight.hotline.HLClient#requestFileDownload}
     *    and {@link 
     *    redlight.hotline.HLClient#requestFileUpload}). <p>
     *
     * 2. Call the {@link redlight.hotline.HLClient#close}
     *    method. 
     * @param msg the reason of the disconnection 
     */
    public void handleDisconnect(String msg);

}
