/**
 * $Id: HLServerDispatcher.java,v 1.12 2001/10/05 02:19:03 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.net.Socket;
import java.net.InetAddress;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.EOFException;
import java.io.File;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;
import java.text.DateFormat;

import redlight.crypto.UnixCrypt;
import redlight.utils.TimeFormat;
import redlight.utils.TextUtils;
import redlight.utils.ToArrayConverters;
import redlight.utils.InterruptableInputStream;
import redlight.utils.DebuggerOutput;
import redlight.macfiles.MacFile;

/**
 * Each user on the server is handled by a new instance of this class.
 */
public class HLServerDispatcher extends Thread {

    /* The socket that this dispatcher is connected to. */

    public Socket client;

    /* A lock for modifying / accessing the icon, color and nick
       parameters. */

    public Integer userStateLock = new Integer(1);

    /* Any accesses to the following from other threads must 
       synchronize on the userStateLock object. */

    public String nick = "???";
    public short icon = 0, color = 0;

    /* Can be read but not written by other threads. */

    public String login = "guest", password = "guest";
    public short sock = -1;
    public String homeDirectory;
    public long privileges;
    public int downloadsInProgress = 0, uploadsInProgress = 0,
        downloadsInQueue = 0, uploadsInQueue = 0;
    public long lastTransmissionTime = 0, firstTransmissionTime = 0;

    /* A bit of storage for the HLServerPolicy that approves the
       actions for this client. */

    public Object policyObject;

    /* Internal, not used by any other threads. */

    private long floodMarker = 0;
    private int numberOfPacketsSinceFloodMarker = 0;
    private int trans = 0;
    private int protocolErrors = 0;
    private boolean protocolErrorOccurredDuringDispatch = false;
    private HLServer hls;
    private DataInputStream input;
    private DataOutputStream output;
    private boolean connected = false, disconnected = false;

    /**
     * Creates a dispatcher for the given server and a client
     * connection on the given socket.
     * @param s the server that this dispatcher belongs to.
     * @param c the client that this dispatcher works for.
     */
    public HLServerDispatcher(HLServer s, Socket c) {

        hls = s;
        client = c;
        numberOfPacketsSinceFloodMarker = 0;
        floodMarker = 0;
        policyObject = hls.hsp.createPolicyObject();
        setName("HLServerDispatcher " + c);

    }

    synchronized int nextTrans() {

        return this.trans++;

    }

    /**
     * Returns a UserListComponent containing some info on
     * this client.
     * @return UserListComponent for this client.
     */
    public HLProtocol.UserListComponent getUser() {

        synchronized(userStateLock) {

            return hls.hlp.new UserListComponent(sock, icon, color, nick);

        }

    }

    /**
     * Notifies the client that the given transfer's queue position
     * changed.
     */
    void sendQueuePosition(int ref, int position) {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] { 
                
                hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_HTXF_REF, ToArrayConverters.intToByteArray(ref)),
                hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_HTXF_QUEUEPOS, ToArrayConverters.intToByteArray(position)),

            };
                    
        send(hls.hlp.new Packet(HLProtocol.HTLS_HDR_QUEUEPOS, nextTrans(), dataComponents));

    }

    /**
     * Sends the given packet to the client.
     * @param packet the packet to send.
     */
    void send(HLProtocol.Packet packet) {
        
        if(connected) {

            try {

                if(!(packet.header.id == HLProtocol.HTLS_HDR_CHAT &&
                     !((privileges & HLProtocol.AccountInfo.CAN_SEND_CHAT) == 
                       HLProtocol.AccountInfo.CAN_SEND_CHAT))) {

                    /* Only send this packet after verifying that we are
                       not trying to send a chat packet to a user which
                       does not have the privileges to read chat. */
                    
                    DebuggerOutput.debug("HLServerDispatcher.send: sending packet " + packet.toString());
                    packet.write(output);

                }

            } catch(IOException e) {

                /* Disconnect the client if writing to it yields an error. */

                hls.log(client.getInetAddress(), "Disconnected client because of a write error: " + e.getMessage());
                connected = false;

            }
        
        }
        
    }

    /**
     * Convenience method for send() to make the packet look like
     * a task reply (sets {@link HLProtocol.PacketHeader.cls} to 1).
     * @param packet the packet to transform into a task reply.
     */
    void sendTaskReply(HLProtocol.Packet packet) {

        packet.header.cls = 1;
        send(packet);

    }

    /**
     * Convenience method to send an error message to the client.
     * @param trans the transaction that failed.
     * @param msg the error message.
     */
    void sendTaskError(int trans, String msg) {
        
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_TASKERROR, msg.getBytes()),
                
            };
        
        HLProtocol.Packet p = hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, trans, dataComponents);
        p.header.isError = 1;
        p.header.cls = 1;

        send(p);

    }

    /**
     * Disconnects the client from the server.
     */
    synchronized void disconnect() {

        DebuggerOutput.debug("HLServerDispatcher.disconnect: disconnecting ...");

        /* This will cause the dispatcher thread to terminate
           on the next iteration. */

        connected = false;

        /* This is to prevent the race that occurs when disconnect()
           is called before castSpells(). */

        disconnected = true;

        try {

            /* Close the socket so any pending reads / writes 
               are interrupted and the dispatcher thread can
               unroll. */

            client.close();

        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

        }

        DebuggerOutput.debug("HLServerDispatcher.disconnect: exiting");

    }

    /**
     * Read and send required magic bytes and set up some necessary
     * variables.
     */
    synchronized void castSpells() throws IOException {
        
        if(!disconnected) {

            output = new DataOutputStream(new BufferedOutputStream(client.getOutputStream()));
            input = new DataInputStream(new InterruptableInputStream(client.getInputStream()));
            
            /* First, read the magic bytes from the client. */
            
            byte[] buf = new byte[HLProtocol.HTLC_MAGIC_LEN];
            input.readFully(buf, 0, HLProtocol.HTLC_MAGIC_LEN);
            DebuggerOutput.debug("HLServerDispatcher.run: read client magic");
            
            if(!HLProtocol.HTLC_MAGIC.equals(new String(buf))) {
                
                client.close();
                hls.log(client.getInetAddress(), "Not a Hotline client: bad magic");
                
            } else {
                
                /* Then, send the magic server bytes. */
                
                output.writeBytes(HLProtocol.HTLS_MAGIC);
                output.flush();
                DebuggerOutput.debug("HLServerDispatcher.run: wrote server magic");
                connected = true;
                
            }

        }
        
    }

    /**
     * Reads packets from the client and responds to them.
     * The general strategy to disconnect a client is to
     * kill this thread by setting the 'connected' variable to
     * false.
     */
    public void run() {

        firstTransmissionTime = System.currentTimeMillis();
        lastTransmissionTime = firstTransmissionTime;

        hls.log(client.getInetAddress(), "Accepted connection.");

        /* If an exception occurs, we close the socket and inform
           the server that the connection has been closed. */

        try {
            
            /* Exchange magic strings. */

            castSpells();

            /* Now wait for the client to defend itself. Also, we
               should set a timer here to disconnect the client if
               it doesn't respond quickly enough. */
            
            protocolErrorOccurredDuringDispatch = false;

            while(connected)
                dispatch(hls.hlp.new Packet(input));
        
            if(protocolErrorOccurredDuringDispatch == false)
                protocolErrors = 0;

        } catch(EOFException e) {

            if(connected) {

                /* Unexpected error (not an explicit disconnect). */

                if(e.getMessage() != null) {

                    hls.log(client.getInetAddress(), "Client dropped connection: " + e.getMessage());

                } else {

                    hls.log(client.getInetAddress(), "Client dropped connection.");

                }
                
            }

        } catch(IOException e) {

            if(connected) {

                /* Unexpected error (not an explicit disconnect). */

                if(e.getMessage() != null) {

                    hls.log(client.getInetAddress(), "I/O error: " + e.getMessage());

                } else {

                    hls.log(client.getInetAddress(), "Unspecified I/O error.");

                }
                
            }

        } catch(Exception e) {

            if(connected) {

                /* Unexpected error (not an explicit disconnect). */

                e.printStackTrace();

                if(e.getMessage() != null) {

                    hls.log(client.getInetAddress(), "Internal error: " + e.getMessage());

                } else {

                    hls.log(client.getInetAddress(), "Unspecified internal error.");

                }

            }

        } finally {

            try {

                client.close();

            } catch(IOException _e) {}

            /* First inform the server that this client no longer exists,
               then broadcast a "user leave" event. */

            hls.log(client.getInetAddress(), "Connection closed.");
            hls.removeClient(this);

        }

        DebuggerOutput.debug("HLServerDispatcher.run: exiting");

    }

    /** 
     * Tries to disconnect clients who are flooding the server
     * before they do too much damage.
     * @return true when the client should be disconnected ASAP.
     */
    boolean dam() {

        /* Try to determine whether we are being flooded. Flooding is
           defined as receiving N packets within M seconds. */

        numberOfPacketsSinceFloodMarker++;

        if(numberOfPacketsSinceFloodMarker == 25) {

            DebuggerOutput.debug("floodMarker = " + floodMarker + ", currentTime = " + System.currentTimeMillis() + ", delta = " + (System.currentTimeMillis() - floodMarker));

            if(System.currentTimeMillis() - floodMarker < 500) {

                /* Uh oh, client is flooding us. */
                
                hls.log(client.getInetAddress(), 
                        "Disconnecting client because of flooding.");
                connected = false;
                return true;

            }

            floodMarker = System.currentTimeMillis();
            numberOfPacketsSinceFloodMarker = 0;

        }

        return false;

    }

    /**
     * Takes a packet (ie. header + payload) and figures out what to 
     * do with it, relaying it to more fine-grained methods.
     */
    void dispatch(HLProtocol.Packet packet) {

        DebuggerOutput.debug("HLServerDispatcher.dispatch: got packet " + packet.toString());

        /* Check whether we are being flooded. */

        if(dam())
            return;

        /* Record time of this transmission. */

        lastTransmissionTime = System.currentTimeMillis();

        /* Unidle us if we're idle. */

        if(hls.hsp.isAuthenticated(this))            
            if((color & HLProtocol.UserListComponent.HAS_BEEN_IDLE) ==
               HLProtocol.UserListComponent.HAS_BEEN_IDLE)
                hls.changeUser(this, sock, nick, icon, (short) (color & ~HLProtocol.UserListComponent.HAS_BEEN_IDLE));

        /* Approve the packet before processing it (this ensures that
           we don't accept any requests until the user is logged
           in). */

        String packetAllowed = hls.hsp.approvePacket(this, packet);

        if(packetAllowed != null) {
            
            hls.log(client.getInetAddress(), packetAllowed);
            connected = false;
            return;

        }

        /* Process the packet. */

        switch(packet.header.id) {

        case HLProtocol.HTLC_HDR_LOGIN:
            dispatchLogin(packet);
            break;

        case HLProtocol.HTLC_HDR_USER_CHANGE:
            dispatchUserChange(packet);
            break;
            
        case HLProtocol.HTLC_HDR_CHAT:
            dispatchChat(packet);
            break;
            
        case HLProtocol.HTLC_HDR_USER_LIST:
            dispatchUserList(packet);
            break;
            
        case HLProtocol.HTLC_HDR_NEWS_GET:
            dispatchFlatnewsGet(packet);
            break;
            
        case HLProtocol.HTLC_HDR_NEWS_POST:
            dispatchFlatnewsPost(packet);
            break;
            
        case HLProtocol.HTLC_HDR_ACCOUNT_READ:
            dispatchAccountRead(packet);
            break;
            
        case HLProtocol.HTLC_HDR_ACCOUNT_CREATE:
            dispatchAccountCreate(packet);
            break;
            
        case HLProtocol.HTLC_HDR_ACCOUNT_MODIFY:
            dispatchAccountModify(packet);
            break;
            
        case HLProtocol.HTLC_HDR_ACCOUNT_DELETE:
            dispatchAccountDelete(packet);
            break;
            
        case HLProtocol.HTLC_HDR_MSG:
            dispatchMessage(packet);
            break;
            
        case HLProtocol.HTLC_HDR_USER_KICK:
            dispatchUserKick(packet);
            break;
            
        case HLProtocol.HTLC_HDR_DIR_LIST:
            dispatchDirectoryListing(packet);
            break;
            
        case HLProtocol.HTLC_HDR_FILE_GET:
            dispatchFileDownload(packet);
            break;
            
        case HLProtocol.HTLC_HDR_FILE_PUT:
            dispatchFileUpload(packet);
            break;
            
        case HLProtocol.HTLC_HDR_FILE_DELETE:
            dispatchFileDelete(packet);
            break;
            
        case HLProtocol.HTLC_HDR_FILE_MOVE:
            dispatchFileMove(packet);
            break;

        case HLProtocol.HTLC_HDR_DIR_CREATE:
            dispatchDirectoryCreate(packet);
            break;
            
        case HLProtocol.HTLC_HDR_FILE_GETINFO:
            dispatchFileInfoGet(packet);
            break;
            
        case HLProtocol.HTLC_HDR_FILE_SETINFO:
            dispatchFileInfoSet(packet);
            break;

        case HLProtocol.HTLC_HDR_PRIVCHAT_CREATE:
            dispatchPrivateChatCreate(packet);
            break;
            
        case HLProtocol.HTLC_HDR_PRIVCHAT_JOIN:
            dispatchPrivateChatJoin(packet);
            break;
            
        case HLProtocol.HTLC_HDR_PRIVCHAT_LEAVE:
            dispatchPrivateChatLeave(packet);
            break;
            
        case HLProtocol.HTLC_HDR_PRIVCHAT_DECLINE:
            dispatchPrivateChatDecline(packet);
            break;
            
        case HLProtocol.HTLC_HDR_PRIVCHAT_INVITE:
            dispatchPrivateChatInvite(packet);
            break;
            
        case HLProtocol.HTLC_HDR_PRIVCHAT_SUBJECT:
            dispatchPrivateChatSubject(packet);
            break;
            
        case HLProtocol.HTLC_HDR_USER_GETINFO:
            dispatchUserInfoGet(packet);
            break;

        default:
            
            /* For requests that we do not understand, return
               an error to the client. */
            
            protocolError("Unknown header type: 0x" + Integer.toHexString(packet.header.id));            
            sendTaskError(packet.header.trans, "This server does not support your request.");            
            break;
            
        }
            
        DebuggerOutput.debug("HLServerDispatcher.dispatch: dispatch finished.");
        
    }

    /**
     * Takes a login packet and processes it.
     */
    void dispatchLogin(HLProtocol.Packet packet) {

        /* Get the user credentials from the packet. */
        
        synchronized(userStateLock) {

            while(packet.hasMoreComponents()) {
                
                HLProtocol.DataComponent dh = packet.nextComponent();

                if(dh.data != null) {
                    
                    switch(dh.type) {
                        
                    case HLProtocol.HTLC_DATA_LOGIN:
                        if(dh.data.length > 0)                           
                            login = new String(HLProtocol.invert(dh.data));
                        break;
                        
                    case HLProtocol.HTLC_DATA_PASSWORD:
                        if(dh.data.length > 0 &&
                           !(dh.data.length == 1 && dh.data[0] == 0))
                            password = new String(HLProtocol.invert(dh.data));
                        break;
                        
                    case HLProtocol.HTLC_DATA_NICK:
                        nick = new String(dh.data);
                        break;
                        
                    case HLProtocol.HTLC_DATA_ICON:
                        icon = (short) ToArrayConverters.byteArrayToInt(dh.data);
                        break;
                        
                    }
                    
                }
            
            }

            /* Hotline 1.5 doesn't send the nickname and icon until
               the agreement has been honoured. */
            
            String loginApproved = hls.hsp.approveLogin(this);
            
            if(loginApproved != null) {
                
                /* Login could not complete. */
                
                sendTaskError(packet.header.trans, loginApproved);
                hls.log(client.getInetAddress(), loginApproved);
                connected = false;
                
            } else {

                /* Authentication was successful. */
            
                hls.log(client.getInetAddress(), "Authentication successful for " + login);

                try {
                    
                    sock = hls.allocateSocketID();
                    
                    /* Send response with server version here. The
                       1.2.1 server that we're emulating actually does
                       not send a server version. */
                    
                    HLProtocol.DataComponent[] dataComponents = 
                        new HLProtocol.DataComponent[] { 
                            
                            hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_VERSION, ToArrayConverters.intToByteArray(121)),
                            
                        };
                    
                    sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));

                    /* Send the agreement (if there is an agreement
                       and the user can't skip it). */

                    if((privileges & HLProtocol.AccountInfo.DONT_SHOW_AGREEMENT) != HLProtocol.AccountInfo.DONT_SHOW_AGREEMENT) {

                        String agreement = hls.getAgreement().get();
                        
                        if(agreement != null) {
                            
                            dataComponents = 
                                new HLProtocol.DataComponent[] { 
                                    
                                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_AGREEMENT, agreement.getBytes()),
                                    
                                };
                            
                            send(hls.hlp.new Packet(HLProtocol.HTLS_HDR_AGREEMENT, packet.header.trans, dataComponents));
                            
                        }

                    }
                    
                    setName(toString());
                
                } catch(OutOfSocketIDsException e) {
                    
                    hls.log(client.getInetAddress(), "Connection refused because the server is out of socket ID's (too many connected users).");
                    connected = false;
                    
                }

            }
                
        }

        if(connected) {

            /* Broadcast a "user change" event to all connected
               clients to let them know there's a new kid in town. All
               the way down here because deadlock occurs when you hold
               the userStateLock on someone when you call
               changeUser(). */
            
            hls.changeUserExcept(this, sock, nick, icon, color);
            
            /* Finally, let the server listener know that a user
               has joined. */
            
            hls.hsl.userJoined(this);
            
        }
        
    }

    /**
     * Takes a chat packet and processes it.
     */
    void dispatchChat(HLProtocol.Packet packet) {
        byte[] msg = null;
        short action = 0;
        Object refObject = new String("Public");

        while(packet.hasMoreComponents()) {

            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_CHAT:
                    msg = dh.data;
                    break;
                    
                case HLProtocol.HTLC_DATA_OPTION:
                    action = ToArrayConverters.byteArrayToShort(dh.data);
                    break;

                case HLProtocol.HTLC_DATA_CHAT_REF:
                    refObject = new Integer(ToArrayConverters.byteArrayToInt(dh.data));
                    break;
                    
                }
                
            }
            
        }
        
        if(msg == null) {

            protocolError("Didn't receive content for public chat event.");
            return;

        }

        String chatAllowed = hls.hsp.approveChat(this, new String(msg), action == 0 ? false : true);

        if(chatAllowed != null) {

            /* FIXME: This is not a task error! This should be an
               administrator message. */

            sendTaskError(packet.header.trans, chatAllowed);
            return;
            
        }
        
        /* Format chat message. */
        
        HLProtocol.DataComponent[] dataComponents;

        if(refObject.toString().equals("Public")) {

            dataComponents = new HLProtocol.DataComponent[1];
         
        } else {

            dataComponents = new HLProtocol.DataComponent[2];
            dataComponents[1] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(((Integer) refObject).intValue()));
            
        }
        
        dataComponents[0] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT, hls.hsp.formatChat(this, new String(msg), (action == 1 ? true : false)).getBytes());

        /* Send chat message. */

        try {

            hls.getRealmTable().broadcastPacket(refObject, hls.hlp.new Packet(HLProtocol.HTLS_HDR_CHAT, nextTrans(), dataComponents));

        } catch(InvalidRealmException e) {}
              
    }

    /**
     * Takes a user change packet and processes it.
     */
    void dispatchUserChange(HLProtocol.Packet packet) {

        byte[] n = null;
        short icon = -1;

        while(packet.hasMoreComponents()) {

            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {

                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_NICK:
                    n = dh.data;
                    break;
                
                case HLProtocol.HTLC_DATA_ICON:
                    icon = ToArrayConverters.byteArrayToShort(dh.data);
                    break;

                }

            }
                
        }
        
        synchronized(userStateLock) {

            String changeAllowed = hls.hsp.approveUserChange(this, hls.hlp.new UserListComponent(sock, icon, color, new String(n)));
            
            if(changeAllowed != null) {

                hls.log(client.getInetAddress(), "User change not allowed: " + changeAllowed);
                HLProtocol.DataComponent[] dataComponents = 
                    new HLProtocol.DataComponent[1];
                dataComponents[0] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_MSG, changeAllowed.getBytes());
                send(hls.hlp.new Packet(HLProtocol.HTLS_HDR_MSG, trans++, dataComponents));
                return;
                
            }

            if(n != null) {
                
                /* Check whether this user is allowed to use any name
                   before changing his nick. */
                
                if((privileges & 
                    HLProtocol.AccountInfo.CAN_USE_ANY_NAME) == 
                   HLProtocol.AccountInfo.CAN_USE_ANY_NAME)
                    this.nick = new String(n);
                
            }
            
            this.icon = (icon == -1 ? this.icon : icon);
            
        }
            
        /* Now construct broadcast the user change event to all
           connected users. */
        
        hls.changeUser(this, sock, nick, icon, color);
        
        /* Finally let the listener know about the event. */
        
        hls.hsl.userChanged(this);
       
    }

    /**
     * Takes a user list request and processes it.
     */
    void dispatchUserList(HLProtocol.Packet packet) {

        HLServerDispatcher[] clients = hls.getClients();

        int numberOfConnectedClients = 0;

        HLProtocol.UserListComponent[] ul = 
            new HLProtocol.UserListComponent[clients.length];
        
        for(int i = 0; i < clients.length; i++) {

            ul[i] = clients[i].getUser();

            if(ul[i].sock != -1)
                numberOfConnectedClients++;

        }

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[numberOfConnectedClients];    

        int j = 0;

        for(int i = 0; i < clients.length; i++)            
            if(ul[i].sock != -1)
                dataComponents[j++] = ul[i];

        sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));

    }


    /**
     * Takes a flat news get request and processes it.
     */
    void dispatchFlatnewsGet(HLProtocol.Packet packet) {

        /* Check with policy manager whether getting news is
           allowed. */

        String flatnewsGetAllowed = hls.hsp.approveFlatnewsGet(this);

        if(flatnewsGetAllowed == null) {

            HLProtocol.DataComponent[] dataComponents = 
                new HLProtocol.DataComponent[] { 
                    
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_NEWS, hls.getFlatnews().get().getBytes()),
                    
                };

            sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));

        } else {

            sendTaskError(packet.header.trans, flatnewsGetAllowed);

        }

    }


    /**
     * Takes a flat news post request and processes it.
     */
    void dispatchFlatnewsPost(HLProtocol.Packet packet) {

        String newPost = null;

        while(packet.hasMoreComponents()) {

            HLProtocol.DataComponent dh = packet.nextComponent();

            switch(dh.type) {
                
            case HLProtocol.HTLC_DATA_NEWS_POST:
                newPost = new String(dh.data);
                break;

            }

        }

        if(newPost == null) {
            
            protocolError("Didn't receive content for flat news post.");
            return;

        }

        String flatnewsPostAllowed = hls.hsp.approveFlatnewsPost(this, newPost);

        if(flatnewsPostAllowed == null) {
            
            sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
            hls.postFlatnews(hls.hsp.formatFlatnewsPost(this, newPost));
            hls.log(client.getInetAddress(), "Accepted flat news post.");
            
        } else {
            
            hls.log(client.getInetAddress(), "Rejected flat news post (" + flatnewsPostAllowed + ")");
            sendTaskError(packet.header.trans, flatnewsPostAllowed);
            
        } 
        
    }

    /**
     * Takes an account read packet and processes it.
     */
    void dispatchAccountRead(HLProtocol.Packet packet) {
        
        String l = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_LOGIN:
                    l = new String(dh.data);
                    break;
                    
                }
                
            }
            
        }
        
        if(l == null) {
            
            protocolError("Did not receive required components for account read.");
            return;

        }

        try {
            
            HLServerAccountsTable accountsTable = hls.getAccountsTable();
            
            accountsTable.lock();
            
            if(accountsTable.exists(l)) {
                
                String accountReadAllowed = hls.hsp.approveAccountRead(this, l);
                
                if(accountReadAllowed == null) {
                    
                    HLProtocol.AccountInfo account = (HLProtocol.AccountInfo) accountsTable.get(l);
                    
                    byte[] pw = new byte[1];
                    pw[0] = 0x07;  /* ASCII BEL */
                    HLProtocol.DataComponent[] dataComponents = 
                        new HLProtocol.DataComponent[] {
                            
                            hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_NICK, account.nick.getBytes()),
                            hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_LOGIN, HLProtocol.invert(account.login.getBytes())),
                                /* Don't send the password. */
                            hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_PASSWORD, pw),
                            hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_PRIVILEGES, ToArrayConverters.swapByteArray(ToArrayConverters.longToByteArray(account.privileges))),
                            
                        };
                    
                    sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));
                    
                        
                } else {
                    
                    sendTaskError(packet.header.trans, accountReadAllowed);
                    hls.log(client.getInetAddress(), "Account read disapproved (" + accountReadAllowed + ")");
                    
                }
         
            } else {
                
                sendTaskError(packet.header.trans, "Cannot read account '" + l + "' because that account does not exist.");
                hls.log(client.getInetAddress(), "Could not read account '" + l + "' because it does not exist.");
                
            }

            accountsTable.unlock();
            
        } catch(InterruptedException _e) {
            
            hls.log(client.getInetAddress(), "[internal error] Disconnecting client because of a failure to acquire the account lock.");
            connected = false;
            
        }
        
    }
    
    /**
     * Takes an account create packet and processes it.
     */
    void dispatchAccountCreate(HLProtocol.Packet packet) {
        
        String l = null, p = null, n = null;
        long pr = 0;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_LOGIN:
                    l = new String(HLProtocol.invert(dh.data));
                    break;
                    
                case HLProtocol.HTLC_DATA_PASSWORD:
                    p = new String(HLProtocol.invert(dh.data));
                    break;
                    
                case HLProtocol.HTLC_DATA_NICK:
                    n = new String(dh.data);
                    break;
                    
                case HLProtocol.HTLS_DATA_PRIVILEGES:
                    pr = ToArrayConverters.byteArrayToLong(ToArrayConverters.swapByteArray(dh.data));
                    break;
                    
                }
                
            }
            
        }

        if(p == null)
            p = "guest";

        if(n == null)
            n = l;

        if(l == null) {
            
            protocolError("Did not receive required components for account creation.");            
            return;
            
        }

        try {
            
            HLServerAccountsTable accountsTable = hls.getAccountsTable();
            
            accountsTable.lock();
            
            if(!accountsTable.exists(l)) {
                
                HLProtocol.AccountInfo newAccount = hls.hlp.new AccountInfo(l, n, UnixCrypt.crypt("aa", p), pr, null);
                
                String accountCreateAllowed = hls.hsp.approveAccountCreate(this, newAccount);
                
                if(accountCreateAllowed == null) {
                    
                    accountsTable.put(l, newAccount);
                    sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                    
                } else {
                    
                    sendTaskError(packet.header.trans, accountCreateAllowed);
                    
                }
                
            } else {
                
                hls.log(client.getInetAddress(), "Could not create account '" + l + "' because it already exists.");
                
                if((privileges & HLProtocol.AccountInfo.CAN_READ_ACCOUNTS) == HLProtocol.AccountInfo.CAN_READ_ACCOUNTS) {
                    
                    /* If this user has permissions to read user
                       accounts, then we let him know that the
                       account could not be created because the
                       account already exists. Otherwise we don't
                       give any explicit reason as to why the
                       account could not be created. */
                    
                    sendTaskError(packet.header.trans, "Cannot create account '" + l + "' because that account already exists.");
                        
                } else {
                    
                    sendTaskError(packet.header.trans, "Cannot create account.");
                    
                }
                
            }
            
            accountsTable.unlock();
            
        } catch(InterruptedException _e) {
            
            hls.log(client.getInetAddress(), "[internal error] Disconnecting client because of a failure to acquire the account lock.");
            connected = false;
            
        }
                    
    }

    /**
     * Takes a user change packet and processes it.
     */
    void dispatchAccountModify(HLProtocol.Packet packet) {
        
        byte[] p = null;
        String l = null, n = null;
        long pr = 0;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_LOGIN:
                    l = new String(HLProtocol.invert(dh.data));
                    break;
                    
                case HLProtocol.HTLC_DATA_PASSWORD:
                    p = dh.data;
                    break;
                    
                case HLProtocol.HTLC_DATA_NICK:
                    n = new String(dh.data);
                    break;
                    
                case HLProtocol.HTLS_DATA_PRIVILEGES:
                    pr = ToArrayConverters.byteArrayToLong(ToArrayConverters.swapByteArray(dh.data));
                    break;
                    
                }
                
            }
            
        }

        if(p == null)
            p = new String("guest").getBytes();

        /* Check that all the required parameters are there. Also,
           verify that no account can be modified to have privileges
           that the modifier does not possess. */
        
        if(l == null || n == null ) {
            
            protocolError("Did not receive required components for account modification.");
            return;

        }

        try {
            
            HLServerAccountsTable accountsTable = hls.getAccountsTable();
            
            accountsTable.lock();
            
            if(accountsTable.exists(l)) {
                
                HLProtocol.AccountInfo oldAccount = (HLProtocol.AccountInfo) accountsTable.get(l);
                
                /* If password is ASCII NUL, then password
                   don't change. */
                
                if(p.length == 1 && p[0] == 0)
                    p = oldAccount.password.getBytes();
                else
                    p = UnixCrypt.crypt("aa", new String(HLProtocol.invert(p))).getBytes();
                
                HLProtocol.AccountInfo newAccount = hls.hlp.new AccountInfo(l, n, new String(p), pr, null);
                
                String accountModifyAllowed = hls.hsp.approveAccountModify(this, oldAccount, newAccount);
                
                if(accountModifyAllowed == null) {
                    
                    accountsTable.put(l, newAccount);
                    sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                    
                } else {
                    
                    sendTaskError(packet.header.trans, accountModifyAllowed);
                    
                }
                
            } else {
                
                hls.log(client.getInetAddress(), "Could not modify account '" + l + "' because it does not exist.");
                
                if((privileges & HLProtocol.AccountInfo.CAN_READ_ACCOUNTS) == HLProtocol.AccountInfo.CAN_READ_ACCOUNTS) {
                    
                    /* If this user has permissions to read user
                       accounts, then we let him know that the
                       account could not be created because the
                       account already exists. Otherwise we don't
                       give any explicit reason as to why the
                       account could not be created. */
                    
                    sendTaskError(packet.header.trans, "Cannot modify account '" + l + "' because that account does not exist.");
                    
                } else {
                    
                    sendTaskError(packet.header.trans, "Cannot modify account.");
                    
                }
                
            }
            
            accountsTable.unlock();
            
        } catch(InterruptedException _e) {
            
            hls.log(client.getInetAddress(), "[internal error] Disconnecting client because of a failure to acquire the account lock.");
            connected = false;
            
        }
        
    }

    /**
     * Takes a user change packet and processes it.
     */
    void dispatchAccountDelete(HLProtocol.Packet packet) {

        String l = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_LOGIN:
                    l = new String(HLProtocol.invert(dh.data));
                    break;
                    
                }
                
            }
            
        }
        
        if(l == null) {

            protocolError("Did not receive required components for account deletion.");
            return;

        }
            
        try {
            
            HLServerAccountsTable accountsTable = hls.getAccountsTable();
            
            accountsTable.lock();
            
            if(accountsTable.exists(l)) {
                
                String accountDeleteAllowed = hls.hsp.approveAccountDelete(this, accountsTable.get(l));
                
                if(accountDeleteAllowed == null) {
                    
                        accountsTable.remove(l);
                        sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                        
                } else {
                    
                    sendTaskError(packet.header.trans, accountDeleteAllowed);
                    
                }
                
            } else {
                
                hls.log(client.getInetAddress(), "Could not delete account '" + l + "' because it does not exist.");
                
                if((privileges & HLProtocol.AccountInfo.CAN_READ_ACCOUNTS) == HLProtocol.AccountInfo.CAN_READ_ACCOUNTS) {
                    
                    /* If this user has permissions to read user
                       accounts, then we let him know that the
                       account could not be created because the
                       account already exists. Otherwise we don't
                       give any explicit reason as to why the
                       account could not be created. */
                    
                    sendTaskError(packet.header.trans, "Cannot delete account '" + l + "' because that account does not exist.");
                    
                } else {
                    
                    sendTaskError(packet.header.trans, "Cannot delete account.");
                    
                }
                
            }
            
            accountsTable.unlock();
            
        } catch(InterruptedException _e) {
            
            hls.log(client.getInetAddress(), "[internal error] Disconnecting client because of a failure to acquire the account lock.");
            connected = false;
            
        }
        
    }

    /**
     * Takes a message packet and processes it.
     */
    void dispatchMessage(HLProtocol.Packet packet) {

        String msg = null;
        int sock = -1;

        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_SOCKET:
                    sock = ToArrayConverters.byteArrayToInt(dh.data);
                    break;

                case HLProtocol.HTLC_DATA_MSG:
                    msg = new String(dh.data);
                    break;
                    
                }
                
            }
            
        }

        if(sock == -1 || msg == null) {

            protocolError("Did not receive required components for message transmit.");
            return;

        }

        String messageAllowed = hls.hsp.approveMessage(this, sock, msg);
        
        if(messageAllowed == null) {
            
            if(hls.sendPrivateMessage(this, sock, msg)) {
                
                sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                
            } else {
                
                sendTaskError(packet.header.trans, "Could not send message because the user is no longer connected to the server.");
                
            }
            
        } else {
            
            sendTaskError(packet.header.trans, messageAllowed);
            
        }
        
    }

    /**
     * Takes a user kick packet and processes it.
     */
    void dispatchUserKick(HLProtocol.Packet packet) {
        int sock = -1;
        short ban = 0;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_SOCKET:
                    sock = ToArrayConverters.byteArrayToInt(dh.data);
                    break;
                    
                case HLProtocol.HTLC_DATA_BAN:
                    ban = ToArrayConverters.byteArrayToShort(dh.data);
                    break;
                    
                }
                
            }
            
        }
        
        if(sock != -1) {
            
            String userKickAllowed = hls.hsp.approveUserKick(this, sock, ban == 1 ? true : false);
            
            if(userKickAllowed == null) {
                
                if(hls.kickUser(sock, ban == 1 ? true : false)) {
                    
                    HLServerDispatcher[] clients = hls.getClients();
                    
                    for(int i = 0; i < clients.length; i++) {
                        
                        if(clients[i].getUser().sock == sock) {
                            
                            hls.log(client.getInetAddress(), 
                                    new String(getUser().nick) + 
                                    " has kicked " + 
                                    (ban == 1 ? "(and banned)" : "") +
                                    new String(clients[i].getUser().nick));
                            break;
                            
                        }
                        
                    }
                    
                    sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                    
                } else {
                    
                    sendTaskError(packet.header.trans, "Could not disconnect user.");
                    
                }
             
            } else {

                sendTaskError(packet.header.trans, userKickAllowed);
                
            }   

        } else {
            
            protocolError("Did not receive required components for user kick.");
            
        }
        
    }

    /**
     * Takes a directory listing packet and processes it.
     */
    void dispatchDirectoryListing(HLProtocol.Packet packet) {

        String path = ":";

        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_DIR:
                    path = ((HLProtocol.PathComponent) dh).path;
                    break;
                    
                }
                
            }
            
        }
        
        String fileListAllowed = hls.hsp.approveDirectoryListing(this, path);
        
        if(fileListAllowed == null) {
            
            try {
                
                HLProtocol.DataComponent[] dataComponents = 
                    hls.hsp.obtainDirectoryListing(this, path);
                
                sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));
            
            } catch(IOException e) {

                DebuggerOutput.stackTrace(e);

                sendTaskError(packet.header.trans, e.toString());

            }

        } else {
            
            sendTaskError(packet.header.trans, fileListAllowed);
            
        }
                
    }

    /**
     * Takes a file download packet and processes it.
     */
    void dispatchFileDownload(HLProtocol.Packet packet) {

        String path = "", file = null;
        int forkOrFile = HLProtocol.DL_WHOLE_FILE;
        
        HLProtocol.ResumeTransferComponent rflt = hls.hlp.new ResumeTransferComponent();
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_RESUME:
                    forkOrFile = ToArrayConverters.byteArrayToInt(dh.data);
                    break;

                case HLProtocol.HTLC_DATA_RFLT:
                    rflt = (HLProtocol.ResumeTransferComponent) dh;
                    break;

                case HLProtocol.HTLC_DATA_DIR:
                    path = ((HLProtocol.PathComponent) dh).path;
                    break;
                    
                case HLProtocol.HTLC_DATA_FILE:
                    file = ((HLProtocol.FileComponent) dh).file;
                    break;

                }
                
            }
            
        }
        
        if(file == null) {
            
            protocolError("Did not receive required components for file download.");
            return;
            
        }

        String fileDownloadAllowed = hls.hsp.approveFileDownload(this, path, HLProtocol.DIR_SEPARATOR + file);

        if(fileDownloadAllowed != null) {
            
            sendTaskError(packet.header.trans, fileDownloadAllowed);
            return;

        }

        try {
            
            MacFile hostPath = hls.hsp.resolvePathOrPartial(this, path + HLProtocol.DIR_SEPARATOR + file);
            
            HLServer.TransferRequest request = hls.getTransferQueue().createDownload(this, hostPath, rflt, forkOrFile);
            
            /* Create and serialize this object because we need to
               know it's length in advance -- apparently Hotline
               expects the length of download reply in it's entirety
               (excluding the HTXF block).  Changes here mean changes
               in HLTransferServer.java. */

            HLProtocol.FileTransferInfo fti = 
                hls.hlp.new FileTransferInfo
                    (hostPath.getFile().getName(),
                     hostPath.getType(),
                     hostPath.getCreator(),
                     hostPath.getComment(),
                     hostPath.getCreationDate(),
                     hostPath.getModificationDate(),
                     hostPath.getFinderFlags());
            
            HLProtocol.DataComponent[] dataComponents = new HLProtocol.DataComponent[request.queuePosition == 0 ? 2 : 3];

            dataComponents[0] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_HTXF_REF, ToArrayConverters.intToByteArray(request.ref));
            
            long dataSize = ((Long) rflt.chunks.get("DATA")).longValue();
            long rsrcSize = ((Long) rflt.chunks.get("MACR")).longValue();
            long dataRemaining = request.local.getDataFork().size() - 
                dataSize;
            long rsrcRemaining = request.local.getResourceFork().size() - 
                rsrcSize;
                    
            if(dataRemaining < 0 || rsrcRemaining < 0) {
                
                String error = "Attempt to resume beyond end of file (dataRemain = " + dataRemaining + ", rsrcRemain = " + rsrcRemaining + ")";
                
                DebuggerOutput.debug("HLServerDispatcher.dispatchFileDownload: " + error);
                sendTaskError(packet.header.trans, error);
                hls.getTransferQueue().destroy(request.ref);
                return;
                
            }

            /* If the client only wants the data fork, send the size
               of the data fork. */

            dataComponents[1] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_HTXF_SIZE, ToArrayConverters.intToByteArray((int) (forkOrFile == HLProtocol.DL_DATA_FORK ? hostPath.getDataFork().size() : dataRemaining + rsrcRemaining + fti.data.length)));

            /* Also send the queue position if applicable. */

            if(request.queuePosition > 0)
                dataComponents[2] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_HTXF_QUEUEPOS, ToArrayConverters.intToByteArray(request.queuePosition));
                        
            sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));
            
        } catch(IOException e) {
            
            DebuggerOutput.stackTrace(e);

            sendTaskError(packet.header.trans, e.toString());
            
        } catch(TooManyTransfersException e) {

            sendTaskError(packet.header.trans, e.getMessage());

        }
        
    }

    /**
     * Takes a file upload packet and processes it.
     */
    void dispatchFileUpload(HLProtocol.Packet packet) {

        String path = "", file = null;
        boolean resume = false;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_RESUME:
                    resume = (ToArrayConverters.byteArrayToShort(dh.data) == 1 ? true : false);
                    break;

                case HLProtocol.HTLC_DATA_DIR:
                    path = ((HLProtocol.PathComponent) dh).path;
                    break;
                    
                case HLProtocol.HTLC_DATA_FILE:
                    file = ((HLProtocol.FileComponent) dh).file;
                    break;

                }
                
            }
            
        }
        
        if(file == null) {
            
            protocolError("Did not receive required components for file upload.");
            return;

        }

        String fileUploadAllowed = hls.hsp.approveFileUpload(this, path, HLProtocol.DIR_SEPARATOR + file, resume);

        if(fileUploadAllowed != null) {
                
            sendTaskError(packet.header.trans, fileUploadAllowed);
            return;

        }

        try {

            MacFile hostPath = hls.hsp.resolvePath(this, path + HLProtocol.DIR_SEPARATOR + file + ".hpf", true);
                    
            HLServer.TransferRequest request = hls.getTransferQueue().createUpload(this, hostPath);

            HLProtocol.DataComponent[] dataComponents = 
                new HLProtocol.DataComponent[resume ? 2 : 1];

            dataComponents[0] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_HTXF_REF, ToArrayConverters.intToByteArray(request.ref));

            if(resume)
                dataComponents[1] = hls.hlp.new ResumeTransferComponent(hostPath.getDataFork().size(), hostPath.getResourceFork().size());

            sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));

        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

            sendTaskError(packet.header.trans, e.toString());

        } catch(TooManyTransfersException e) {

            sendTaskError(packet.header.trans, e.getMessage());

        }
        
    }

    /**
     * Takes a file delete packet and processes it.
     */
    void dispatchFileDelete(HLProtocol.Packet packet) {
        
        String path = "", file = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_DIR:
                    path = ((HLProtocol.PathComponent) dh).path;
                    break;

                case HLProtocol.HTLC_DATA_FILE:
                    file = ((HLProtocol.FileComponent) dh).file;
                    break;

                }
                
            }
            
        }
        
        if(file == null) {
            
            protocolError("Did not receive required components for file delete.");
            return;

        }

        String fileDeleteAllowed = hls.hsp.approveFileDelete(this, path, file);

        if(fileDeleteAllowed != null) {
                
            sendTaskError(packet.header.trans, fileDeleteAllowed);
            return;

        }

        MacFile hostPath = null;

        try {

            hostPath = hls.hsp.resolvePathOrPartial(this, path + HLProtocol.DIR_SEPARATOR + file);
                    
            if(hostPath.renameTo(new File(hls.getTrashDirectory(), file)))
                sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                    
            else
                sendTaskError(packet.header.trans, "Could not delete the file or directory. Perhaps the trash is full or there is a file or directory in the way.");
                    
        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

            sendTaskError(packet.header.trans, e.toString());

        } finally {

            if(hostPath != null) {

                try {

                    hostPath.close();

                } catch(IOException e) {

                    DebuggerOutput.stackTrace(e);

                }

            }

        }
        
    }

    /**
     * Takes a file move packet and processes it.
     */
    void dispatchFileMove(HLProtocol.Packet packet) {
        
        String source = null, target = null, file = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_FILE:
                    file = new String(dh.data);
                    break;

                case HLProtocol.HTLC_DATA_DIR:
                    source = ((HLProtocol.PathComponent) dh).path;
                    break;

                case HLProtocol.HTLC_DATA_DIR_RENAME:
                    target = ((HLProtocol.PathComponent) dh).path;
                    break;

                }
                
            }
            
        }
        
        if(source == null) 
            source = "";

        if(file == null || target == null) {
            
            protocolError("Did not receive required components for file move.");
            return;

        }

        String fileMoveAllowed = hls.hsp.approveFileMove(this, source + HLProtocol.DIR_SEPARATOR + file, target);

        if(fileMoveAllowed != null) {
                
            sendTaskError(packet.header.trans, fileMoveAllowed);
            return;

        }

        MacFile sourceFile = null, targetFile = null;

        try {

            sourceFile = hls.hsp.resolvePathOrPartial(this, source + HLProtocol.DIR_SEPARATOR + file);
            targetFile = hls.hsp.resolvePathOrPartial(this, target);
            
            DebuggerOutput.debug("HLServerDispatcher.dispatchFileMove: sourceFile = " + sourceFile);
            DebuggerOutput.debug("HLServerDispatcher.dispatchFileMove: targetFile = " + targetFile);

            if(sourceFile.renameTo(new File(targetFile.getFile(), file)))
                sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                    
            else
                sendTaskError(packet.header.trans, "Could not move the file or directory. Perhaps there is a file or directory in the way.");

        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

            sendTaskError(packet.header.trans, e.toString());

        } finally {

            if(sourceFile != null) {

                try {

                    sourceFile.close();

                } catch(IOException e) {

                    DebuggerOutput.stackTrace(e);

                }

            }


            if(targetFile != null) {

                try {

                    targetFile.close();

                } catch(IOException e) {

                    DebuggerOutput.stackTrace(e);

                }

            }

        }
        
    }

    /**
     * Takes a directory create packet and processes it.
     */
    void dispatchDirectoryCreate(HLProtocol.Packet packet) {
        
        String path = "", file = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_DIR:
                    path = ((HLProtocol.PathComponent) dh).path;
                    break;

                case HLProtocol.HTLC_DATA_FILE:
                    file = ((HLProtocol.FileComponent) dh).file;
                    break;

                }
                
            }
            
        }
        
        if(file == null) {

            protocolError("Did not receive required components for directory create.");
            return;

        }
            
        String dirCreateAllowed = hls.hsp.approveDirectoryCreate(this, path, file);

        if(dirCreateAllowed != null) {

            sendTaskError(packet.header.trans, dirCreateAllowed);
            return;

        }
                
        MacFile hostPath = null;

        try {

            hostPath = hls.hsp.resolvePath(this, path + HLProtocol.DIR_SEPARATOR + file, true);
            File newPath = hostPath.getFile();

            if(newPath.mkdir())
                sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));
                    
            else
                sendTaskError(packet.header.trans, "Could not create directory.");
                    
        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

            sendTaskError(packet.header.trans, e.toString());

        } finally {

            if(hostPath != null) {

                try {

                    hostPath.close();

                } catch(IOException e) {

                    if(DebuggerOutput.on)
                        e.printStackTrace();

                }

            }

        }
        
    }

    /**
     * Takes a get file info packet and processes it.
     */
    void dispatchFileInfoGet(HLProtocol.Packet packet) {
        
        String path = "", file = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_DIR:
                    path = ((HLProtocol.PathComponent) dh).path;
                    break;

                case HLProtocol.HTLC_DATA_FILE:
                    file = ((HLProtocol.FileComponent) dh).file;
                    break;

                }
                
            }
            
        }
        
        if(file == null) {
            
            protocolError("Did not receive required components for file get info.");
            return;

        }

        String fileGetInfoAllowed = hls.hsp.approveFileGetInfo(this, path, file);

        if(fileGetInfoAllowed != null) {
                
            sendTaskError(packet.header.trans, fileGetInfoAllowed);
            return;

        }

        try {

            HLProtocol.FileInfo fileInfo = hls.hsp.obtainFileInfo(this, path, file);
                    
            HLProtocol.DataComponent[] dataComponents = 
                new HLProtocol.DataComponent[] { 
                            
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_FILE_NAME, fileInfo.name.getBytes()),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_FILE_COMMENT, fileInfo.comment.getBytes()),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_FILE_SIZE, ToArrayConverters.intToByteArray((int) fileInfo.size)),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_FILE_TYPE, fileInfo.type.getBytes()),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_FILE_CREATOR, fileInfo.creator.getBytes()),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_FILE_ICON, fileInfo.icon.getBytes()),
                    hls.hlp.new DateComponent(HLProtocol.HTLS_DATA_FILE_DATE_CREATED, fileInfo.created),
                    hls.hlp.new DateComponent(HLProtocol.HTLS_DATA_FILE_DATE_MODIFIED, fileInfo.modified),
                            
                };
                    
            sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));
                    
        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

            sendTaskError(packet.header.trans, e.toString());

        }
        
    }

    /**
     * Takes a set file info packet and processes it.
     */
    void dispatchFileInfoSet(HLProtocol.Packet packet) {
        
        String path = "", file = null, newName = null, newComment = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_DIR:
                    path = ((HLProtocol.PathComponent) dh).path;
                    break;

                case HLProtocol.HTLC_DATA_FILE:
                    file = ((HLProtocol.FileComponent) dh).file;
                    break;

                case HLProtocol.HTLC_DATA_FILE_RENAME:
                    newName = new String(dh.data);
                    break;

                case HLProtocol.HTLS_DATA_FILE_COMMENT:
                    newComment = new String(dh.data);
                    break;

                }
                
            }
            
        }
        
        if(file == null || (newName == null && newComment == null)) {
            
            protocolError("Did not receive required components for file set info.");
            return;

        }

        String fileSetInfoAllowed = hls.hsp.approveFileSetInfo(this, path, file, newName, newComment);

        if(fileSetInfoAllowed != null) {
                
            sendTaskError(packet.header.trans, fileSetInfoAllowed);
            return;

        }

        MacFile hostPath = null;

        try {

            hostPath = hls.hsp.resolvePathOrPartial(this, path + HLProtocol.DIR_SEPARATOR + file);

            if(newName != null) {

                /* Request to rename. */

                if(!hostPath.renameTo(new File(hostPath.getFile().getParent(),  newName)))
                    throw new IOException("Could not rename " + file + ". Perhaps a problem with symbolic links.");

            }

            if(newComment != null) {

                /* Request to change comment. */

                hostPath.writeHeader(hostPath.getFile().getName(), 
                                     hostPath.getType(),
                                     hostPath.getCreator(),
                                     newComment,
                                     hostPath.getCreationDate(),
                                     hostPath.getModificationDate(),
                                     hostPath.getFinderFlags());

            }
            
            sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, null));

        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

            sendTaskError(packet.header.trans, e.toString());

        } finally {

            if(hostPath != null) {

                try {

                    hostPath.close();

                } catch(IOException e) {

                    if(DebuggerOutput.on)
                        e.printStackTrace();

                }

            }

        }

    }

    /**
     * Takes a create private chat packet and processes it.
     */
    void dispatchPrivateChatCreate(HLProtocol.Packet packet) {
        
        int sock = -1;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_SOCKET:
                    sock = ToArrayConverters.byteArrayToInt(dh.data);
                    break;

                }
                
            }
            
        }
        
        if(sock == -1) {
            
            protocolError("Did not receive required components for private chat create.");
            return;

        }

        HLServerDispatcher client = hls.resolveSocketID(sock);

        if(client == null) {

            sendTaskError(packet.header.trans, "Could not create private chat because your party is no longer connected to the server.");
            return;

        }

        String privChatCreateAllowed = hls.hsp.approvePrivateChatCreate(this, sock);
        
        if(privChatCreateAllowed != null) {
            
            sendTaskError(packet.header.trans, privChatCreateAllowed);
            return;

        }    
        
        Integer refObject = 
            (Integer) hls.getRealmTable().createRealm(this.sock);
        
        HLProtocol.DataComponent[] dataComponents = null;
        
        synchronized(userStateLock) {
            
            dataComponents = 
                new HLProtocol.DataComponent[] {
                    
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(refObject.intValue())),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_ICON, ToArrayConverters.intToByteArray(icon)),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_NICK, nick.getBytes()),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_COLOUR, ToArrayConverters.intToByteArray(color)),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_SOCKET, ToArrayConverters.intToByteArray(this.sock)),
                    
                };
            
        }
        
        sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));

        if(this.sock != sock) {

            /* Invite the other user. */
            
            dataComponents = 
                new HLProtocol.DataComponent[] {
                    
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(refObject.intValue())),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_SOCKET, ToArrayConverters.intToByteArray(this.sock)),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_NICK, nick.getBytes()),                
                };
            
            client.send(hls.hlp.new Packet(HLProtocol.HTLS_HDR_CHAT_INVITE, 0, dataComponents));
            
        }

    }

    /**
     * Takes a join private chat packet and processes it.
     */
    void dispatchPrivateChatJoin(HLProtocol.Packet packet) {
        
        Integer refObject = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_CHAT_REF:
                    refObject = new Integer(ToArrayConverters.byteArrayToInt(dh.data));
                    break;

                }
                
            }
            
        }
        
        if(refObject == null) {

            protocolError("Did not receive required components for private chat join.");
            return;

        }

        synchronized(userStateLock) {
            
            if(!hls.getRealmTable().addUser(refObject, sock)) {
                
                sendTaskError(packet.header.trans, "Cannot join that private chat because it has already been closed.");
                return;

            }

        }
        
        String privChatJoinAllowed = hls.hsp.approvePrivateChatJoin(this, refObject);
        
        if(privChatJoinAllowed != null) {

            sendTaskError(packet.header.trans, privChatJoinAllowed);
            return;

        }

        /* Grab the list of users in this realm and send it to the
           client that is joining. */

        try {

            HLProtocol.UserListComponent[] dataComponents = 
                hls.getRealmTable().getUsers(refObject);
            
            sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));

            /* Finally, broadcast a user change event to the realm. */

            hls.changeUser(this, sock, nick, icon, color, refObject, HLProtocol.HTLS_HDR_CHAT_USER_CHANGE, -1);
                
        } catch(InvalidRealmException e) {}
                
    }

    /**
     * Takes a leave private chat packet and processes it.
     */
    void dispatchPrivateChatLeave(HLProtocol.Packet packet) {
        
        Integer refObject = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_CHAT_REF:
                    refObject = new Integer(ToArrayConverters.byteArrayToInt(dh.data));
                    break;

                }
                
            }
            
        }
        
        if(refObject == null) {

            protocolError("Did not receive required components for private chat leave.");
            return;

        }
        
        synchronized(userStateLock) {

            if(!hls.getRealmTable().removeUser(refObject, sock)) {
                
                sendTaskError(packet.header.trans, "Cannot leave private chat that does not exist (anymore).");
                return;
                
            }

        }
        
    }

    /**
     * Takes a decline private chat packet and processes it.
     */
    void dispatchPrivateChatDecline(HLProtocol.Packet packet) {
        
        Integer refObject = null;
        
        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_CHAT_REF:
                    refObject = new Integer(ToArrayConverters.byteArrayToInt(dh.data));
                    break;

                }
                
            }
            
        }
        
        if(refObject == null) {

            protocolError("Did not receive required components for private chat decline.");
            return;

        }
        
        try {
            
            /* Send notification of decline in chat. */
            
            HLProtocol.DataComponent[] dataComponents;
            
            dataComponents = new HLProtocol.DataComponent[2];
            
            synchronized(userStateLock) {
                
                dataComponents[0] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT, new String("<<< " + nick + " declined invitation to chat >>>").getBytes());
                
            }
            
            dataComponents[1] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(((Integer) refObject).intValue()));
            
            hls.getRealmTable().broadcastPacket(refObject, hls.hlp.new Packet(HLProtocol.HTLS_HDR_CHAT, nextTrans(), dataComponents));
            
        } catch(InvalidRealmException e) {
            
            sendTaskError(packet.header.trans, "Cannot decline private chat that does not exist (anymore).");

        }

    }

    /**
     * Takes a private chat invitation packet and processes it.
     */
    void dispatchPrivateChatInvite(HLProtocol.Packet packet) {
        
        int sock = -1;
        Integer refObject = null;

        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_SOCKET:
                    sock = ToArrayConverters.byteArrayToInt(dh.data);
                    break;

                case HLProtocol.HTLC_DATA_CHAT_REF:
                    refObject = new Integer(ToArrayConverters.byteArrayToInt(dh.data));
                    break;

                }
                
            }
            
        }
        
        if(sock == -1 || refObject == null) {
            
            protocolError("Did not receive required components for private chat invite.");
            return;

        }

        HLServerDispatcher client = hls.resolveSocketID(sock);

        if(client == null) {

            sendTaskError(packet.header.trans, "Could not invite user because your party is no longer connected to the server.");
            return;

        }

        String privChatInviteAllowed = hls.hsp.approvePrivateChatInvite(this, refObject, sock);
        
        if(privChatInviteAllowed != null) {
            
            sendTaskError(packet.header.trans, privChatInviteAllowed);
            return;

        }    
        
        HLProtocol.DataComponent[] dataComponents = null;
        
        if(this.sock != sock) {

            /* Invite the user. */
            
            dataComponents = 
                new HLProtocol.DataComponent[] {
                    
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(refObject.intValue())),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_SOCKET, ToArrayConverters.intToByteArray(this.sock)),
                    hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_NICK, nick.getBytes()),                
                };
            
            client.send(hls.hlp.new Packet(HLProtocol.HTLS_HDR_CHAT_INVITE, 0, dataComponents));
            
        }

    }

    /**
     * Takes a private chat subject packet and processes it.
     */
    void dispatchPrivateChatSubject(HLProtocol.Packet packet) {
        
        String subject = null;
        Integer refObject = null;

        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();
            
            if(dh.data != null) {
                
                switch(dh.type) {

                case HLProtocol.HTLC_DATA_CHAT_SUBJECT:
                    subject = new String(dh.data);
                    break;

                case HLProtocol.HTLC_DATA_CHAT_REF:
                    refObject = new Integer(ToArrayConverters.byteArrayToInt(dh.data));
                    break;

                }
                
            }
            
        }
        
        if(subject == null || refObject == null) {
            
            protocolError("Did not receive required components for private chat subject.");
            return;

        }

        String privChatSubjectAllowed = hls.hsp.approvePrivateChatSubject(this, refObject, subject);
        
        if(privChatSubjectAllowed != null) {
            
            sendTaskError(packet.header.trans, privChatSubjectAllowed);
            return;

        }    
        
        try {
            
            /* Broadcast the new chat subject. */
            
            HLProtocol.DataComponent[] dataComponents;
            
            dataComponents = new HLProtocol.DataComponent[2];
            
            dataComponents[0] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_REF, ToArrayConverters.intToByteArray(((Integer) refObject).intValue()));
            dataComponents[1] = hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_CHAT_SUBJECT, subject.getBytes());
            
            hls.getRealmTable().broadcastPacket(refObject, hls.hlp.new Packet(HLProtocol.HTLS_HDR_CHAT_SUBJECT, nextTrans(), dataComponents));
            
        } catch(InvalidRealmException e) {
            
            sendTaskError(packet.header.trans, "Cannot set chat subject on that chat because it does not exist (anymore).");

        }
        
    }

    /**
     * Takes a message packet and processes it.
     */
    void dispatchUserInfoGet(HLProtocol.Packet packet) {

        int sock = -1;

        while(packet.hasMoreComponents()) {
            
            HLProtocol.DataComponent dh = packet.nextComponent();

            if(dh.data != null) {
                
                switch(dh.type) {
                    
                case HLProtocol.HTLC_DATA_SOCKET:
                    sock = ToArrayConverters.byteArrayToInt(dh.data);
                    break;

                }
                
            }
            
        }

        if(sock == -1) {

            protocolError("Did not receive required components for user get info.");
            return;

        }

        String userInfoGetAllowed = hls.hsp.approveUserGetInfo(this, sock);
        
        if(userInfoGetAllowed == null) {
            
            String userInfo = hls.getUserInfo(sock);
            
            if(userInfo != null) {
                
                HLProtocol.DataComponent[] dataComponents = 
                    new HLProtocol.DataComponent[] {
                        
                        hls.hlp.new DataComponent(HLProtocol.HTLS_DATA_USER_INFO, TextUtils.findAndReplace(userInfo, System.getProperty("line.separator"), "\r").getBytes()),
                        
                    };

                sendTaskReply(hls.hlp.new Packet(HLProtocol.HTLS_HDR_TASK, packet.header.trans, dataComponents));
                            
            } else {
                
                sendTaskError(packet.header.trans, "Could not get info because the user is no longer connected to the server.");
                
            }
            
        } else {
            
            sendTaskError(packet.header.trans, userInfoGetAllowed);
            
        }
        
    }

    /**
     * Increases the protocol error count, logs the problem, and
     * disconnects the client if the number of protocol errors exceeds
     * the maximum allowed amount.
     * @param err the error message.  
     */
    void protocolError(String err) {

        hls.log(client.getInetAddress(), "[protocol error] " + err);
        protocolErrors++;
        protocolErrorOccurredDuringDispatch = true;
        
        if(protocolErrors > hls.getMaximumProtocolErrors()) {
            
            hls.log(client.getInetAddress(), "Disconnecting client because too many consecutive protocol errors occurred (" + protocolErrors + ").");
            
            connected = false;
            
        }
        
    }

    public String toString() {

        return "HLServerDispatcher[address: " + client.getInetAddress() + ", login: " + login + ", privs: " + privileges + ", home: " + homeDirectory + ", " + getUser() + "]";

    }

}
