/**
 * $Id: HLServerListener.java,v 1.2 2001/10/07 04:43:39 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.hotline;

import java.text.DateFormat;

import redlight.utils.TextUtils;
import redlight.utils.TimeFormat;

/**
 * This class implements stubs for server behaviour in response to
 * various events (users joining, chatting, etcetera). More
 * sophisticated policies can be implemented by subclassing this class
 * and by registering it using the {@link
 * HLServer#setHLServerListener} method.  
 */
public class HLServerListener {

    public void serverReady() {

    }

    /**
     * Called just after a user has successfully logged in.
     * @param client the client logging in.
     */
    public void userJoined(HLServerDispatcher client) {
        
    }

    /**
     * Called when a user changes nick, icon or state. If this method
     * returns null, the user's state is changed and everything
     * proceeds normally. If this method returns a non-null value,
     * this value is sent to the client.
     * @param client the client changing (with it's state already
     * changed).
     * ({@see HLProtocol.UserListComponent#CAN_DISCONNECT_USERS}).  
     */
    public void userChanged(HLServerDispatcher client) {

    }

    /**
     * Called when a user leaves.
     * @param client the client that has left.
     */
    public void userLeft(HLServerDispatcher client) {

    }

    /**
     * Called when chat has been received.
     */
    public void chatSent(HLServerDispatcher client,
                         String chat,
                         boolean isAction) {
        
    }

    /**
     * Called when a flat news post has been made.
     */
    public void flatnewsPosted(HLServerDispatcher client,
                               String newPost) {

    }

    /**
     * Called when an account is created.
     * @param client the user requesting account create.
     * @param account the account that is about to be created.
     * @return null if the account is allowed to be created,
     * an error string otherwise.
     */
    public void accountCreated(HLServerDispatcher client, 
                               HLProtocol.AccountInfo account) {
        
    }
    
    /**
     * Called when an account is modified.
     * @param client the user requesting account modification.
     * @param account the account that is about to be created.
     * @return null if the account is allowed to be created,
     * an error string otherwise.
     */
    public void accountModified(HLServerDispatcher client, 
                                HLProtocol.AccountInfo account) {

    }
    
    /**
     * Called when an account is deleted. 
     * @param client the user requesting account deletion.
     * @param account the account that is about to be created.
     * @return null if the account is allowed to be created,
     * an error string otherwise.
     */
    public void accountDeleted(HLServerDispatcher client, 
                               String login) {

    }

    /**
     * Called when a transfer goes in progress.
     * @param tr the transfer request in progress.
     */
    public void transferProgressStart(HLServer.TransferRequest tr) {

    }

    /**
     * Called when a transfer stops being in progress.
     * @param tr the transfer request.
     */
    public void transferProgressStop(HLServer.TransferRequest tr) {

    }

    /**
     * Called when a transfer enters the queue.
     * @param tr the transfer request.
     */
    public void transferQueueStart(HLServer.TransferRequest tr) {

    }

    /**
     * Called when a transfer leaves the queue.
     * @param tr the transfer request.
     */
    public void transferQueueStop(HLServer.TransferRequest tr) {

    }

    /**
     * Called when a single line should be logged.
     * @param line the log line.
     */
    public void logLine(String line) {

        String msg = TimeFormat.formatCurrentDateTime(DateFormat.SHORT, DateFormat.MEDIUM) + ": " + line + "\n";
        System.out.print(msg);

    }

    /**
     * Called when a piece of text containing multiple lines
     * (separated by \n) should be logged.
     * @param text the text to log.  
     */
    public void logText(String text) {

        logLine(text);

    }

}
