/**
 * $Id: rltest.java,v 1.7 2001/10/10 05:36:30 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.rltools;

import java.net.*;
import java.io.*;
import redlight.hotline.*;
import redlight.utils.*;
import redlight.macfiles.*;

/**
 * rltest is a simple program to load a Hotline server by starting
 * a number of threads that keep connecting and disconnecting.
 * 
 * Start rltest doing something like this:
 *
 * $ java -classpath <jarfile> redlight.rltools.rltest
 */
public class rltest {

    static public boolean verbose = false;
    static int numberOfThreads = 5;
    static int ONLINE_TIME_MAX = 10000;
    static int OFFLINE_TIME_MAX = 5000;
    static int numberOfLoops = 0;
    static int port = 5500;
    static int mode = 0;
    static void printUsage() {
        
        System.out.println("Usage: rltest [-v] [--debug] [-n nthreads] [-ut secs] [-dt secs] [-l numberOfLoops] [-p port] [-m mode] REMOTE");
        System.out.println("-v     verbose output");
        System.out.println("-n     number of threads");
        System.out.println("-ut    max uptime per thread");
        System.out.println("-dt    max downtime per thread");
        System.out.println("-p     port (default 5500)");
        System.out.println("-m     test mode (0 or 1)");
        System.out.println("-l     number of loops (0 = practically forever)");
        System.out.println("Examples:");
        System.out.println("rltest 24.132.4.80");
        
    }

    static InetAddress parseArg(String arg) throws IllegalArgumentException {

        DebuggerOutput.debug("parseArg: arg = " + arg);

        try {

            return InetAddress.getByName(arg);

        } catch(UnknownHostException e) {

            throw new IllegalArgumentException("Unknown host: " + e.getMessage());

        }

    }

    static InetAddress[] parseArgs(String[] args, int offset) {

        InetAddress[] fl = new InetAddress[(args.length) - offset];

        for(int i = offset; i < args.length; i++)
            fl[i - offset] = parseArg(args[i]);

        return fl;

    }

    static int parseOptions(String args[]) {
        
        int offset = 0;
        int maxOptions = 14;
        
        for(int i = 0; i < maxOptions && i < args.length; i++) {
            
            if(args[i].equals("--debug")) {
                
                DebuggerOutput.setEnabled(true);
                offset++;

            } else if(args[i].equals("-v")) {

                verbose = true;
                offset++;

            } else if(args[i].equals("-p")) {

                port = Integer.parseInt(args[i + 1]);
                offset += 2;

            } else if(args[i].equals("-ut")) {

                ONLINE_TIME_MAX = Integer.parseInt(args[i + 1]) * 1000;
                offset += 2;

            } else if(args[i].equals("-dt")) {

                OFFLINE_TIME_MAX = Integer.parseInt(args[i + 1]) * 1000;
                offset += 2;

            } else if(args[i].equals("-n")) {

                numberOfThreads = Integer.parseInt(args[i + 1]);
                offset += 2;

            } else if(args[i].equals("-m")) {

                mode = Integer.parseInt(args[i + 1]);
                offset += 2;

            } else if(args[i].equals("-l")) {

                numberOfLoops = Integer.parseInt(args[i + 1]);
                offset += 2;

            }

        }

        return offset;

    }

    /**
     * Main.
     */
    public static void main(String[] args) throws IOException, HLException, InterruptedException {

        if(args.length < 1 || args.length > 15) {

            printUsage();
            System.exit(1);

        }

        int offset = parseOptions(args);

        InetAddress[] hosts = null;

        try {

            hosts = parseArgs(args, offset);

        } catch(IllegalArgumentException e) {

            System.out.println("rltest: illegal argument: " + e.getMessage());
            System.exit(1);

        }

        verboseln("no. threads = " + numberOfThreads);
        Thread[] threads = new Thread[numberOfThreads];
        ThreadGroup testers = new ThreadGroup("Testers");

        if(mode == 0) {

            for(int i = 0; i < numberOfThreads; i++)
                threads[i] = new TestThread(testers, i, hosts[0], port, numberOfLoops);

        } else if(mode == 1) {

            for(int i = 0; i < numberOfThreads; i++) {

                threads[i] = new Test2Thread(testers, i, hosts[0], port, numberOfLoops, i * ONLINE_TIME_MAX, (numberOfThreads * OFFLINE_TIME_MAX) - (i * OFFLINE_TIME_MAX));

            }

        }

        //        int errors = 0;

        
    }

    static void verbose(String s) {

        if(verbose)
            System.out.print(s);

    }

    static void verboseln(String s) {

        if(verbose)
            System.out.println(s);

    }

    static void verboseln() {

        if(verbose)
            System.out.println();

    }

}

class TestThread extends Thread {

    InetAddress host;
    int id, numberOfLoops, port;

    TestThread(ThreadGroup tg, int id, InetAddress host, int port, int numberOfLoops) {

        super(tg, "TestThread " + id);

        this.numberOfLoops = numberOfLoops;
        this.host = host;
        this.port = port;
        this.id = id;
        start();

    }

    public void run() {

        System.out.println("TestThread[" + id + "]: started");

        for(int currentLoop = 0; numberOfLoops == 0 ||
                currentLoop < numberOfLoops;
            currentLoop++) {

            try {
             
                HLClient hlc = new HLClient(host, port);
                System.out.println("TestThread[" + id + "]: connecting...");
                hlc.connect();
                System.out.println("TestThread[" + id + "]: logging in...");
                hlc.login("guest", "", "rltest " + id, 2505);

                int sleepTime = (int) (Math.random() * (double) rltest.ONLINE_TIME_MAX);
                System.out.println("TestThread[" + id + "]: sleeping for " + sleepTime + " ms...");
                Thread.currentThread().sleep(sleepTime);
                System.out.println("TestThread[" + id + "]: woke up, closing connection");
                hlc.close();

            } catch(IOException e) {

                e.printStackTrace();

            } catch(HLException e) {

                e.printStackTrace();

            } catch(InterruptedException e) {
            }

            int sleepTime = (int) (Math.random() * (double) rltest.OFFLINE_TIME_MAX);
            System.out.print("TestThread[" + id + "]: sleeping for " + sleepTime + " ms ");

            if(numberOfLoops != 0 && currentLoop == numberOfLoops - 1)
                System.out.println("before going down...");
            else
                System.out.println("before going online again...");

            try {

                Thread.currentThread().sleep(sleepTime);
                
            } catch(InterruptedException e) {}

        }
        
        System.out.println("TestThread[" + id + "]: exiting");

    }

}

class Test2Thread extends Thread {

    InetAddress host;
    int id, numberOfLoops, port;
    long attackTime, decayTime;

    Test2Thread(ThreadGroup tg, int id, InetAddress host, int port, int numberOfLoops, long attackTime, long decayTime) {

        super(tg, "Test2Thread " + id);

        this.numberOfLoops = numberOfLoops;
        this.host = host;
        this.port = port;
        this.id = id;
        this.attackTime = attackTime;
        this.decayTime = decayTime;

        start();

    }

    public void run() {

        System.out.println("Test2Thread[" + id + "]: started");

        for(int currentLoop = 0; numberOfLoops == 0 ||
                currentLoop < numberOfLoops;
            currentLoop++) {

            try {
             
                System.out.println("Test2Thread[" + id + "]: sleeping before connect (" + attackTime + " ms)...");
                Thread.currentThread().sleep(attackTime);

                HLClient hlc = new HLClient(host, port);
                System.out.println("Test2Thread[" + id + "]: connecting...");
                hlc.connect();
                System.out.println("Test2Thread[" + id + "]: logging in...");
                hlc.login("guest", "", "rltest " + id, 2505);

                System.out.print("Test2Thread[" + id + "]: sleeping " + decayTime + " ms ");

                if(numberOfLoops != 0 && currentLoop == numberOfLoops - 1)
                    System.out.println("before going down...");
                else
                    System.out.println("before disconnect...");
                
                try {
                    
                    Thread.currentThread().sleep(decayTime);
                    
                } catch(InterruptedException e) {}
                
                hlc.close();

            } catch(IOException e) {

                e.printStackTrace();

            } catch(HLException e) {

                e.printStackTrace();

            } catch(InterruptedException e) {
            }

        }
        
        System.out.println("Test2Thread[" + id + "]: exiting");

    }

}
