package retypar.converter;

import java.io.File;
import java.awt.*;
import java.awt.image.BufferedImage;

/**
 * Implement this interface to add an algorithm to retypar.<BR>
 * Place the algorithm class file into the directory <i>retypar/converter/</i> and all Your extra classes in one extra subpackage.
 * Send Your algorithm source(s) to <a href="mailto:jens.heuser@users.sourceforge.net">jens.heuser@users.sourceforge.net</a> to make it publicly available
 * and get it integrated into the next version of retypar.
 * <br><br>
 * The directory structure should look like this:<br>
 * <br>
 * <PRE><OL>
 *          installation folder
 *      ____________|____________
 *      |                       |
 * retypar_0.9.3.jar        retypar
 *                              |
 *                          converter
 *                  ____________|____________
 *                  |                       |
 *          MyAlgo.class                MyPackage
 *                                          |
 *                              MyPackageClasses.class
 *  </PRE></OL>
 * <br>
 * <br>
 * <b>Note:</b><br>
 * <li>
 * You can also extract the jar file and place Your classes into the above mentioned folder already used by retypar.
 * </li>
 * <li>
 * In case of any questions use the forum at <a href="http://www.retypar.de.vu">http://www.retypar.de.vu</a>
 * </li>
 * @version 0.9.3
 * @author <a href="mailto:jens.heuser@users.sourceforge.net">Jens Heuser</a>
 */
public interface Converter{

    /**
     * For compatibility reasons later on.
     * @return The earliest version of retypar this algorithm was testet with (e.g. "0.9.2").
     */
    public String worksWith();

    /**
     * @return A brief description of the algorithm.
     */
    public String getDescription();


    /**
     * @return The author of this algorithm.
     */
    public String getAuthor();


    /**
     * @return The website of the author.
     */
    public String getWebsite();


    /**
     * Don't forget to override this method!
     * @return The name of this algorithm.
     */
    public String toString();


    /**
     * Used to identify the algorithm (i.e. in batch mode).
     * @return A two letter String identifying this algorithm
     */
    public String getShortName();


    /**
     * If Your algorithm needed several different classes, return the package name of them.<BR>
     * All needed extra classes have to be placed in one extra package.
     * @return The package name of the algorithm's subclasses.
     */
    public String broughtPackage();


    /**
     * @return <code>true</code> if this algorithm supports a threshold value.
     */
    public boolean usesThreshold();


    /**
     * If Your algorithm returns pure ASCII, set this to <code>true</code>.
     * @return <code>true</code> if <code>getASCII()</code> returns an ASCII representation of the image.
     * @see Converter#getASCII()
     * @see Converter#getColors()
     */
    public boolean returnsASCII();


    /**
     * If Your algorithm returns an image, set this to <code>true</code>.
     * If not, <code>getASCII()</code> is used to draw the image.
     * @return <code>true</code> if <code>getImage()</code> returns an image.
     * @see Converter#getImage()
     */
    public boolean returnsImage();


    /**
     * If the algorithm does not support a threshold value, the return value is irrelevant.
     * @return The minimum value of the threshold.
     */
    public int getMinThreshold();


    /**
     * If the algorithm does not support a threshold value, the return value is irrelevant.
     * @return The maximum value of the threshold.
     */
    public int getMaxThreshold();


    /**
     * @return The initial value for the threshold
     */
    public int getInitThreshold();


    /**
     * @return The step size for increasing the threshold. 1 is always a good choice.
     */
    public int getThresholdStep();


    /**
     * @return <code>true</code> if the conversion progress is monitored in {@link Converter#getProgress()}
     */
    public boolean monitorsProgess();


    /**
     * The progress (from 0 to 100) of an ongoing conversion.<BR>
     * If progress monitoring is not implemented, this value is irrelevant.
     * @see Converter#monitorsProgess()
     */
    public int getProgress();


    /**
     * All necessary information will be passed to this method.<BR>
     * Do not start conversion in here! Conversion should be started in {@link Converter#getASCII()} respectively
     * in {@link Converter#getImage()}.
     * @param img The image to be converted
     * @param uppercase Use uppercase letters
     * @param lowercase Use lowercase letters
     * @param numbers Use numbers
     * @param special Use special characters
     * @param colored Produce a colored image
     * @param bgColor The background color chosen by the user
     * @param scale The factor by which the original image has to be scaled
     * @param threshold The threshold value
     */
    public void prepare(Image img,
                        boolean uppercase,
                        boolean lowercase,
                        boolean numbers,
                        boolean special,
                        boolean colored,
                        Color bgColor,
                        int scale,
                        int threshold);


    /**
     * This method should not only return the converted image, but trigger its conversion before that.
     * @return The converted image.
     * @see Converter#returnsImage()
     */
    public BufferedImage getImage();


    /**
     * This method should not only return the ASCII version, but trigger its conversion before that.
     * @return A <code>String</code> array representing the image in ASCII
     * @see Converter#getColors()
     * @see Converter#returnsASCII()
     */
    public String[] getASCII();


    /**
     * If colored text was generated,
     * this method has to return an array of colors corresponding to each of the generated characters.
     * @return A two dimensional array of <code>Color</code>s representing every single character's color.
     * @see Converter#getASCII()
     */
    public Color[][] getColors();


    /**
     * The call of this method should immediately stop the conversion.
     */
    public void stop();


    /**
     * If errors occured, return them here.<BR>
     * In case of an error, return from {@link Converter#getASCII()} respectively from {@link Converter#getImage()}
     * in order to get the errors displayed by retypar in a popup window.
     * @return Error messages, or <code>null</code> if no errors occured.
     */
    public String getErrors();


    /**
     *  The array of <code>Parameter</code>s used by this algorithm.
     * @return
     */
    //public Parameter[] getParameters();
}
