/* MakePieChart.java - jfig.utils.MakePieChart 
 *
 * demonstrate how to expand jfig with custom components - this one
 * creates pie charts 
 *
 * 16.03.05 - refactoring
 * 27.02.00 - first version
 *
 */

package jfig.utils;                                // use your own package here


//import java.awt.*;
import java.awt.Color;

import jfig.commands.FigBasicEditor;
import jfig.gui.*;
import jfig.objects.*;



public class MakePieChart {

  static final double _2PI = 2 * Math.PI;

  FigBasicEditor editor = null; // the editor we're using


  /**
   * construct a slice of an pie chart. 
   * We use the following parameters:
   * radius - the pie radius in jfig units (2400dpi)
   * ratio  - the size of this slice, as a fraction of the total 0 .. 1.0
   * angle  - the start angle of this slice
   * Note that you can always set the color, attributes, angle etc. 
   * afterwards by calling standard jfig functions on the FigXXX objects.
   * This method creates FigPolylines to approximate the pie segments
   * (instead of FigArc) to demonstrate setPoints() etc.
   */
  public FigObject makePieSlice( Point center,
                                 double radius, 
                                 double ratio, 
                                 double angle,
                                 Color  color )
  {
    // we approximate pie segments by 200 points per full circle
    //
    int    n_points = 2 + (int) (200.0 * ratio); 
    double deltaphi = ratio * _2PI / (n_points-2);
    Point  points[] = new Point[n_points];

    points[0] = new Point( center.x, center.y );

    double phi = angle;
    for( int i=1; i < n_points; i++) {
      points[i] = new Point( center.x + (int) (radius*Math.cos(phi)+0.5),
                             center.y + (int) (radius*Math.sin(phi)+0.5) 
                           );
      phi += deltaphi;
    }

    FigPolyline poly = new FigPolyline();
    poly.setIsClosed( true );
    poly.setLineColor( Color.white );
    poly.setLineWidth( 1 );

    poly.setFillStyle( FigConstants.SOLID_FILL );
    poly.setFillColor( color );

    // poly.setTrafo( editor.getObjectCanvas().getTrafo() );
    poly.setPoints( points );
    return poly;
  }



  public FigCompound makePieChart( double ratios[] )
  {
    // many jfig method need references to the editor coordinate transformation
    // or the currently active object attributes (line width, fill color, etc.)
    //
    //               R  G  B  gold ltblue cyan4 magenta4 red2 green2 blue2
    int[] colors = { 4, 2, 1, 31,   11,   15,   21,      20,  14,    10 };
    FigCompound  compound = new FigCompound();
    // compound.setTrafo( editor.getObjectCanvas().getTrafo() );

    Point        center   = new Point( 4800, 4800 );
    double       radius   = 2400;         // 2 inch default
    double       angle    = 0.0;
    
    for( int i=0; i < ratios.length; i++ ) {
      Color     color = ColorCache.getColorCache().get( colors[i%colors.length] );
      FigObject slice = makePieSlice( center, radius, ratios[i], angle, color );
      compound.addMember( slice );

      angle += ratios[i] * _2PI;
    }
    return compound;
  }


  public static double[] normalize( double[] data ) {
    double sum = 0.0;
    for( int i=0; i < data.length; i++ ) sum += data[i];
    if (sum != 0.0) {
      for( int i=0; i < data.length; i++ ) data[i] /= sum;
    }
    return data;
  }
  

  public static void main( String argv[] ) {
    MakePieChart MPC   = new MakePieChart();

    // create a jfig editor, this automatically loads the global and user
    // property settings, if any. We are not interested in verbose error
    // messages.
    //
    jfig.utils.ExceptionTracer.setEnabled( false );
    MPC.editor = null;


    // create two pie charts from the raw data
    //
    double[] sales     = { 0.1, 0.2, 0.3, 0.1, 0.15, 0.15 };
    double[] market    = { 0.0, 0.55, 0.23, 0.22 };
    // FigCompound chart1 = MPC.makePieChart( sales );
    // FigCompound chart2 = MPC.makePieChart( market );
    double[] vdh = { 0, 22, 34, 7, 4, 5, 1, 12, 2, 5 };
    double[] ss  = { 0, 20, 54, 16, 8, 5, 4, 26, 4, 21 };
    double[] tnb = { 9, 28, 88, 5, 0, 0, 0, 6, 26, 122 };

    FigCompound chart1 = MPC.makePieChart( normalize( vdh ));
    FigCompound chart2 = MPC.makePieChart( normalize( ss  ));
    FigCompound chart3 = MPC.makePieChart( normalize( tnb ));

    // move the charts, 2400 dpi: 2400 = 1 inch, 7200 = 3 inch, etc.
    //
    chart1.move(     0, 2400 ); 
    chart2.move(  7200, 2400 );         
    chart3.move( 14400, 2400 );         

    // add the chart objects to the editors object list
    //
    MPC.editor.insertIntoObjectList( chart1 );
    MPC.editor.insertIntoObjectList( chart2 );
    MPC.editor.insertIntoObjectList( chart3 );

    MPC.editor.doRedraw();
  }

}
