/**
 * Album Cover Grabber - Grabs album covers automatically off the internet
 * Copyright (C) 2002  Kevin M Schmidt This program is free software; you can
 * redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package com.bluedragontavern.albumgrabber.gui;

import java.io.*;
import java.util.*;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import com.bluedragontavern.albumgrabber.*;
import com.bluedragontavern.albumgrabber.google.*;
import com.bluedragontavern.albumgrabber.util.*;

public class ImageChooser
	implements ActionListener
{
	public final static String COMMAND_YES = "Yes";
	public final static String COMMAND_NO = "No";
	public final static String COMMAND_CROP = "Crop";
	public final static String COMMAND_UNCROP = "Uncrop";
	public final static String COMMAND_SHOWFULL = "ShowFull";
	public final static String COMMAND_NEXT = "Next";
	public final static String COMMAND_PREV = "Prev";
	private JFrame frame;
	private JPanel mainPanel;
	private JScrollPane centerScrollPane;
	private JLabel searchingForLabel = new JLabel("");
	private JLabel foundLabel = new JLabel("");
	private ImagePanel imagePanel;
	private ThumbnailsPanel thumbsPanel;
	private ItemToGrab item;
	private JButton cropButton;
	private JButton uncropButton;
	private JButton showFullButton;
	private JButton nextButton;
	private JButton prevButton;
	private int thumbsIndex = 0;

	/**
	 * Constructor
	 */
	public ImageChooser()
	{
		//Construct the frame
		frame = new JFrame();
		frame.setTitle("Image Chooser");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setLocation(300, 200);

		//Construct the main panel and add to the frame
		mainPanel = new JPanel(new BorderLayout());
		frame.getContentPane().add(mainPanel);

		//Construct the north panel
		Box infoBox = Box.createVerticalBox();
		infoBox.add(new JLabel("Searching for..."));
		infoBox.add(searchingForLabel);
		infoBox.add(new JLabel("Found..."));
		infoBox.add(foundLabel);

		//Add the infoBox to the main panel
		mainPanel.add(infoBox, BorderLayout.NORTH);

		//Construct the button panel
		JPanel buttonPanel = new JPanel();
		JButton yesButton = new JButton("Save");
		yesButton.setActionCommand(COMMAND_YES);
		yesButton.addActionListener(this);

		JButton noButton = new JButton("Skip");
		noButton.setActionCommand(COMMAND_NO);
		noButton.addActionListener(this);
		cropButton = new JButton("Crop");
		cropButton.setActionCommand(COMMAND_CROP);
		cropButton.addActionListener(this);
		uncropButton = new JButton("Uncrop");
		uncropButton.setActionCommand(COMMAND_UNCROP);
		uncropButton.addActionListener(this);
		showFullButton = new JButton("Show Full");
		showFullButton.setActionCommand(COMMAND_SHOWFULL);
		showFullButton.addActionListener(this);
		nextButton = new JButton("Next");
		nextButton.setActionCommand(COMMAND_NEXT);
		nextButton.addActionListener(this);
		prevButton = new JButton("Prev");
		prevButton.setActionCommand(COMMAND_PREV);
		prevButton.addActionListener(this);
		buttonPanel.add(yesButton);
		buttonPanel.add(noButton);
		buttonPanel.add(cropButton);
		buttonPanel.add(uncropButton);
		buttonPanel.add(showFullButton);
		buttonPanel.add(prevButton);
		buttonPanel.add(nextButton);

		//Add the button panel to the main panel
		mainPanel.add(buttonPanel, BorderLayout.SOUTH);
	}

	/**
	 * Sets up the image chooser to display a given album
	 *
	 * @param item DOCUMENT ME!
	 * @throws Exception DOCUMENT ME!
	 */
	public void displayItem(ItemToGrab item)
		throws Exception
	{
		this.item = item;
		cropButton.setVisible(false);
		uncropButton.setVisible(false);
		showFullButton.setVisible(false);
		prevButton.setVisible(false);
		nextButton.setVisible(false);
		searchingForLabel.setText("   " + item.toStringSearch());

		//If the center scroll is not null, clear out the center
		if (centerScrollPane != null)
		{
			mainPanel.remove(centerScrollPane);
		}

		//First see if we are displaying thumbnails or not
		if (item.hasThumbnails())
		{
			//Set location of the frame
			frame.setLocation(100, 50);
			//We want to show only 20 thumbnails at a time, starting at the thumb index
			Vector thumbs = item.getThumbnails();
			int startIndex = thumbsIndex;
			int endIndex = startIndex + 20;
			//If the ending index is greater than the size of the thumbnail vector
			//then just use the size of the vector as the endindex
			if (endIndex > thumbs.size())
			{
				endIndex = thumbs.size();
			}
			thumbsPanel = new ThumbnailsPanel(thumbs.subList(startIndex, endIndex));
			centerScrollPane = new JScrollPane(thumbsPanel);
			centerScrollPane.setPreferredSize(new Dimension(800, 500));
			showFullButton.setVisible(true);
			prevButton.setVisible(true);
			nextButton.setVisible(true);
			foundLabel.setText("");
		}
		else
		{
			//Set location of the frame
			frame.setLocation(300, 200);
			//Create and add the imagePanel to the main panel
			Image image = item.getImage();
			imagePanel = new ImagePanel(image);
			centerScrollPane = new JScrollPane(imagePanel);
			centerScrollPane.setPreferredSize(new Dimension(image.getWidth(null) +
				50,
				image.getHeight(null) + 50));
			foundLabel.setText("   " + item.toStringFound());

			if (item instanceof Artist)
			{
				if (Mp3RootChooser.getInstance().isCropArtistSelected())
				{
					uncropButton.setVisible(true);
				}
				else
				{
					cropButton.setVisible(true);
				}
			}
		}

		//Add the newly created Scroll Pane to the center panel
		mainPanel.add(centerScrollPane, BorderLayout.CENTER);

		frame.pack();
		frame.setVisible(true);
	}

	/**
	 * Listener for the 'yes' and 'no' buttons
	 *
	 * @param e DOCUMENT ME!
	 */
	public void actionPerformed(ActionEvent e)
	{
		try
		{
			String command = e.getActionCommand();

			if (command.equals(COMMAND_YES))
			{
				//If the item has thumbnails, change the list to be only the ones
				//that are selected
				if (item.hasThumbnails())
				{
					Vector thumbs = new Vector();
					Component[] thumbPanels = thumbsPanel.getComponents();

					for (int i = 0; i < thumbPanels.length; ++i)
					{
						ThumbnailPanel thumbPanel = (ThumbnailPanel) thumbPanels[i];

						if (thumbPanel.isSelected())
						{
							thumbs.add(thumbPanel.getThumbnail());
						}
					}

					item.setThumbnails(thumbs);

					//Only allow 1 thumbnail to be stored for now.
					if (thumbs.size() > 1)
					{
						JOptionPane.showMessageDialog(null,
																"Please select only one picture to save",
																"Warning",
																JOptionPane.WARNING_MESSAGE);

						return;
					}
				}

				//Store the image file
				item.writeImage();
				frame.setVisible(false);
				ImageChooserMgr.getInstance().setDisplayingChooser(false);
				//Reset the thumbsIndex
				thumbsIndex = 0;
			}
			else if (command.equals(COMMAND_NO))
			{
				frame.setVisible(false);
				ImageChooserMgr.getInstance().setDisplayingChooser(false);
				//Reset the thumbsIndex
				thumbsIndex = 0;
			}
			else if (command.equals(COMMAND_CROP))
			{
				( (Artist) item).setCropped(true);
				imagePanel.setImage(item.getImage());
				cropButton.setVisible(false);
				uncropButton.setVisible(true);
				imagePanel.repaint();
				frame.pack();
			}
			else if (command.equals(COMMAND_UNCROP))
			{
				( (Artist) item).setCropped(false);
				imagePanel.setImage(item.getImage());
				cropButton.setVisible(true);
				uncropButton.setVisible(false);
				imagePanel.repaint();
				frame.pack();
			}
			else if (command.equals(COMMAND_SHOWFULL))
			{
				//Loop through the thumbnail panels and display any selected
				Component[] thumbPanels = thumbsPanel.getComponents();

				for (int i = 0; i < thumbPanels.length; ++i)
				{
					ThumbnailPanel thumbPanel = (ThumbnailPanel) thumbPanels[i];

					if (thumbPanel.isSelected())
					{
						new FullImageFrame(thumbPanel.getThumbnail());
					}
				}
			}
			else if (command.equals(COMMAND_NEXT))
			{
				//If there are less than 20 displayed on the current page
				//then there are no more to retrieve
				if (thumbsPanel.getComponents().length < 20)
				{
					JOptionPane.showMessageDialog(null,
															"There are no more to retrieve",
															"Warning",
															JOptionPane.WARNING_MESSAGE);
					return;
				}

				Vector currentThumbs = item.getThumbnails();

				//Skip ahead by 20
				thumbsIndex += 20;

				//If we are at the end of the thubnail vector, then go get more
				if (thumbsIndex >= currentThumbs.size())
				{
					currentThumbs.addAll(ImageSearch.getThumbnails(item.
						toStringSearch(),
						currentThumbs.size()));
				}

				displayItem(item);
			}
			else if (command.equals(COMMAND_PREV))
			{
				//Skip to the next 20
				thumbsIndex -= 20;

				//If less than zero, start at 0
				if (thumbsIndex < 0)
				{
					JOptionPane.showMessageDialog(null,
															"There are no more previous",
															"Warning",
															JOptionPane.WARNING_MESSAGE);

					thumbsIndex = 0;
					return;
				}

				displayItem(item);
			}
		}
		catch (Exception ex)
		{
			System.out.println("Exception: " + ex);
		}
	}

	/**
	 * Test main
	 *
	 * @param args DOCUMENT ME!
	 */
	public static void main(String[] args)
	{
		try
		{
			//If proxy is checked, set proxy data
			//Properties systemProps = System.getProperties();
			//systemProps.put("proxySet", "true");
			//systemProps.put("proxyHost", "cdchttpprxy.img.convergys.com");
			//systemProps.put("proxyPort", "80");

			//String authentication = "Basic " +
			//Base64.encodeBytes( ("kschmidt" + ":" +
			//							"XV9408").getBytes());
			//systemProps.put("authentication", authentication);

			ImageChooser imageChooser = new ImageChooser();
			Artist artist = new Artist(new File(
				"C:/Documents and Settings/kschmidt/My Documents/My Music/Better Than Erza"));
			Vector thumbs = new Vector();
			thumbs.add(new Thumbnail(
				"http://www.intellicast.com/images/increquest.jpg",
				"http://www.intellicast.com/SmallWeatherImages/Thumbnails/UVReport/cvg.gif",
				"200 x 300 9k"));
			thumbs.add(new Thumbnail(
				"http://www.intellicast.com/images/increquest.jpg",
				"http://www.intellicast.com/SmallWeatherImages/Thumbnails/UVReport/cvg.gif",
				"200 x 300 9k"));
			thumbs.add(new Thumbnail(
				"http://www.intellicast.com/images/increquest.jpg",
				"http://www.intellicast.com/SmallWeatherImages/Thumbnails/UVReport/cvg.gif",
				"200 x 300 9k"));
			thumbs.add(new Thumbnail(
				"http://www.intellicast.com/images/increquest.jpg",
				"http://www.intellicast.com/SmallWeatherImages/Thumbnails/UVReport/cvg.gif",
				"200 x 300 9k"));
			thumbs.add(new Thumbnail(
				"http://www.intellicast.com/images/increquest.jpg",
				"http://www.intellicast.com/SmallWeatherImages/Thumbnails/UVReport/cvg.gif",
				"200 x 300 9k"));
			artist.setThumbnails(thumbs);
			imageChooser.displayItem(artist);
		}
		catch (Exception ex)
		{
			System.out.println(ex);
		}
	}
}
