package dks.src.gradientEditor;

import java.awt.Color;
import java.awt.LinearGradientPaint;
import java.awt.MultipleGradientPaint;
import java.awt.RadialGradientPaint;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.io.ObjectStreamException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.utils.XML.XMLWritable;
import dks.src.utils.listener.CChangeListenerDelegate;
import dks.src.utils.listener.Changeable;

//$ANALYSIS-IGNORE,codereview.java.rules.cloneable.RuleCloneableOverrideClone
/**
 * This class implements a gradient which can be saved in a XML format <br> date : 3 sept. 07
 * @author   DarK Sidious
 */
public class CGradient extends ArrayList<CGradientColor> implements XMLWritable, Changeable {

	private static final long serialVersionUID = 797246883701189130L;
	protected static final String ERROR_GRADIENT_RADIAL_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : La radialit du dgrad n'a pas t trouv";
	protected static final String ERROR_ROTATION_ANGLE_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : l'angle de rotation du dgrad n'a pas t trouv";
	protected static final String XML_ANGLE_PROPERTY = "angle";
	protected static final String XML_RADIAL_PROPERTY = "radial";
	protected static final String XML_COLORS_PROPERTY = "colors";

	protected int _angle = 0;
	protected boolean _radial = false;

	protected transient CChangeListenerDelegate _listeners;
	protected transient boolean _changed;
	protected transient MultipleGradientPaint _paint;

	protected transient ColorsChangeListener _colorsChangeListener;

	public CGradient() {
		super();
		_listeners = new CChangeListenerDelegate();
		_colorsChangeListener = new ColorsChangeListener();
	}

	/**
	 * @param angle the angle of the gradient's line
	 * @param radial if true, the gradiant is a radial gradient
	 */
	public CGradient(int angle, boolean radial) {
		this();
		_angle = angle;
		_radial = radial;
		_changed = true;
	}


	/* (non-Javadoc)
	 * @see java.util.ArrayList#add(java.lang.Object)
	 */
	@Override
	public boolean add(CGradientColor color) {
		boolean result = super.add(color);
		if (result) {
			addChangeListener(color);
			_changed = true;
		}
		return result;
	}

	/* (non-Javadoc)
	 * @see java.util.ArrayList#add(int, java.lang.Object)
	 */
	@Override
	public void add(int index, CGradientColor color) {
		super.add(index, color);
		_changed = true;
		addChangeListener(color);
	}

	/* (non-Javadoc)
	 * @see java.util.ArrayList#addAll(java.util.Collection)
	 */
	@Override
	public boolean addAll(Collection<? extends CGradientColor> colors) {
		boolean result = super.addAll(colors);
		if (result) {
			for (CGradientColor color : colors) {
				addChangeListener(color);
			}
			_changed = true;
		}
		return result;
	}

	/* (non-Javadoc)
	 * @see java.util.ArrayList#addAll(int, java.util.Collection)
	 */
	@Override
	public boolean addAll(int index, Collection<? extends CGradientColor> colors) {
		boolean result = super.addAll(index, colors);
		if (result) {
			for (CGradientColor color : colors) {
				addChangeListener(color);
			}
			_changed = true;
		}
		return result;
	}

	/* (non-Javadoc)
	 * @see java.util.ArrayList#set(int, java.lang.Object)
	 */
	@Override
	public CGradientColor set(int index, CGradientColor color) {
		CGradientColor resultColor = super.set(index, color);
		addChangeListener(resultColor);
		_changed = true;
		return resultColor;
	}

	protected void addChangeListener(CGradientColor color) {
		color.removeChangeListener(_colorsChangeListener);
		color.addChangeListener(_colorsChangeListener);
	}

	/**
	 * @see java.util.ArrayList#clone()
	 * @return the clone of the object
	 */
	@Override
	public CGradient clone() {
		CGradient result = (CGradient) super.clone();
		result.setAngle(getAngle());
		result.setRadial(isRadial());
		return result;
	}

	/**
	 * @return the angle of the gradient's line
	 */
	public int getAngle() {
		return _angle;
	}

	/**
	 * @param angle the angle of the gradient's line
	 */
	public void setAngle(int angle) {
		if (_angle != angle) {
			_angle = angle;
			_changed = true;
			if (_listeners != null) {
				_listeners.notifyChanged();
			}
		}
	}

	/**
	 * @return if true, the gradient is a radial gradient
	 */
	public boolean isRadial() {
		return _radial;
	}

	/**
	 * @param radial if true, the gradient is a radial gradient
	 */
	public void setRadial(boolean radial) {
		if (_radial != radial) {
			_radial = radial;
			_changed = true;
			_listeners.notifyChanged();
		}
	}

	/**
	 * @param position the position of the gradient
	 * @return the Paint to use to fill a shape
	 */
	public MultipleGradientPaint getPaint(Rectangle position) {
		if (_changed) {
			Collections.sort(this);
			float[] fraction = new float[size()];
			Color[] colors = new Color[size()];
			float temp = -1f;
			for (int i = 0; i < size(); ++i) {
				if (get(i).getPosition() / 100.0f > temp) {
					fraction[i] = get(i).getPosition() / 100.0f;
				} else {
					if (temp < 1) {
						fraction[i] = temp + 0.001f;
					}
				}
				temp = fraction[i];
				colors[i] = get(i).getColor();
			}
			if (_radial) {
				float rayon = 0;
				if (position.width > position.height) {
					rayon = position.height;
				} else {
					rayon = position.width;
				}
				_paint = new RadialGradientPaint(new Point2D.Double(position.x + position.width / 2, position.y + position.height / 2), rayon, fraction, colors);
			} else {
				final float angle = (2.0f * ((float) Math.PI) / 360.0f) * _angle;
				final Point2D start = new Point2D.Double((position.x + position.width / 2) + Math.cos(angle) * position.width / 2, (position.y + position.height / 2) + Math.sin(angle) * position.height
						/ 2);
				final Point2D end = new Point2D.Double((position.x + position.width / 2) + Math.cos(Math.PI + angle) * position.width / 2, (position.y + position.height / 2) + Math.sin(Math.PI + angle)
						* position.height / 2);
				_paint = new LinearGradientPaint(start, end, fraction, colors);
			}
		}
		return _paint;
	}

	public void addChangeListener(ChangeListener listener) {
		_listeners.addListener(listener);
	}

	public void removeChangeListener(ChangeListener listener) {
		_listeners.removeListener(listener);
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLload(org.jdom.Element)
	 * @param root the XML DOM element to use for loading the properties of the gradient
	 */
	@SuppressWarnings("unchecked")
	public void XMLload(Element root) throws JDOMException {
		final String angle = root.getAttributeValue(XML_ANGLE_PROPERTY);
		if (angle == null) {
			throw new JDOMException(ERROR_ROTATION_ANGLE_NOT_FOUND_DESCRIPTION);
		}
		_angle = Integer.parseInt(angle);
		final String radial = root.getAttributeValue(XML_RADIAL_PROPERTY);
		if (radial == null) {
			throw new JDOMException(ERROR_GRADIENT_RADIAL_NOT_FOUND_DESCRIPTION);
		}
		_radial = Boolean.parseBoolean(radial);
		for (Element colorElement : (List<Element>) root.getChildren(XML_COLORS_PROPERTY)) {
			CGradientColor color = new CGradientColor();
			color.XMLload(colorElement);
			add(color);
		}
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLsave(org.jdom.Element)
	 * @param root the XML DOM element to use for saving the properties of the gradient
	 */
	public void XMLsave(Element root) {
		root.setAttribute(XML_ANGLE_PROPERTY, Integer.valueOf(_angle).toString());
		root.setAttribute(XML_RADIAL_PROPERTY, Boolean.valueOf(_radial).toString());
		for (CGradientColor colors : this) {
			Element colorsElement = new Element(XML_COLORS_PROPERTY);
			root.addContent(colorsElement);
			colors.XMLsave(colorsElement);
		}
	}

	/**
	 * @see java.lang.Object#hashCode()
	 * @return the hashcode of the object
	 */
	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = super.hashCode();
		result = PRIME * result + _angle;
		result = PRIME * result + (_radial ? 1231 : 1237);
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @param obj the object to compare
	 * @return the equality of the object
	 */
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof CGradient)) {
			return false;
		}
		if (!super.equals(obj)) {
			return false;
		}
		final CGradient other = (CGradient) obj;
		if (_angle != other._angle) {
			return false;
		}
		if (_radial != other._radial) {
			return false;
		}
		return true;
	}

	public String toString() {
		return "dks.src.gradientEditor.CGradient[angle=" + _angle + ",radial=" + _radial + "]";
	}

	protected class ColorsChangeListener implements ChangeListener {
		public void stateChanged(ChangeEvent e) {
			_changed = true;
			_listeners.notifyChanged();
		}
	}

	protected Object readResolve() throws ObjectStreamException {
		_changed = true;
		_listeners = new CChangeListenerDelegate();
		_colorsChangeListener = new ColorsChangeListener();
		return this;
	}
}
