package dks.src.gradientEditor;

import java.awt.Color;
import java.io.ObjectStreamException;
import java.io.Serializable;

import javax.swing.event.ChangeListener;

import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.utils.XML.XMLWritable;
import dks.src.utils.XML.XMLWriter;
import dks.src.utils.listener.CChangeListenerDelegate;
import dks.src.utils.listener.Changeable;

/**
 * The class implements a color which can be save in a XML format <br> <br> date : 3 sept. 07
 * @author   DarK Sidious
 */
public class CGradientColor implements Comparable<CGradientColor>, XMLWritable, Serializable, Changeable {

	private static final long serialVersionUID = -7982536437816353293L;

	protected static final String ERROR_COLOR_POSITION_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : la position de la couleur du dgrad n'a pas t trouve";
	protected static final String ERROR_GRADIENT_COLOR_NOT_FOUND_DESCRIPTION = "Le fichier xml est invalide : la couleur du dgrad n'a pas t trouve";

	protected static final String XML_COLOR_PROPERTY = "color";
	protected static final String XML_POSITION_PROPERTY = "position";

	protected Color _color;
	protected int _position;

	protected transient CChangeListenerDelegate _listeners;

	protected CGradientColor(){}

	/**
	 * @param color the color to use
	 * @param position the position of the color
	 */
	public CGradientColor(Color color, int position) {
		_color = color;
		_position = position;
		_listeners = new CChangeListenerDelegate();
	}

	/**
	 * @return the color to use
	 */
	public Color getColor() {
		return _color;
	}

	/**
	 * @param color the color to use
	 */
	public void setColor(Color color) {
		if (!_color.equals(color)) {
			_color = color;
			_listeners.notifyChanged();
		}
	}

	/**
	 * @return the position of the color
	 */
	public int getPosition() {
		return _position;
	}

	/**
	 * @param position the position of the color
	 */
	public void setPosition(int position) {
		if (_position != position) {
			_listeners.notifyChanged();
			_position = position;
		}
	}

	/**
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(CGradientColor color) {
		if (_position < color.getPosition()) {
			return -1;
		} else if (_position == color.getPosition()) {
			return 0;
		}
		return 1;
	}

	public void addChangeListener(ChangeListener listener) {
		_listeners.addListener(listener);
	}

	public void removeChangeListener(ChangeListener listener) {
		_listeners.removeListener(listener);
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLload(org.jdom.Element)
	 * @param root the XML Dom element used to save the properties of the color
	 */
	public void XMLload(Element root) throws JDOMException {
		_color = XMLWriter.getColorElement(XML_COLOR_PROPERTY, root);
		if (_color == null) {
			throw new JDOMException(ERROR_GRADIENT_COLOR_NOT_FOUND_DESCRIPTION);
		}

		final String position = root.getAttributeValue(XML_POSITION_PROPERTY);
		if (position == null) {
			throw new JDOMException(ERROR_COLOR_POSITION_NOT_FOUND_DESCRIPTION);
		}
		_position = Integer.parseInt(position);
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLsave(org.jdom.Element)
	 * @param root the XML DOM element used to load the properties of the color
	 */
	public void XMLsave(Element root) {
		root.addContent(XMLWriter.createColorElement(XML_COLOR_PROPERTY, _color));
		root.setAttribute(XML_POSITION_PROPERTY, Integer.valueOf(_position).toString());
	}

	/**
	 * @see java.lang.Object#hashCode()
	 * @return the hashcode of the object
	 */
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((_color == null) ? 0 : _color.hashCode());
		result = PRIME * result + _position;
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @param obj the object to compare
	 * @return the equality of the object
	 */
	public boolean equals(Object obj) {
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof CGradientColor)) {
			return false;
		}
		final CGradientColor other = (CGradientColor) obj;
		if (_color == null) {
			if (other._color != null) {
				return false;
			}
		} else if (!_color.equals(other._color)) {
			return false;
		}
		if (_position != other._position) {
			return false;
		}
		return true;
	}

	public String toString() {
		return "dks.src.gradientEditor.CGradientcolor[color" + _color + ",position=" + _position + "]";
	}

	protected Object readResolve() throws ObjectStreamException {
		_listeners = new CChangeListenerDelegate();
		return this;
	}
}
