package dks.src.warpEditor;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.renderable.ParameterBlock;
import java.io.ObjectStreamException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.media.jai.InterpolationNearest;
import javax.media.jai.JAI;
import javax.media.jai.RenderedOp;
import javax.media.jai.WarpGrid;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jdom.Attribute;
import org.jdom.Element;
import org.jdom.JDOMException;

import dks.src.utils.XML.XMLWritable;
import dks.src.utils.exception.NotEnoughElementException;
import dks.src.utils.listener.CChangeListenerDelegate;
import dks.src.utils.listener.Changeable;
import dks.src.utils.pictures.CPicture;

/**
 * This class implements a warp (a transformation of a picture) which can be saved in a XML format <br> date : 5 sept. 07
 * @author   DarK Sidious
 */
public class CWarp implements XMLWritable, Serializable, Changeable {

	private static final long serialVersionUID = -916309488496901005L;

	protected static final String XML_POINTS_PROPERTY = "points";
	protected static final String XML_LINE_NUMBER_PROPERTY = "lineNumber";
	protected static final String XML_COLUMN_NUMBER_PROPERTY = "columnNumber";
	protected static final String XML_ACTIVE_PROPERTY = "active";

	protected int _lineNumber;
	protected int _colNumber;
	protected boolean _active;

	protected transient ArrayList<CPoints> _points;
	protected transient BufferedImage _image;
	protected transient BufferedImage _tempImage;
	protected transient BufferedImage _destinationImage;
	protected transient boolean _changed;
	protected transient Change _changeListener;
	protected transient CChangeListenerDelegate _changeListenerDelegate;
	protected transient int _xOffset;
	protected transient int _yOffset;

	public CWarp() {
		super();
		_active = true;
		_points = new ArrayList<CPoints>();
		_changeListener = new Change();
		_changeListenerDelegate = new CChangeListenerDelegate();
		_xOffset = 0;
		_yOffset = 0;
		init();
	}

	protected void init() {
		addVerticalLine();
		addVerticalLine();
		addHorizontalLine();
		addHorizontalLine();
	}

	/**
	 * @return the columns number
	 */
	public int getColNumber() {
		return _colNumber;
	}

	/**
	 * @return the lines number
	 */
	public int getLineNumber() {
		return _lineNumber;
	}

	/**
	 * Add a vertical line
	 */
	public void addVerticalLine() {
		_colNumber++;
		for (final CPoints points : _points) {
			points.add(new CPoint());
		}
	}

	/**
	 * Remove a vertical line
	 */
	public void removeVerticalLine() throws NotEnoughElementException {
		if (_colNumber <= 2) {
			throw new NotEnoughElementException();
		}
		_colNumber--;
		for (final CPoints points : _points) {
			points.remove(points.size() - 1);
		}
	}

	/**
	 * Add a horizontal line
	 */
	public void addHorizontalLine() {
		_lineNumber++;
		CPoints points = new CPoints(_colNumber);
		points.addListener(_changeListener);
		_points.add(points);
	}

	/**
	 * Remove a horizontal line
	 */
	public void removeHorizontalLine() throws NotEnoughElementException {
		if (_lineNumber <= 2) {
			throw new NotEnoughElementException();
		}
		_lineNumber--;
		_points.remove(_points.size() - 1);
	}

	/**
	 * the activity of the warp
	 * @return the activity of the warp
	 */
	public boolean isActive() {
		return _active;
	}

	/**
	 * the activity of the warp
	 * @param active the activity of the warp
	 */
	public void setActive(boolean active) {
		if (_active != active) {
			_active = active;
			_changed = true;
		}
	}

	/**
	 * @param g the graphics used to draw the transform picture
	 * @param x the position in the X axis of the picture to draw
	 * @param y the position in the Y axis of the picture to draw
	 * @param width the width of the picture to draw
	 * @param height the height of the picture to draw
	 */
	public void draw(Graphics g, int x, int y, int width, int height) {
		g.drawImage(getWarpImage(width, height), x, y, null);
	}

	/**
	 * @param width the width of the picture
	 * @param height the height of the picture
	 * @return the picture after transformation
	 */
	public BufferedImage getWarpImage(int width, int height) {
		if (_changed) {
			float[] positions = new float[2 * _lineNumber * _colNumber];

			final double horizontalStep = width / (_colNumber - 1.0);
			final double verticalStep = height / (_lineNumber - 1.0);
			int minX = 0;
			int minY = 0;
			int maxX = 0;
			int maxY = 0;
			for (int i = 0; i < _colNumber; i++) {
				final int drawX = (int) (horizontalStep * i);
				for (int j = 0; j < _lineNumber; j++) {
					final int drawY = (int) (verticalStep * j);
					final int index = (j * (_colNumber ) + i) * 2;
					positions[index + 1] = drawY - getPoint(i, j).getTransformationY();
					positions[index] = drawX - getPoint(i, j).getTransformationX();
					minX = getMinValue(minX, (int) getPoint(i, j).getTransformationX());
					maxX = getMaxValue(maxX, (int) getPoint(i, j).getTransformationX());
					minY = getMinValue(minY, (int) getPoint(i, j).getTransformationY());
					maxY = getMaxValue(maxY, (int) getPoint(i, j).getTransformationY());
				}
			}

			int imageWidth = getImageSize(minX, maxX, _image.getWidth());
			int imageHeight = getImageSize(minY, maxY, _image.getHeight());

			_tempImage = CPicture.resizeAndClearImage(_tempImage, imageWidth, imageHeight);
			Graphics2D graphic = (Graphics2D) _tempImage.getGraphics();
			_xOffset = (imageWidth - _image.getWidth()) / 2;
			_yOffset = (imageHeight - _image.getHeight()) / 2;
			graphic.drawImage(_image, _xOffset, _yOffset, null);

			final WarpGrid warp = new WarpGrid(0, (int) horizontalStep, _colNumber - 1, 0, (int) verticalStep, _lineNumber - 1, positions);
			final ParameterBlock pb = new ParameterBlock();
			pb.addSource(_tempImage);
			pb.add(warp);
			pb.add(new InterpolationNearest());

			final RenderedOp dest = JAI.create("warp", pb);

			_destinationImage = dest.getAsBufferedImage();
		}
		return _destinationImage;
	}

	/**
	 * @param x the x index of the point
	 * @param y the y index of the point
	 * @return the point of the indexes
	 */
	public CPoint getPoint(int x, int y) {
		if (x >= 0 && x <= _colNumber && y >= 0 && y <= _lineNumber) {
			return _points.get(y).get(x);
		}
		return null;
	}

	/**
	 * @return the image to use
	 */
	public BufferedImage getImage() {
		return _image;
	}

	/**
	 * @param image the image to use
	 */
	public void setImage(BufferedImage image) {
		if (_image != image) {
			_image = image;
			_changed = true;
		}
	}

	/**
	 * If the warp is over the picture's dimension, the image must be translate to be well-computed.
	 * The final drawing must use the X and Y offset to draw the final picture
	 * @return the offset in the X axis of the picture
	 */
	public int getXOffset() {
		return _xOffset;
	}

	/**
	 * If the warp is over the picture's dimension, the image must be translate to be well-computed.
	 * The final drawing must use the X and Y offset to draw the final picture
	 * @return the offset in the Y axis of the picture
	 */
	public int getYOffset() {
		return _yOffset;
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLload(org.jdom.Element)
	 * @param root the XML DOM Element used to load the warp properties
	 */
	@SuppressWarnings("unchecked")
	public void XMLload(Element root) throws JDOMException {
		final Attribute attribute = root.getAttribute(XML_ACTIVE_PROPERTY);
		if (attribute != null) { // Fonctionnalit rajoute dans la version 1.0.1 => compatibilit avec les fichiers de la version 1.0.0
			_active = Boolean.parseBoolean(attribute.getValue());
		} else {
			_active = true;
		}
		final String lineNumberParse = root.getAttributeValue(XML_LINE_NUMBER_PROPERTY);
		if (lineNumberParse == null) {
			throw new JDOMException("Le fichier xml est invalide : le nombre de ligne du warp n'a pas t trouve");
		}
		final int lineNumber = Integer.parseInt(lineNumberParse);

		final String colNumberParse = root.getAttributeValue(XML_COLUMN_NUMBER_PROPERTY);
		if (colNumberParse == null) {
			throw new JDOMException("Le fichier xml est invalide : le nombre de colonne du warp n'a pas t trouve");
		}
		final int colNumber = Integer.parseInt(colNumberParse);
		_points.clear();
		_lineNumber = 0;
		_colNumber = 0;
		for (int i = 0; i < colNumber; i++) {
			addVerticalLine();
		}
		for (int i = 0; i < lineNumber; i++) {
			addHorizontalLine();
		}
		int i = 0;
		for (final Element element : (List<Element>) root.getChildren(XML_POINTS_PROPERTY)) {
			_points.get(i).XMLload(element);
			i++;
		}
	}

	/**
	 * @see dks.src.utils.XML.XMLWritable#XMLsave(org.jdom.Element)
	 * @param root the XML DOM Element used to save the warp properties
	 */
	public void XMLsave(Element root) {
		root.setAttribute(XML_ACTIVE_PROPERTY, Boolean.valueOf(_active).toString());
		root.setAttribute(XML_LINE_NUMBER_PROPERTY, Integer.valueOf(_lineNumber).toString());
		root.setAttribute(XML_COLUMN_NUMBER_PROPERTY, Integer.valueOf(_colNumber).toString());
		for (final CPoints points : _points) {
			final Element pointsElement = new Element(XML_POINTS_PROPERTY);
			root.addContent(pointsElement);
			points.XMLsave(pointsElement);
		}
	}

	/**
	 * @param listener the change listener to add
	 * @see dks.src.utils.listener.CListenerDelegate#addListener(java.lang.Object)
	 */
	public void addChangeListener(ChangeListener listener) {
		_changeListenerDelegate.addListener(listener);
	}

	/**
	 * @param listener the change listener to remove
	 * @see dks.src.utils.listener.CListenerDelegate#removeListener(java.lang.Object)
	 */
	public void removeChangeListener(ChangeListener listener) {
		_changeListenerDelegate.removeListener(listener);
	}

	protected int getMinValue(int value, int referenceValue) {
		if (value <= referenceValue) {
			return value;
		} else {
			return referenceValue;
		}
	}

	protected int getMaxValue(int value, int referenceValue) {
		if (value >= referenceValue) {
			return value;
		} else {
			return referenceValue;
		}
	}

	protected int getImageSize(int min, int max, int size) {
		int imageSize = size;
		if (max > 0) {
			imageSize += max;
		}
		if (min < 0) {
			imageSize -= min;
		}
		return imageSize;
	}

	/**
	 * @see java.lang.Object#hashCode()
	 * @return the hashcode of the object
	 */
	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + (_active ? 1231 : 1237);
		result = PRIME * result + _colNumber;
		result = PRIME * result + _lineNumber;
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @param obj the object to compare
	 * @return the equality of the object
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof CWarp)) {
			return false;
		}
		final CWarp other = (CWarp) obj;
		if (_active != other._active) {
			return false;
		}
		if (_colNumber != other._colNumber) {
			return false;
		}
		if (_lineNumber != other._lineNumber) {
			return false;
		}
		return true;
	}

	public String toString() {
		String result = "dks.src.warpEditor.CWarp[";
		int compteur = 0;
		for (final CPoints points : _points) {
			if (compteur > 0) {
				result += ",";
			}
			result += "point" + compteur + "=" + points;
			compteur++;
		}
		result += "]";
		return result;
	}

	protected class Change implements ChangeListener {
		public void stateChanged(ChangeEvent e) {
			_changed = true;
		}
	}

	protected Object readResolve() throws ObjectStreamException {
		_changeListener = new Change();
		_changeListenerDelegate = new CChangeListenerDelegate();
		_points = new ArrayList<CPoints>();
		_changed = true;
		_xOffset = 0;
		_yOffset = 0;
		return this;
	}
}
