package typecontrol.controls;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTextField;
import javax.swing.event.ChangeListener;

import typecontrol.CTypeControl;
import typecontrol.IdksTypeControl;
import typecontrol.filters.AbstractFilter;
import typecontrol.filters.DateFilter;
import typecontrol.forms.FrmCalendar;
import typecontrol.forms.FrmCalendar.ValidateListener;
import typecontrol.tools.CCommon;

public class JdksTextField extends JComponent implements IdksTypeControl {

	private static final long serialVersionUID = -4795761055230567364L;
	
	private GUI _gui;
	
	private Action _action;
	private Focus _focus;
	private Keyboard _keyboardListener;
	
	private boolean _passwordable = false;
	private Color _normalBackground;
	
	private AbstractFilter _format;
	private Boolean _autoSelect = true;
	private CTypeControl _typeControl; 
	
	public JdksTextField(String label, String info, AbstractFilter format) {
		super();
		initialize(label, info, format);
	}
	
	public JdksTextField(String label, String info) {
		super();
		initialize(label, info, null);
	}
	
	public JdksTextField(String label, AbstractFilter format) {
		super();
		initialize(label, "", format);
	}
	
	public JdksTextField(String label) {
		super();
		initialize(label, "", null);
	}
	
	public JdksTextField(AbstractFilter format) {
		super();
		initialize("", "", format);
	}

	private void initialize(String label, String info, AbstractFilter format) {
		_typeControl = new CTypeControl();
		setFocusBackground(CTypeControl.FOCUS_COLOR);
		_typeControl.setInfo(info);
		_typeControl.setLabel(label);
		_format = format;
		_focus = new Focus();
		_action = new Action();
		_keyboardListener = new Keyboard();
		_gui = new GUI();
	}
	
	public void setPasswordable(boolean passwordable) {
		_passwordable = passwordable;
		_gui.rebuildGUI();
	}
	
	public void setNormalBackground(Color background) {
		_normalBackground = background;
		_gui.getTxtControl().setBackground(background);
	}
	
	public void setValue(String value) {
		if (isDate()) {
			if (value.equals("")) {
				_gui.getTxtControl().setText("");
				return;
			}
			if (value.contains("-")) {
				Date date = getDateFromString(value, "-");
				GregorianCalendar calendar = new GregorianCalendar();
				calendar.setTime(date);
				NumberFormat format = NumberFormat.getInstance();
				format.setMinimumIntegerDigits(2);
				_gui.getTxtControl().setText(format.format(calendar.get(Calendar.DAY_OF_MONTH)) + "/" + format.format(calendar.get(Calendar.MONTH) + 1) + "/" + calendar.get(Calendar.YEAR));
			}
		}
		else {
			_gui.getTxtControl().setText(value);
		}
		if (_format != null) {
			_format.validate();
		}
	}
	
	public String getValue() {
		return _gui.getTxtControl().getText();
	}
		
	private boolean isDate() {
		if (_format != null && _format instanceof DateFilter) {
			return true;
		}
		return false;
	}
			
	public void setAutoSelect(Boolean autoSelect) {
		_autoSelect = autoSelect;
	}
		
	public void setInfo(String info) {
		_typeControl.setInfo(info);
		_gui.rebuildGUI();
	}
	
	public void setLabel(String label) {
		setLabel(label);
		_gui.rebuildGUI();
	}
		
	private void btnDate_Click() {
		FrmCalendar calendar = _gui.getFrmCalendar();
		calendar.setModal(true);
		calendar.setVisible(true);
		calendar.dispose();
	}
	
	protected void btnInfo_Click() {
		JOptionPane.showMessageDialog(null, getInfo(), "Informations", JOptionPane.OK_OPTION + JOptionPane.INFORMATION_MESSAGE);
	}
	
	private void calCalendar_Validate(Boolean isValid, Date date) {
		if (isValid) {
			if (date != null) {
				DateFormat format = DateFormat.getDateInstance(DateFormat.SHORT);
				_gui.getTxtControl().setText(format.format(date));
			} else {
				_gui.getTxtControl().setText("");
			}
		}
	}
	
	private static Date getDateFromString(String hour, String separator) {
		String date[] = hour.split(separator);
		if (date.length == 3) {
			if (date[2].length() > 2) {
				date[2] = date[2].substring(0, 2);
			}
			GregorianCalendar calendar = new GregorianCalendar();
			calendar.set(GregorianCalendar.YEAR, Integer.parseInt(date[0]));
			calendar.set(GregorianCalendar.MONTH, Integer.parseInt(date[1]) - 1);
			calendar.set(GregorianCalendar.DAY_OF_YEAR, Integer.parseInt(date[2]));
			return calendar.getTime();
		}
		return null;
	}
	
	public void setEnabled(Boolean enabled) {
		_gui.getTxtControl().setEnabled(enabled);
	}
	
	public boolean isEnabled() {
		if (_gui != null) {
			return _gui.getTxtControl().isEnabled();
		} else {
			return true;
		}
	}
	
	private void txtControl_Change() {
		raiseChangeListener(this);
	}
	
	private class GUI {
		protected static final String FRM_CALENDAR_TITLE = "Slectionnez une date";
		protected static final String BTN_INFO_NAME = "btnInfo";
		protected static final String BTN_DATE_NAME = "btnDate";
		protected static final String CAL_CALENDAR_NAME = "calCalendar";
		protected static final String TXT_CONTROL_NAME = "txtControl";
		
		private final Dimension BTN_SIZE = new Dimension(18, 18);
		private final Dimension TXT_DEFAULT_SIZE = new Dimension(100, 20);
		
		private JButton _btnInfo;
		private JButton _btnDate;
		private JTextField _txtControl;
		private JPanel _panButtons;
		private FrmCalendar _frmCalendar;
		
		public GUI() {
			rebuildGUI();
		}
		
		protected void rebuildGUI() {
			setLayout(new BorderLayout());
			_txtControl = null;
			removeAll();
			if (getLabel() != null && !getLabel().equals("")) {
				add(getLabLabel(), BorderLayout.WEST);
			}
			
			add(getTxtControl(), BorderLayout.CENTER);
			getTxtControl().addFocusListener(_focus);

			_panButtons = null;
			add(getPanButtons(), BorderLayout.EAST);
			updateUI();
		}
						
		protected JTextField getTxtControl() {
			if (_txtControl == null) {
				if (_passwordable) {
					_txtControl = new JPasswordField();
				} else {
					_txtControl = new JTextField();
				}
				if (_format != null) {
					_txtControl.setDocument(_format);
				}
				_txtControl.setPreferredSize(TXT_DEFAULT_SIZE);
				_txtControl.setName(TXT_CONTROL_NAME);
				_txtControl.addKeyListener(_keyboardListener);
			}
			return _txtControl;
		}
		
		protected JButton getBtnInfo() {
			if (_btnInfo == null) {
				_btnInfo = new JButton();
				_btnInfo.setIcon(new ImageIcon(Toolkit.getDefaultToolkit().getImage(getClass().getClassLoader().getResource("ressources/info.png"))));
				_btnInfo.setPreferredSize(BTN_SIZE);
				_btnInfo.setName(BTN_INFO_NAME);
				_btnInfo.addActionListener(_action);
			}
			return _btnInfo;
		}

		protected JButton getBtnDate() {
			if (_btnDate == null) {
				_btnDate = new JButton();
				_btnDate.setIcon(new ImageIcon(Toolkit.getDefaultToolkit().getImage(getClass().getClassLoader().getResource("ressources/Date.PNG"))));
				_btnDate.setPreferredSize(BTN_SIZE);
				_btnDate.setName(BTN_DATE_NAME);
				_btnDate.addActionListener(_action);
			}
			return _btnDate;
		}

		protected JPanel getPanButtons() {
			if (_panButtons == null) {
				_panButtons = new JPanel();
				_panButtons.setLayout(new BorderLayout());
				if (isDate()) {
					_panButtons.add(getBtnDate(), BorderLayout.WEST);
				}
				if (getInfo() != null && !getInfo().equals("")) {
					_panButtons.add(getBtnInfo(), BorderLayout.EAST);
				}
			}
			return _panButtons;
		}
	
		protected FrmCalendar getFrmCalendar() {
			if (_frmCalendar == null) {
				_frmCalendar = new FrmCalendar(FRM_CALENDAR_TITLE, false, CCommon.getDateFromString(getTxtControl().getText()));
				_frmCalendar.addValidateListener(new Validate());
			}
			return _frmCalendar;
		}
	}
	
	private class Action implements ActionListener {
		public void actionPerformed(ActionEvent arg0) {
			String name = ((JComponent) arg0.getSource()).getName();
			if (name.equals(GUI.BTN_INFO_NAME)) {
				btnInfo_Click();
			} else if (name.equals(GUI.BTN_DATE_NAME)) {
				btnDate_Click();
			}
		}
	}
	
	private class Focus implements FocusListener {		
		public Focus() {
			super();
		}
				
		public void focusGained(FocusEvent arg0) {
			if (_typeControl.getFocusBackground() != null) {
				_gui.getTxtControl().setBackground(_typeControl.getFocusBackground());
			}
			if (_autoSelect) {
				if (_gui != null) {
					_gui.getTxtControl().setSelectionStart(0);
					_gui.getTxtControl().setSelectionEnd(_gui.getTxtControl().getText().length());
				}
			}
		}

		public void focusLost(FocusEvent arg0) {
			if (_typeControl.getFocusBackground() != null) {
				_gui.getTxtControl().setBackground(_normalBackground);
			}
			if (_gui.getTxtControl().getDocument() instanceof AbstractFilter) {
				AbstractFilter filter = (AbstractFilter) _gui.getTxtControl().getDocument();
				if (!filter.validate()) {
					_gui.getTxtControl().setText("");
				}
			}
		}
	}

	private class Validate implements ValidateListener {
		public void validate(Boolean isValid, Date date) {
			calCalendar_Validate(isValid, date);
		}
	}

	private class Keyboard extends KeyAdapter {
		public void keyTyped(KeyEvent arg0) {
			String name = ((JComponent) arg0.getSource()).getName();
			if (name.equals(GUI.TXT_CONTROL_NAME)) {
				txtControl_Change();
			}
		}
	}

	public void addChangeListener(ChangeListener changeListener) {
		_typeControl.addChangeListener(changeListener);
	}

	public JButton getBtnInfo() {
		return _typeControl.getBtnInfo();
	}

	public Color getFocusBackground() {
		return _typeControl.getFocusBackground();
	}

	public String getInfo() {
		return _typeControl.getInfo();
	}

	public String getLabel() {
		return _typeControl.getLabel();
	}

	public JLabel getLabLabel() {
		return _typeControl.getLabLabel();
	}

	public boolean isLabelled() {
		return _typeControl.isLabelled();
	}

	public void raiseChangeListener(IdksTypeControl source) {
		_typeControl.raiseChangeListener(source);
	}

	public void removeChangeListener(ChangeListener changeListener) {
		_typeControl.removeChangeListener(changeListener);
	}

	public void setFocusBackground(Color focusBackground) {
		_typeControl.setFocusBackground(focusBackground);
	}

	public void setLabelAlignment(int alignment) {
		_typeControl.setLabelAlignment(alignment);
	}

	public void setLabelWidth(int width) {
		_typeControl.setLabelWidth(width);
	}
}
