#include <pnm.h>
#include <stdio.h>
#include <getopt.h>
#include "Unshake.h"

int main (int argc, char **argv)
{
  /* This example program allows a PNM image to be deconvolved, using a
     sub-image to derive the blur (kernel). For convenience, it uses
     "UnshakeRasters" to hold the images. Note that UnshakeBasic (the
     routine which does the work) does not require this - it is merely for
     convenience. UnshakeBasic accepts a 3-d array of type Uns_int, by default
     short unsigned int, the number of colours, width and height of the image
     in that array. UnshakeRaster supplies these.

     Also note that the sub-image (kernel image) is created by the routine
     getKernImage.*/

  /* These are respectively the image to be deconvolved, and the sub-image
     from which the kernel will be derived.*/
  struct UnshakeRaster image, kImage;

  /* This structure carries all of the important parameters of the
     deconvolution, including the address of the workspace in memory.*/
  struct UP parameters;
  /* Not needed by UnshakeBasic, this structure is used to generate the
     kernel image.*/
  struct UnshakeGeom kGeometry;

  int ch, opt, retval;
  const char* optstring = "hmr:a:b:i:p:f";
  float relNoise, blur, ACTIND, recoverTime;


  /* Required by libpnm*/  
  pnm_init(&argc, argv);
  
  /* Initialise!*/
  kGeometry.width = 0;
  kGeometry.height = 0;
  kGeometry.x = 0;
  kGeometry.y = 0;
  /* These are vital for UnshakeRaster method.*/
  image.raster = (uns_int3)NULL;
  kImage.raster = (uns_int3)NULL;

  /* This does all the required initialisation (including default values)
     for UnshakeBasic. */
  paramInit(&parameters);
  
  /* Obvious - options changing the defaults.*/
  while ((opt=getopt(argc, argv, optstring))!=-1) {
    switch(opt) {
    case 'h':
      fprintf(stderr, "Usage Unshake [opts] < infile > outfile\n");
      fprintf(stderr, "Where opts are:\n");
      fprintf(stderr, "\t-a float  \tRMS amplification of noise (default=3.0).\n");
      fprintf(stderr, "\t-b integer\tEstimate of diameter of blur in pixels.\n");
      fprintf(stderr, "\t\t\tWill be ignored if too large for kernel image.\n");
      fprintf(stderr, "\t-i float  \tAmount of detail in scene - 1.5 is boring,\n");
      fprintf(stderr, "\t\t\t0.7 is detailed (default=1.2).\n");
      fprintf(stderr, "\t-m        \tEstimate the detail directly.\n");
      fprintf(stderr, "\t-p integer\tNumber of iterations per pixel in the kernel space for phase\n");
      fprintf(stderr, "\t\trecovery.\n");
      fprintf(stderr, "\t-r WxH+X+Y\tCalculate kernel from kernel image, width W,\n");
      fprintf(stderr, "\t\t\theight H, beginning at (X, Y) from top left of image.\n");
      fprintf(stderr, "\t-h        \tPrint this message.\n");

      exit(0);
      break;
      
    case 'r':
      if (sscanf(optarg, "%dx%d+%d+%d", &(kGeometry.width), &(kGeometry.height), &(kGeometry.x), &(kGeometry.y))==-1) {
	fprintf(stderr, "Bad geometry for -r option.\n");
	exit(1);
      }
      break;

    case 'a':
      if (sscanf(optarg, "%g", &relNoise)==-1) {
	fprintf(stderr, "Bad -a option.\n");
	exit(1);
      }
      parameters.relNoise = relNoise;
      break;

    case 'b':
      if (sscanf(optarg, "%f", &blur)==-1) {
	fprintf(stderr, "Bad -a option.\n");
	exit(1);
      }
      parameters.blur = blur;
      break;

    case 'i':
      if (sscanf(optarg, "%g", &ACTIND)==-1) {
	fprintf(stderr, "Bad -a option.\n");
	exit(1);
      }
      parameters.ACTIND = ACTIND;
      break;

    case 'm':
      parameters.autoMode = 1;
      break;

    case 'p':
      if (sscanf(optarg, "%g", &recoverTime)==-1) {
	fprintf(stderr, "Bad -p option.\n");
	exit(1);
      }
      parameters.recoverTime = recoverTime;
      break;

    case 'f':
      parameters.flatWin = 1;
      break;

    default:
      exit(1);
    }
  }

  while (!feof(stdin)) {
    if ((filepToURas(stdin, &image))!=0) {
      exit(1);
    }

    /* Get subset of the image to be used for the kernel*/
    if ((retval=getKernImage(&image, &kImage, &parameters, &kGeometry))!=0) {
      fprintf(stderr, parameters.comment);
      exit(retval);
    }

    UnshakeBasic(kImage.raster, kImage.width, kImage.height,
		 image.raster, image.width, image.height, image.colours,
		 &parameters);
    freeUns(parameters.space);
    if (parameters.comment!= (char*) NULL) {
      fprintf(stderr, "%s\n", parameters.comment);
    } else {
      if (saveURas(stdout, &image)!=0) {
	exit(1);
      }
    }
    
    while (isspace((ch = getc(stdin))));
    ungetc(ch, stdin);
  }
  freeUns(kImage.raster);
  freeUns(image.raster);
  exit(0);
}

