/**
 *  Classroom Scheduler
 *  Copyright (C) 2004 Colin Archibald, Ph.D.
 *  https://sourceforge.net/projects/cr-scheduler/
 *
 *  Licensed under the Academic Free License version 2.0
 */

package resources;

import java.awt.*;
import java.util.*;
import java.io.*;
import java.text.DecimalFormat;

/**
 * Maintains and manipulates Time for the Classroom Scheduler.  Stores only
 * hours minutes and the AM flag.  Used by TimeSlot class.  Assumes that 7 AM
 * is 0, and 10PM is 900.  There are 900 schedulable minutes in the day.
 */

public class CRTime implements Serializable, Comparable {
    static final long serialVersionUID = 911;
    
    private int     crMinutes;
    private int     hours, minutes;
    private boolean am;
    
    /**
     * Constructor declaration
     *
     *
     * @param crMinutes
     *
     */
    
    public CRTime(int crMinutes) throws CRTimeException {
        if (crMinutes < 0 || crMinutes > 900) {
            throw new CRTimeException("Not in range 7 AM - 10 PM");
        }
        
        this.crMinutes = crMinutes;
        
        if (this.crMinutes < 300) {
            am = true;
        } else {
            am = false;
        }
        
        minutes = this.crMinutes % 60;
        hours = ((this.crMinutes - minutes) / 60 + 7) % 12;
        
        if (hours == 0) {
            hours = 12;
        }
    }
    
    /**
     * Constructor declaration
     *
     *
     * @param int hours
     * @param int minutes
     * @param boolean am
     *
     */
    
    public CRTime(int hours, int minutes, boolean am) throws CRTimeException {
        this.am = am;
        
        if (hours < 1 || hours > 12) {
            throw new CRTimeException("Hours out of range");
        }
        
        if (minutes < 0 || minutes > 60) {
            throw new CRTimeException("Minutes out of range");
        }
        
        if (hours < 7 && am) {
            throw new CRTimeException("Nothing can be scheduled before 7:00 AM");
        }
        
        if (hours > 10 && hours != 12 && !am) {
            throw new CRTimeException("Nothing can be scheduled after 10:00 PM");
        }
        
        this.hours = hours;
        this.minutes = minutes;
        
        if (am) {
            this.crMinutes = (this.hours - 7) * 60 + this.minutes;
        } else {
            this.crMinutes = 300 + this.hours % 12 * 60 + this.minutes;
        }
    }
    
    public int getcrMinutes() {
        return crMinutes;
    }
    
    public int getHours() {
        return hours;
    }
    
    public int getMinutes() {
        return minutes;
    }
    
    public boolean getam() {
        return am;
    }
    
    public String toString() {
        DecimalFormat df = new DecimalFormat("00");
        
        return hours + ":" + df.format(minutes) + (am ? "A" : "P");
    }
    
    // implement the Comparable interface for sorting
    
    public int compareTo(Object obj) {
        if (!(obj instanceof CRTime)){
            return 0;
        } 
        CRTime time2 = (CRTime) obj;        
        return crMinutes - time2.crMinutes;
    }
    
    // determine if 2 CRTimes are equal - avoid duplicates
    
    public boolean equals(Object obj) {
        if(!(obj instanceof CRTime)){
            return false;
        }
        CRTime time2 = (CRTime) obj;
        
        if (crMinutes == time2.crMinutes && am == time2.am)
            return true;
        return false;
    }
}

