/* (C) 1999-2000 Samuel Audet <guardia@cam.org>

Profitable use of this source code based on its execution or sale
excluding the cost of the media, shipping, manwork or supporting
hardware is not allowed unless granted by the author himself.  Any
modifications or inclusion of this code in other non-profitable programs
must contain this message and the original author's name. Programs based
on any of this source code must therefore contain the original or
modified source code files.  Use of this source code in a commercial
program will require permission from the author.  No more than 50% of
the original size of the source code from Disk Indexer can be used in a
non-commercial program, unless granted by the author.

*/

import java.io.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.*;
import java.util.zip.*;

public class AddDiskDialog extends JDialog implements ActionListener, Runnable
{
   JLabel message = new JLabel("Root directory of drive:");
   JTextField rootDir, desc;
   JButton ok = new JButton("Ok"), cancel = new JButton("Cancel");
   JPanel buttonPane = new JPanel();
   JFrame mainFrame;
   JCheckBox doZipButton, doID3TagButton, doGIFJPGButton;
   JLabel status = new JLabel("");

   boolean[] scanOptions = new boolean[3];

   HDatabase database;
   DynamicTreeNode parent;
   Properties options;

   boolean success = false;

public AddDiskDialog(JFrame frame, DynamicTreeNode parentContainer, HDatabase database, Properties options)
{
   super(frame,true);
   this.setTitle("Adding a New Disk Volume to \"" + parentContainer + "\"");
   mainFrame = frame;
   this.database = database;
   parent = parentContainer;
   this.options = options;

   rootDir = new JTextField(options.getProperty("disk.add"));
   desc = new JTextField(options.getProperty("disk.desc"));

   message.setAlignmentX(LEFT_ALIGNMENT);

   rootDir.setPreferredSize(new Dimension(rootDir.getPreferredSize().width,rootDir.getPreferredSize().height+5));
   rootDir.setMaximumSize(new Dimension(Short.MAX_VALUE, rootDir.getPreferredSize().height));
   rootDir.setAlignmentX(LEFT_ALIGNMENT);

   desc.setPreferredSize(new Dimension(desc.getPreferredSize().width,desc.getPreferredSize().height+5));
   desc.setMaximumSize(new Dimension(Short.MAX_VALUE, desc.getPreferredSize().height));
   desc.setAlignmentX(LEFT_ALIGNMENT);

   doZipButton = new JCheckBox("Scan Zip Files"); doZipButton.setMnemonic(KeyEvent.VK_Z);
   doZipButton.setSelected(Boolean.valueOf(options.getProperty("disk.dozip")).booleanValue());
   doID3TagButton = new JCheckBox("Scan ID3 Tags"); doID3TagButton.setMnemonic(KeyEvent.VK_3);
   doID3TagButton.setSelected(Boolean.valueOf(options.getProperty("disk.doid3tag")).booleanValue());
   doGIFJPGButton = new JCheckBox("Scan GIF and JPG Images"); doGIFJPGButton.setMnemonic(KeyEvent.VK_I);
   doGIFJPGButton.setSelected(Boolean.valueOf(options.getProperty("disk.dogifjpg")).booleanValue());

   buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.X_AXIS));
   buttonPane.add(ok);
   buttonPane.add(Box.createRigidArea(new Dimension(5,0)));
   buttonPane.add(cancel);
   buttonPane.setAlignmentX(LEFT_ALIGNMENT);
   ok.addActionListener(this);
   cancel.addActionListener(this);
   ok.setMnemonic(KeyEvent.VK_O);
   cancel.setMnemonic(KeyEvent.VK_C);


   JPanel dialogPane = new JPanel();
   dialogPane.setLayout(new BoxLayout(dialogPane, BoxLayout.Y_AXIS));
   dialogPane.add(message);
   dialogPane.add(Box.createRigidArea(new Dimension(0,5)));
   dialogPane.add(rootDir);
   dialogPane.add(Box.createRigidArea(new Dimension(0,5)));
   dialogPane.add(desc);
   dialogPane.add(Box.createRigidArea(new Dimension(0,5)));
   dialogPane.add(doZipButton);
   dialogPane.add(doID3TagButton);
   dialogPane.add(doGIFJPGButton);
   dialogPane.add(Box.createRigidArea(new Dimension(0,5)));
   dialogPane.add(buttonPane);

   dialogPane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

   addWindowListener(new WindowAdapter()
   {
      public void windowClosing(WindowEvent e)
      {
         mainFrame.requestFocus();
      }
      boolean focusSet = false;
      public void windowActivated(WindowEvent e)
      {
         if(!focusSet)
            rootDir.requestFocus();
         focusSet = true;
      }
   });

   this.getContentPane().add(dialogPane,BorderLayout.CENTER);
   this.getContentPane().add(status,BorderLayout.SOUTH);
   this.pack();
   setLocationRelativeTo(frame);
}

public void actionPerformed(ActionEvent e)
{
   String command = e.getActionCommand();

   if(command == "Ok")
   {
      scanOptions[0] = doZipButton.isSelected();
      scanOptions[1] = doID3TagButton.isSelected();
      scanOptions[2] = doGIFJPGButton.isSelected();

      ok.setEnabled(false);
      cancel.setEnabled(true);

      addNewDisk();
   }
   else if(command == "Cancel")
   {
      if(addDiskThread == null) // we are not in any operations
      {
         this.dispose();
         mainFrame.requestFocus();
      }
      else
      {
         addDiskThread = null;
      }
   }
}

Thread addDiskThread = null;
HDataNode addDiskToNode = null;
File addDirectory = null;


   class UpdateStatus implements Runnable
   {
      String path = null;

      public void run()
      {
         status.setText("Adding " + path);
      }
   }

   // only one instance to minimize object creation
   UpdateStatus updateStatus = new UpdateStatus();

private void updateStat(String path)
{
   updateStatus.path = path;
   SwingUtilities.invokeLater(updateStatus);
}


// add disk thread
public void run()
{
   try
   {
      Thread myThread = Thread.currentThread();

      updateStat(addDirectory.getPath());
      addFiles(addDiskToNode, scanOptions, addDirectory, myThread);
   }
   catch(IOException e)
   {
      SwingUtilities.invokeLater(new Runnable()
      {
         public void run()
         {
            status.setText("");
            ok.setEnabled(true);
            JOptionPane.showMessageDialog(mainFrame,"Error: Could not add the complete Disk Volume. Database corrupted?","Error",JOptionPane.ERROR_MESSAGE);
            AddDiskDialog.this.dispose();
            mainFrame.requestFocus();
         }
      });
   }

   if(addDiskThread != null) // not cancelled
   {
      success = true;
      addDiskThread = null;

      SwingUtilities.invokeLater(new Runnable()
      {
         public void run()
         {
            status.setText("");
            ok.setEnabled(true);

            options.put("disk.add", rootDir.getText());
            options.put("disk.desc", desc.getText());
            options.put("disk.dozip", String.valueOf(doZipButton.isSelected()));
            options.put("disk.doid3tag", String.valueOf(doID3TagButton.isSelected()));
            options.put("disk.dogifjpg", String.valueOf(doGIFJPGButton.isSelected()));

            AddDiskDialog.this.dispose();
            DiskIndexer.saveProperties();
            mainFrame.requestFocus();
         }
      });
   }
   else
   {
      try
      {
         if(addDiskToNode != null)
            database.remove(addDiskToNode);
         addDiskToNode = null;
      }
      catch(IOException e)
      {
         SwingUtilities.invokeLater(new Runnable()
         {
            public void run()
            {
               JOptionPane.showMessageDialog(mainFrame,"Database corrupted. Could not Delete all nodes added.","Error",JOptionPane.ERROR_MESSAGE);
            }
         });
      }

      SwingUtilities.invokeLater(new Runnable()
      {
         public void run()
         {
            status.setText("");
            ok.setEnabled(true);
         }
      });
   }

}


// scans and adds a new drive
protected boolean addNewDisk()
{
   if(addDiskThread != null)
      return false;

   File aDirectory = new File(rootDir.getText());
   if(!aDirectory.isDirectory())
   {
      JOptionPane.showMessageDialog(mainFrame,"Error " + rootDir.getText() + " is not a valid directory","Error",JOptionPane.ERROR_MESSAGE);
      ok.setEnabled(true); // try again, better luck next time!
      return false;
   }

   try
   {
      HDataNode anode = new HDataNode();
      anode.fileName = rootDir.getText();
      anode.desc = desc.getText();
      anode.type = anode.v;
      anode.length = 0;
      anode.dateModified = System.currentTimeMillis();
      anode.scanOptions = scanOptions; // clone?
      database.write((HDataNode) parent.getUserObject(),anode);

      addDiskToNode = anode;
      addDirectory = aDirectory;
      addDiskThread = new Thread(this,"Adding");
      addDiskThread.start();
   }
   catch(IOException ee)
   {
      JOptionPane.showMessageDialog(this,"Error trying to add new disk volume: Could not write to database.","Error",JOptionPane.ERROR_MESSAGE);
      this.dispose();
      mainFrame.requestFocus();
   }

   return true;
}


public static String[] splitPath(String path)
{
   Vector returnVector = new Vector(10);
   int nextSlash = path.indexOf('/'),
       lastSlash = -1;

   while(nextSlash != -1)
   {
      returnVector.addElement(path.substring(lastSlash+1,nextSlash));
      lastSlash = nextSlash;
      nextSlash = path.indexOf('/',lastSlash+1);
   }

   returnVector.addElement(path.substring(lastSlash+1));
   String[] returnArray = new String[returnVector.size()];
   returnVector.copyInto(returnArray);

   return returnArray;
}

public void addZipFile(HDataNode parent, ZipFile zipFile, Thread myThread) throws IOException
{
   HDataNode curParent = parent;

   // process directory entries.  Done before in case of buggy ZIP or JDK
   // (like IBM's) where directory entries can come after other entries that
   // have already used directories of the same name
   Enumeration files = zipFile.entries();
   String lastPath = null;
   while(files.hasMoreElements() && myThread == addDiskThread)
   {
      ZipEntry anEntry = (ZipEntry)files.nextElement();
      HDataNode aNode = new HDataNode();
      if(!anEntry.isDirectory())
         continue;

      String curPath = anEntry.getName();
      int lastSlash = curPath.lastIndexOf('/',curPath.length()-2);
      if(lastSlash == -1)
      {
         curPath = null;
         curParent = parent;
      }
      else
         curPath = curPath.substring(0,lastSlash);

      if(curPath != null && !curPath.equals(lastPath))
      {
         String[] dirNames = splitPath(curPath);
         curParent = parent; // we go back up from the root
         for(int i = 0; i < dirNames.length; i++)
         {
            HDataNode newParent = database.findNode(curParent,dirNames[i]);
            if(newParent == null)
            {
               HDataNode oldParent = curParent;
               curParent = new HDataNode();
               curParent.fileName = dirNames[i];
               curParent.type = aNode.d;
               curParent.length = 0;
               curParent.dateModified = System.currentTimeMillis();
               database.write(oldParent,curParent);
               oldParent = null;
            }
            else
               curParent = newParent;
         }
      }

      //updateStat(zipFile.getName() + " (" + anEntry.getName() + ")" );
      aNode.copyFrom(anEntry);
      HDataNode sameNode = database.findNode(curParent,aNode.fileName);
      if(sameNode != null) // the directory was already added due to buggy ZIP or JDK
      {
         HDataNode oldNode = (HDataNode) sameNode.clone();
         sameNode.length = aNode.length;
         sameNode.dateModified = aNode.dateModified;
         database.update(oldNode, sameNode);
      }
      else
         database.write(curParent, aNode);

      lastPath = curPath;
   }

   // process non-directory entries, pretty similar, but subtile differences
   curParent = parent;
   files = zipFile.entries();
   lastPath = null;
   while(files.hasMoreElements() && myThread == addDiskThread)
   {
      ZipEntry anEntry = (ZipEntry)files.nextElement();
      HDataNode aNode = new HDataNode();
      if(anEntry.isDirectory())
         continue;

      String curPath = anEntry.getName();
      int lastSlash = curPath.lastIndexOf('/'); // the difference
      if(lastSlash == -1)
      {
         curPath = null;
         curParent = parent;
      }
      else
         curPath = curPath.substring(0,lastSlash);

      if(curPath != null && !curPath.equals(lastPath))
      {
         String[] dirNames = splitPath(curPath);
         curParent = parent; // we go back up from the root
         for(int i = 0; i < dirNames.length; i++)
         {
            HDataNode newParent = database.findNode(curParent,dirNames[i]);
            if(newParent == null)
            {
               HDataNode oldParent = curParent;
               curParent = new HDataNode();
               curParent.fileName = dirNames[i];
               curParent.type = aNode.d;
               curParent.length = 0;
               curParent.dateModified = System.currentTimeMillis();
               database.write(oldParent,curParent);
               oldParent = null;
            }
            else
               curParent = newParent;
         }
      }

      //updateStat(zipFile.getName() + " (" + anEntry.getName() + ")" );
      aNode.copyFrom(anEntry);
      database.write(curParent, aNode);

      lastPath = curPath;
   }

}

public void addFiles(HDataNode parent, boolean[] scanOptions, File dir, Thread myThread) throws IOException
{
   String[] list = dir.list();

   // scanning directories
   for(int i=0; i < list.length && myThread == addDiskThread; i++)
   {
      File aFile = new File(dir,list[i]);
      HDataNode anode = new HDataNode();

      if(aFile.isDirectory())
      {
         updateStat(aFile.getPath());
         anode.copyFrom(aFile,scanOptions);
         database.write(parent, anode);
         addFiles(anode,scanOptions,aFile,myThread);
      }
      aFile = null;
      anode = null;
   }

   updateStat(dir.getPath());
   boolean lastWasZip = false;

   // scanning files
   for(int i=0; i < list.length && myThread == addDiskThread; i++)
   {
      File aFile = new File(dir,list[i]);
      HDataNode anode = new HDataNode();

      if(!aFile.isDirectory())
      {
         if(scanOptions[0]) // scanZip
         {
            try
            {
               ZipFile aZipFile = new ZipFile(aFile);
               updateStat(aFile.getPath());
               anode.copyFrom(aFile,scanOptions);
               anode.type = anode.z;
               database.write(parent, anode);
               addZipFile(anode,aZipFile,myThread);
               lastWasZip = true;
            }
            catch(ZipException e)
            {
               if(lastWasZip)
                  updateStat(dir.getPath());
               //updateStat(aFile.getPath());
               anode.copyFrom(aFile,scanOptions);
               database.write(parent, anode);
               lastWasZip = false;
            }
         }
         else
         {
            //updateStat(aFile.getPath());
            anode.copyFrom(aFile,scanOptions);
            database.write(parent, anode);
         }
      }
      aFile = null;
      anode = null;
   }
   list = null;
}



public static HDataNode showDialog(JFrame frame, DynamicTreeNode container, HDatabase database, Properties options)
{
   AddDiskDialog dialog = new AddDiskDialog(frame,container,database,options);
   dialog.setVisible(true);
   return dialog.addDiskToNode;
}

public static HDataNode showAutomaticDialog(JFrame frame, DynamicTreeNode container, HDatabase database, String disk, String desc, boolean[] scanOptions)
{
   AddDiskDialog dialog = new AddDiskDialog(frame,container,database,new Properties());
   dialog.rootDir.setText(disk);
   dialog.desc.setText(desc);
   if(scanOptions != null)
   {
      dialog.doZipButton.setSelected(scanOptions[0]);
      dialog.doID3TagButton.setSelected(scanOptions[1]);
      dialog.doGIFJPGButton.setSelected(scanOptions[2]);
      dialog.scanOptions[0] = scanOptions[0];
      dialog.scanOptions[1] = scanOptions[1];
      dialog.scanOptions[2] = scanOptions[2];
   }
   dialog.ok.setEnabled(false);
   dialog.cancel.setEnabled(true);
   dialog.addNewDisk();
   dialog.setVisible(true);
   return dialog.addDiskToNode;
}


}

