/* (C) 1999-2000 Samuel Audet <guardia@cam.org>

Profitable use of this source code based on its execution or sale
excluding the cost of the media, shipping, manwork or supporting
hardware is not allowed unless granted by the author himself.  Any
modifications or inclusion of this code in other non-profitable programs
must contain this message and the original author's name. Programs based
on any of this source code must therefore contain the original or
modified source code files.  Use of this source code in a commercial
program will require permission from the author.  No more than 50% of
the original size of the source code from Disk Indexer can be used in a
non-commercial program, unless granted by the author.

*/


import java.io.*;
import java.util.*;

// only for timer
import javax.swing.*;
import java.awt.event.*;

public class HBufferManager implements ActionListener
{
   Hashtable buffer = null;
   int bufferMaxSize = 0;
   Timer timer;
   RandomAccessFile dataFile = null;

   public static final int DATABASE_VERSION = 1;
   public int databaseVersion = DATABASE_VERSION;
   protected int firstFreePage = -1;
   public HDataNode rootNode = null;

   HDataPage templateFreePage;

// bufferSize in bytes is approximate
public HBufferManager(RandomAccessFile dataFile, int bufferSize)
{
   this.dataFile = dataFile;
   setBufferMaxSize(bufferSize);
   buffer = new Hashtable(4*bufferMaxSize/3+1,(float)0.75);
   timer = new Timer(2000,this);
   timer.setRepeats(false);

   readHeader();

   templateFreePage = new HDataPage(rootNode);

   templateFreePage.primaryPointer = -2;
   templateFreePage.parentPointer = -2;
   templateFreePage.prevBrother = -1;
   templateFreePage.nextBrother = -1;
   templateFreePage.used = templateFreePage.headerSize;
   templateFreePage.startAt = 0;
   templateFreePage.amount = 0;
}

public void setBufferMaxSize(int bufferSize)
{
   if(bufferSize < HDatabase.PAGE_SIZE*10)
      bufferMaxSize = 10;
   else
      bufferMaxSize = bufferSize/HDatabase.PAGE_SIZE;
}

public void startLazyWriteTimer()
{
   if(!timer.isRunning())
      timer.start();
}
// timer
public void actionPerformed(ActionEvent e)
{
   flushAllPages();
}

public void flushAllPages()
{
//   System.out.println("waiting...");
   synchronized(rootNode)
   {

   long time = System.currentTimeMillis();
   // timer for lazy write
   try
   {
      writeHeader();
      int i = 0;

      Enumeration allPages = buffer.elements();
//   System.out.println("starting trash " + (System.currentTimeMillis()-time) );
      while(allPages.hasMoreElements())
      {
//   System.out.println(i++ + " " + (System.currentTimeMillis()-time) );

         HDataPage page = (HDataPage) allPages.nextElement();
         if(page.needUpdateOnDisk)
         {
            // if(page.getNodeSize() == 0) hey
            page.writePage(dataFile);
         }
      }
//      System.out.println("flushing "  + (System.currentTimeMillis()-time) );
      ((BufferedRandomAccessFile) dataFile).flushBuffer();

   }
   catch(IOException ee)
   {
      System.out.println("error: " + ee);
      // error
   }
//   System.out.println("done! " +  (System.currentTimeMillis()-time) );

   }
}

protected Object putInBuffer(int key, Object value) throws IOException
{
   synchronized(rootNode)
   {

   // flush some pages
   if(buffer.size() >= bufferMaxSize)
   {

//System.out.println("oops, buffer filled " + buffer.size() );
      long time = System.currentTimeMillis();
      Vector pagesToFlush = new Vector(bufferMaxSize/10+1); // 10%
      Enumeration allPages = buffer.elements();
      while(allPages.hasMoreElements())
      {
         HDataPage page = (HDataPage) allPages.nextElement();
         boolean added = false;

         for(int i = 0; i < pagesToFlush.size() && !added; i++)
         {
            if(page.timestamp < ((HDataPage) pagesToFlush.elementAt(i)).timestamp) // LRU
            {
               pagesToFlush.insertElementAt(page,i);
               added = true;
            }
         }
         if(!added && pagesToFlush.size() < bufferMaxSize/10)
            pagesToFlush.addElement(page);

         if(pagesToFlush.size() > bufferMaxSize/10)
            pagesToFlush.setSize(bufferMaxSize/10);
      }

//   System.out.println("found all pages to trash" + (System.currentTimeMillis()-time));
      // flush the 10% LRU pages
      for(int i = 0; i < pagesToFlush.size(); i++)
      {
         HDataPage page = (HDataPage) pagesToFlush.elementAt(i);
         if(page.needUpdateOnDisk)
            page.writePage(dataFile);
         buffer.remove(new Integer(page.myPointer));
      }
//      System.out.println("finished trashing " + buffer.size() + " " + (System.currentTimeMillis()-time) );
   }

   return buffer.put(new Integer(key),value);

   }
}

protected void readHeader()
{
   rootNode = new HDataNode();

   synchronized(rootNode)
   {

   rootNode.fileName = "root";
   rootNode.type = rootNode.c;
   rootNode.length = 0;
   rootNode.myPointer = 0;
   rootNode.primaryPointer = 0;
   try
   {
      dataFile.seek(0);
      databaseVersion = dataFile.readInt();
      rootNode.dateModified = dataFile.readLong();
      rootNode.childPointer = dataFile.readInt();
      firstFreePage = dataFile.readInt();
   }
   catch(IOException e)
   {
      rootNode.dateModified = System.currentTimeMillis();
      rootNode.childPointer = -1;
   }

   }
}

public synchronized boolean writeHeader() throws IOException
{
   synchronized(rootNode)
   {

   dataFile.seek(0);
   dataFile.writeInt(databaseVersion);
   dataFile.writeLong(rootNode.dateModified);
   dataFile.writeInt(rootNode.childPointer);
   dataFile.writeInt(firstFreePage);

   }

   return false;
}


public HDataPage getPage(int pointer) throws IOException
{
   HDataPage loadedPage;

//   if(lastUsedPage != null && pointer == lastUsedPage.myPointer)
//      loadedPage = lastUsedPage;
//   else
      loadedPage = (HDataPage) buffer.get(new Integer(pointer));

   if(loadedPage == null)
      loadedPage = readPage(pointer);

//   lastUsedPage = loadedPage;

   return loadedPage;
}

protected HDataPage readPage(int pointer) throws IOException
{
   synchronized(rootNode)
   {

   HDataPage loadedPage = new HDataPage(rootNode);

   loadedPage.readPage(dataFile, pointer);
   putInBuffer(pointer,loadedPage);

   return loadedPage;

   }
}



public void removePage(int pointer) throws IOException
{
   synchronized(rootNode)
   {

   buffer.remove(new Integer(pointer));

   templateFreePage.myPointer = pointer;
   if(firstFreePage == -1 || pointer < firstFreePage)
      firstFreePage = pointer;

   templateFreePage.writePage(dataFile);

   }
}

public HDataPage createChildPage(HDataNode parent) throws IOException
{
   HDataPage newPage = new HDataPage(rootNode);

   // creating new child
   newPage.primaryPointer = parent.childPointer = getFreePagePointer();
   newPage.parentPointer = parent.myPointer;
   newPage.prevBrother = -1;
   newPage.nextBrother = -1;
   newPage.used = newPage.headerSize;
   newPage.startAt = 0;
   newPage.amount = 0;
   newPage.myPointer = newPage.primaryPointer;

   newPage.timestamp = System.currentTimeMillis();
   newPage.needUpdateOnDisk = true;

   putInBuffer(newPage.myPointer,newPage);
   if(parent != rootNode)
      getPage(parent.myPointer).needUpdateOnDisk = true;

   return newPage;
}

public HDataPage createBrotherPage(HDataPage brother) throws IOException
{
   HDataPage newPage = new HDataPage(rootNode);

   // creating new child
   newPage.primaryPointer = brother.primaryPointer;
   newPage.parentPointer = brother.parentPointer;
   newPage.prevBrother = brother.myPointer;
   newPage.nextBrother = -1;
   newPage.used = newPage.headerSize;
   if(brother.amount == -1)
   {
      newPage.startAt = 0;
      newPage.amount = -1;
   }
   else
   {
      newPage.startAt = (short) (brother.startAt+brother.amount);
      newPage.amount = 0;
   }
   newPage.myPointer = getFreePagePointer();

   newPage.timestamp = System.currentTimeMillis();
   newPage.needUpdateOnDisk = true;

   brother.setNextBrother(newPage.myPointer);

   putInBuffer(newPage.myPointer,newPage);
   putInBuffer(brother.myPointer,brother);

   return newPage;
}

public HDataPage createBlobPage(HDataNode aNode) throws IOException
{
   HDataPage newPage = new HDataPage(rootNode);

   // creating new child
   newPage.primaryPointer = aNode.blobPointer = getFreePagePointer();
   newPage.parentPointer = aNode.myPointer;
   newPage.prevBrother = -1;
   newPage.nextBrother = -1;
   newPage.used = newPage.headerSize;
   newPage.startAt = 0;
   newPage.amount = -1;
   newPage.myPointer = newPage.primaryPointer;

   newPage.timestamp = System.currentTimeMillis();
   newPage.needUpdateOnDisk = true;

   putInBuffer(newPage.myPointer,newPage);
   getPage(aNode.myPointer).needUpdateOnDisk = true;

   return newPage;
}

protected int getFreePagePointer() throws IOException
{
   synchronized(rootNode)
   {

   int endFilePointer = (int)((dataFile.length()-1)/HDatabase.PAGE_SIZE+1);
   int foundPage = -1;
   HDataPage dummyPage = new HDataPage(rootNode);

   while(buffer.get(new Integer(endFilePointer)) != null)
      endFilePointer++;

   // -1 means the first free page is at the end of the file
   if(firstFreePage == -1)
   {
      foundPage = Math.max(1, endFilePointer);
   }
   // -2 means there are some pages free around
   // otherwize, it means there are some pages free at or after page #
   else
   {
      int i;
      if(firstFreePage == -2)
         i = 1;
      else
         i = firstFreePage;

      for(; i < endFilePointer && foundPage == -1; i++)
      {
         if(buffer.get(new Integer(i)) == null && dummyPage.checkFreePage(dataFile,i))
         {
            firstFreePage = i;
            foundPage = i;
         }
      }

      if(foundPage == -1) // couldn't find anything, last free page at EOF
      {
         foundPage = endFilePointer;
         firstFreePage = -1;
      }
   }

   return foundPage;

   }
}

public void trimFile() throws IOException
{
   synchronized(rootNode)
   {

   int lastPointer = (int)(dataFile.length()/HDatabase.PAGE_SIZE - 1);

   // nothing to trim, since valid pages were found in the buffer after the EOF on disk
   Enumeration allKeys = buffer.keys();
   while(allKeys.hasMoreElements())
   {
      Integer key = (Integer) allKeys.nextElement();
      if(key.intValue() >= lastPointer)
         return;
   }
   allKeys = null;

   HDataPage dummyPage = new HDataPage(rootNode);
   dummyPage.readPage(dataFile, lastPointer--);

   while(lastPointer > 0 && dummyPage.primaryPointer == -2 &&
         buffer.get(new Integer(lastPointer+1)) == null) // while free pages are found
   {
      dummyPage.readPage(dataFile, lastPointer--);
   }

   // trim pages from the end...
   try
   {
      dataFile.setLength(HDatabase.PAGE_SIZE*(lastPointer+2));
   }
   catch(java.lang.NoSuchMethodError e) { }
   //System.out.println(" shaved up to " + HDatabase.PAGE_SIZE*(lastPointer+2));

   }
}

}
