import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;


// There are 2 partially separate bugs that are fixed here:
//
// Bug 1: If the menu has the same mnemonic as an item in the menu, then
// pressing ALT-Mnemonic would not just open the menu, but also execute
// the item with the same mnemonic at once.
// This has been fixed by first checking the menu's mnemonic against the
// KeyEvent. If the event has the menu's mnemonic, then the MenuItem
// does no further processing of it (otherwise it too would be activated).
//
// Bug 2: If a menu has the same mnemonic as an item in *another* menu, then
// opening this other menu and pressing the mnemonic key would both executed
// the command behind the MenuItem and open the menu with the same mnemonic.
// This has been fixed by checking, if the temporarily created MenuKeyEvent
// has been set to "consumed" by the MenuItem. If so, then KeyEvent must also
// be consumed before it is passed on, otherwise it would be used to activate
// the other menu.
//
// JHT

public class MyJMenuItem extends JMenuItem
{
   public MyJMenuItem() {
      super();
   }
   public MyJMenuItem(Icon icon) {
      super(icon);
   }

   public MyJMenuItem(String text) {
      super(text);
   }

   public MyJMenuItem(String text, Icon icon) {
      super(text,icon);
   }

   public MyJMenuItem(String text, int mnemonic) {
      super(text,mnemonic);
   }

   public void processKeyEvent(KeyEvent e,MenuElement path[],MenuSelectionManager manager)
   {
      JMenu myMenu = (JMenu)((JPopupMenu)this.getParent()).getInvoker();
      // Fix for Bug 1:
      // return if the KeyEvent's value is the same as my menu's mnemonic
      if (((char) myMenu.getMnemonic() == e.getKeyChar() ||
           (char) myMenu.getMnemonic() == e.getKeyChar() - ('a'-'A')) &&
           (e.getModifiers() & Event.ALT_MASK) > 0)
         return;

      // Fix for Bug 2:
      // call "processMenuKeyEvent" with a temporarily created MenuKeyEvent
      // as usual (taken from original code), but check if this event is
      // consumed, and if so, consume the original event too.
      MenuKeyEvent ev = new MenuKeyEvent(e.getComponent(), e.getID(),
                            e.getWhen(), e.getModifiers(), e.getKeyCode(),
                            e.getKeyChar(), path, manager);
      processMenuKeyEvent(ev);
      if (ev.isConsumed())
         e.consume();
   }
}

// Note: This class is defined as a private inner class to the class
// "ExMenu" which extends JMenu and which was created earlier for other
// reasons. It contains for example convenience methods to add items to
// the menu (which now use instances of MyJMenuItem instead of JMenuItem)
// and other stuff.
// Since this class already existed, it was easiest for me to make the
// extended MyJMenuItem an inner class, because then I could easily
// check the menu-mnemonic (for Bugfix 1) of the whole menu by
// accessing ExMenu.this.getMnemonic().
//
// JHT

// I modified it using JMenu myMenu = (JMenu)((JPopupMenu)this.getParent()).getInvoker();
// and added all constructors to make it an external class - Samuel Audet <guardia@cam.org>
