/*
 * $Id: DataModel.java,v 1.2 2004/09/02 00:50:37 aim Exp $
 *
 * Copyright 2004 Sun Microsystems, Inc., 4150 Network Circle,
 * Santa Clara, California 95054, U.S.A. All rights reserved.
 */

package org.jdesktop.swing.data;

import java.util.List;

/**
 * <p>
 * Abstract model interface for representing a record of named data fields.
 * The map provides a uniform API for accessing data which may be contained
 * in a variety of data model constructs underneath, such as RowSet,
 * TabularDataModel, or arbitrary JavaBean classes.  The user-interface
 * Binding classes use this interface to &quot;bind&quot; user-interface
 * components to field elements in a data model without having to understand
 * the specific flavor of data model being used by the application.
 * For example, a field element may map to a named column on a RowSet
 * or a property on a JavaBean, but the binding classes don't need to
 * understand those underlying data structures in order to read and write
 * values.</p>
 * <p>
 * For each named field, the data model provides access to:
 * <ul>
 * <li>meta-data: information describing the data field, such as type
 *                and edit constraints</li>
 * <li>value: the current value of the field</li>
 * </ul>
 * </p>
 * <p>
 * Often data models are collections of like-objects, such as the rows in a
 * RowSet, or a list of JavaBeans.  This interface provides a mechanism
 * to index into such a collection such that at any given time, the data model
 * contains the element values associated with the &quot;current&quot; record index
 * into that collection (the current row, or the current bean, etc).
 * </p>
 *
 *
 * @author Amy Fowler
 * @version 1.0
 */
public interface DataModel extends MetaDataProvider {

    /**
     * @return array containing the names of all data fields in this map
     */
    String[] getFieldNames();

    MetaData[] getMetaData();

    /**
     *
     * @param fieldName String containing the name of the field
     * @return MetaData object which describes the named field
     */
    MetaData getMetaData(String fieldName);

    /**
     *
     * @param fieldName String containing the name of the field
     * @return Object containing the current value of the named field
     */
    Object getValue(String fieldName);

    /**
     *
     * @param fieldName String containing the name of the field
     * @param value Object containing the current value of the named field
     */
    void setValue(String fieldName, Object value);

    /**
     *
     * @return integer containing the number of fields in this data model
     */
    int getFieldCount();

    /**
     *
     * @return integer containing the number of records accessible
     *         from this data model
     */
    int getRecordCount();

    /**
     * Sets the current record index such that the field values
     * in this data model represent the values contained at the specified index
     * in the underlying data model.
     * @param index integer representing the current index into the underlying
     *        data model's records
     */
    void setRecordIndex(int index);

    /**
     *
     * @return integer representing the current index into the underlying
     *        data model's records
     */
    int getRecordIndex();

    /**
     * Adds the specified validator for the fields represented by this
     * data model.
     * A validator object may be used to perform validation checks which
     * require analyzing more than one field value in a single check.
     * This DataModel instance will be passed in as the <code>value</code>
     * parameter to the validator's <code>validate</code> method.
     *
     * @see #removeValidator
     * @see #getValidators
     * @param validator Validator object which performs validation checks on
     *        this set of data field values
     */
    public void addValidator(Validator validator);

    /**
     * Removes the specified validator from this data model.
     * @see #addValidator
     * @param validator Validator object which performs validation checks on
     *        this set of data field values
     */
    public void removeValidator(Validator validator);

    /**
     *
     * @return array containing the validators registered for data model
     */
    Validator[] getValidators();

    /**
     * Adds the specified value change listener to be notified when
     * the value is changed outside of calling <code>setValue</code> directly.
     * @param valueChangeListener ValueChangeListener object to receive events
     *        when the field value changes
     */
    void addValueChangeListener(ValueChangeListener valueChangeListener);

    /**
     * Removes the specified value change listener from this value adapter.
     * @param valueChangeListener ValueChangeListener object to receive events
     *        when the field value changes
     */
    void removeValueChangeListener(ValueChangeListener valueChangeListener);

    /**
     *
     * @return array containing the ValueChangeListener objects registered
     *         on this data model
     */
    ValueChangeListener[] getValueChangeListeners();


}
