/*
 * 01/09/2002 - 20:43:57
 *
 * PluginInfoSheet.java -
 * Copyright (C) 2002 Csaba Kertsz
 * kcsaba@jdictionary.info
 * www.jdictionary.info
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


package info.jdictionary.pluginstuff;

import info.jdictionary.JDictionary;
import java.io.File;
import java.io.InputStream;
import java.io.DataInputStream;
import java.net.URL;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;
import java.util.ArrayList;
import java.io.Serializable;

//PluginInfoSheet object contains many useful information
//about plugins, and it is essential for plugin handling.
//This class also contains some static methods that help
//you to set up a PluginInfoSheet object from an url,
//or a pluginfile automatically or return null, if it's
//impossibe :)


public class PluginInfoSheet implements Serializable {
    String name;
    String url;
    String version;
    String minJdictionaryVersion;
    String author;
    String fileName;
    String size;
    String description;
    File file;


    public static PluginInfoSheet CreateInfoSheet(File pluginFile) {
        PluginInfoSheet s = CreateInfoSheet(pluginFile, JDictionary.getString("PluginInfoSheetFileName") + "_" + JDictionary.getLanguage());
        if (s != null)
            return s;
        System.out.println("InfoSheetCreator: Local specific info sheet is not found in plugin named: " + pluginFile.getName() + " Trying to load the default sheet");
        return CreateInfoSheet(pluginFile, JDictionary.getString("PluginInfoSheetFileName"));
    }


    public static PluginInfoSheet CreateInfoSheet(File pluginFile, String fileName) {
        ZipFile pluginZipFile = null;
        try {
            pluginZipFile = new ZipFile(pluginFile);
        }
        catch(java.util.zip.ZipException e) {return null;}
        catch(java.io.IOException e) {return null;}

        PluginInfoSheet sheet;
        ZipEntry entry = pluginZipFile.getEntry(fileName);
        if (entry == null) {
            //System.out.println("InfoSheetCreator: Invalid plugin: " + pluginFile.getName() + " " + fileName + " not found in this file.");
            return null;
        }
        byte[] b = new byte[(int) entry.getSize()];
        try {
            InputStream in = pluginZipFile.getInputStream(entry);
            in.read(b);
        } catch (java.io.IOException ex) {
            return null;
        }
        sheet = createInfoSheet(b);

        sheet.setSize(Long.toString(pluginFile.length()));
        sheet.setFileName(pluginFile.getName());
        sheet.file = pluginFile;

        return sheet;
    }


    public static Object[] CreateInfoSheets(ZipFile pluginFile) {

        Object[] s = CreateInfoSheets(pluginFile, "InfoSheets_" + JDictionary.getLanguage());
        if (s != null)
            return s;
        return CreateInfoSheets(pluginFile, "InfoSheets");
    }


    public static Object[] CreateInfoSheets(ZipFile pluginFile, String fileName) {
        ZipEntry entry = pluginFile.getEntry(fileName);
        if (entry == null) {
            //System.out.println("InfoSheetCreator: Invalid plugin: " + pluginFile.getName() + " Info file not found in this file.");
            return null;
        }
        byte[] b = new byte[(int) entry.getSize()];
        try {
            InputStream in = pluginFile.getInputStream(entry);
            in.read(b);
        } catch (java.io.IOException ex) {
            return null;
        }
        return createInfoSheets(b);
    }

    public static PluginInfoSheet CreateInfoSheet(URL url) {

        String temp = null;
        DataInputStream din = null;
        byte[] b = new byte[4096];
        try {
            din = new DataInputStream(url.openStream());
            din.read(b);
        } catch (java.io.IOException e) {
            return null;
        }

        return createInfoSheet(b);
    }


    public static Object[] CreateInfoSheets(URL url) {
        byte[] b = null;
        try {
            int offset = 0;
            int loadedBytesCount = 0;
            int length = url.openConnection().getContentLength();
            if (length == -1)
                return null;
            b = new byte[length];
            DataInputStream din = new DataInputStream(url.openStream());
            while (loadedBytesCount < length) {
                length -= loadedBytesCount;
                offset += loadedBytesCount;
                loadedBytesCount = din.read(b, offset, length);
                if (loadedBytesCount == -1) {
                    System.out.println("CreateInfoSheet(): Wrong thing happened when I attempted to download the information about the new plugins");
                    din.close();
                    return null;
                }
            }
        din.close();
        }
        catch (java.io.IOException e) {
            System.out.println("CreateInfoSheet(): IOException.");
            return null;
        }
        catch(java.lang.Exception e) {
            System.out.println("CreateInfoSheet(): Wrong thing happened. See the stack trace below:");
            e.printStackTrace();
            return null;
        }
        return createInfoSheets(b);
    }


    public static PluginInfoSheet createInfoSheet(byte[] infoSheetData) {

        int offset = 0;
        int n = 0;
        PluginInfoSheet sheet = new PluginInfoSheet();

        while (n < infoSheetData.length) {
            if (infoSheetData[n] == '\n') {
                String temp = new String(infoSheetData, offset, n - offset - 1); //-1 will cut off the \n at the end of the String
                offset = n + 1;
                if (temp.charAt(0) == 'E')
                    break;
                determineLine(temp, sheet);
            }
            n++;
        }
        return sheet;
    }


    public static Object[] createInfoSheets(byte[] infoSheetDatas) {
        int offset = 0;
        int n = 0;
        PluginInfoSheet sheet = new PluginInfoSheet();
        ArrayList sheets = new ArrayList();

        while (n < infoSheetDatas.length) {
            if (infoSheetDatas[n] == '\n') {
                String temp = new String(infoSheetDatas, offset, n - offset - 1);
                offset = n + 1;
                if (temp.charAt(0) == 'E') {
                    sheets.add(sheet);
                    sheet = new PluginInfoSheet();
                }
                determineLine(temp, sheet);
            }
            n++;
        }
        sheets.add(sheet);
        return (sheets.toArray());
    }


    static void determineLine(String line, PluginInfoSheet sheet) {
        switch (line.charAt(0)) {
        case'N':
            sheet.name = line.substring(2, line.length());
            break;
        case'V':
            sheet.version = line.substring(2, line.length());
            break;
        case'S':
            sheet.size = line.substring(2, line.length());
            break;
        case'A':
            sheet.author = line.substring(2, line.length());
            break;
        case'D':
            sheet.description = line.substring(2, line.length());
            break;
        case'F':
            sheet.fileName = line.substring(2, line.length());
            if(sheet.fileName.indexOf("http://") != -1) {
                sheet.url = sheet.fileName;
                sheet.fileName = urlToFileName(sheet.fileName);
            }
            break;
        case'J':
            sheet.minJdictionaryVersion = line.substring(2, line.length());
            break;
        default:
        }
    }

    static String urlToFileName(String url) {
                int s = url.lastIndexOf("/");
                if(s > -1)
                    return url.substring(s + 1, url.length());
                return null;
            }

    public PluginInfoSheet() {}


    public PluginInfoSheet(String name, String version, String Author, String fileName, String size, String Description, String minJDictionaryVersion) {
        this.name = name;
        this.version = version;
        this.author = author;
        this.fileName = fileName;
        this.size = size;
        this.description = description;
        this.minJdictionaryVersion = minJdictionaryVersion;
    }

    public String getName() {
        return name;
    }


    public String getVersion() {
        return version;
    }


    public String getAuthor() {
        return author;
    }


    public String getFileName() {
        return fileName;
    }


    public String getSize() {
        return size;
    }

    public String getSizeInKB() {
        if(getSize() != null)
            return new Integer(Integer.parseInt(getSize()) / 1024).toString();
        return null;
    }

    public String getDescription() {
        return description;
    }

    public File getFile() {
        return file;
    }

    public String getMinJDictionaryVersion() {
        if(minJdictionaryVersion == null)
            return "0";
        return minJdictionaryVersion;
    }


    public void setFileName(String fileName) {
        this.fileName = fileName;
    }

    public String getURL() {
        return url;
    }

    public void setSize(String size) {
        this.size = size;
    }
}