﻿<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
	<!--
	 -->
	<xsl:template match="content">
	   <xsl:apply-templates select="*"/>
	</xsl:template>
	<!--
    -->
	<xsl:template match="doc-title">
		<xsl:value-of select="text[(not(@lang) and $lang='en') or @lang = $lang]"/>
	</xsl:template>
	<!--
		Main elements of a documentation.
		section
			IS A SEQUENCE OF:
				title 1
				CONTAINTS ALL:
				  table-of-content ?
				  para *
				  figure *
				  list *
				  n-list *
				  info *
				  warn *
				  HTML element *
				  section *
    -->
	<xsl:template match="section">
		<xsl:apply-templates select="*"/>
	</xsl:template>
	<!--
		Table of content generation.
   -->
	<xsl:template match="table-of-content">
		<div class="toc-section">
			<xsl:if test="@main = 'true'">
				<xsl:attribute name="id">toc</xsl:attribute>
			</xsl:if>
			<xsl:choose>
				<xsl:when test="part">
					<xsl:apply-templates select="part"/>
				</xsl:when>
				<xsl:otherwise>
					<xsl:apply-templates select="." mode="PART"/>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:if test="part">
				<div style="clear: both"><xsl:text> </xsl:text></div>
			</xsl:if>
		</div>
	</xsl:template>
	<!-- -->
	<xsl:template match="table-of-content" mode="PART">
		<xsl:param name="part"/>
		<xsl:choose>
			<xsl:when test="@level-max">
				<xsl:apply-templates select="following-sibling::section[not(title/@toc) or title/@toc = 'true'][1]" mode="TOC">
					<xsl:with-param name="part" select="$part"/>
					<xsl:with-param name="part-mask" select="@part-mask"/>
					<xsl:with-param name="level-max" select="number(@level-max)"/>
				</xsl:apply-templates>
			</xsl:when>
			<xsl:otherwise>
				<xsl:apply-templates select="following-sibling::section[not(title/@toc) or title/@toc = 'true'][1]" mode="TOC">
					<xsl:with-param name="part" select="$part"/>
					<xsl:with-param name="part-mask" select="@part-mask"/>
				</xsl:apply-templates>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	<!--
		table of content in part : a part identifies sections to be generated
		in this part (usfull to split the whole toc in parts).
		Three parameters definie a part : mask (ex. "1.") a section start (from, ex. 4) a section end (to, ex. 6)
		for example: <table-of-content part-mask="1.">
		                <part from="1" to="3"/>
						<part from="4" to="6"/>
					  </table-of-content>
					generates two parts: one with sections 1.1* to 1.3* and one with sections 1.4* to 1.6*
	-->
	<xsl:template match="part">
		<div class="toc-part">
			<xsl:apply-templates select=".." mode="PART">
				<xsl:with-param name="part" select="."/>
			</xsl:apply-templates>
		</div>
	</xsl:template>
	<!--
		Deep first exploration of sections in 2 templates TOC and TOC-LIST
		Bug : generation of empty ul element (<ul/>), with part element. To fix it,
		avoid to start a recurtion when it's not needed, ie when no section or
		subsection is concerned by current part specification given by the 'from'
		and 'two' attribiutes.
		Very complex test: for the momement adding a second transformation to clean
		empty ul elements, see ulCleaner.xsl
 	-->
	<xsl:template match="section" mode="TOC">
		<xsl:param name="part"/>
		<xsl:param name="part-mask"/>
		<xsl:param name="level-max" select="9"/>
		<xsl:variable name="level" select="count(ancestor::section)"/>
		<xsl:if test="$level &lt;= $level-max">
			<ul>
				<xsl:apply-templates select="." mode="TOC-LIST">
					<xsl:with-param name="part" select="$part"/>
					<xsl:with-param name="part-mask" select="$part-mask"/>
					<xsl:with-param name="level-max" select="$level-max"/>
				</xsl:apply-templates>
				<xsl:apply-templates select="following-sibling::section[not(title/@toc) or title/@toc = 'true']" mode="TOC-LIST">
					<xsl:with-param name="part" select="$part"/>
					<xsl:with-param name="part-mask" select="$part-mask"/>
					<xsl:with-param name="level-max" select="$level-max"/>
				</xsl:apply-templates>
			</ul>
		</xsl:if>
	</xsl:template>
	<!-- -->
	<xsl:template match="section" mode="TOC-LIST">
		<xsl:param name="part"/>
		<xsl:param name="part-mask"/>
		<xsl:param name="level-max"/>
		<xsl:variable name="a-num">
			<xsl:apply-templates select="." mode="ABSOLUT-NUM"/>
		</xsl:variable>
		<xsl:variable name="num">
			<xsl:choose>
				<xsl:when test="$part-mask != ''"><xsl:value-of select="substring-after($a-num, $part-mask)"/></xsl:when>
				<xsl:otherwise><xsl:value-of select="$a-num"/></xsl:otherwise>
			</xsl:choose>
		</xsl:variable>
		<xsl:variable name="part-num">
			<xsl:choose>
				<xsl:when test="contains($num, '.')"><xsl:value-of select="number(substring-before($num, '.'))"/></xsl:when>
				<xsl:otherwise><xsl:value-of select="number($num)"/></xsl:otherwise>
			</xsl:choose>
		</xsl:variable>
		<xsl:choose>
			<xsl:when test="not($part) or ($part-num &gt;= number($part/@from) and $part-num &lt;= number($part/@to))">
				<li>
					<xsl:apply-templates select="title">
						<xsl:with-param name="toc" select="'true'"/>
					</xsl:apply-templates>
					<xsl:apply-templates select="child::section[not(title/@toc) or title/@toc = 'true'][1]" mode="TOC">
						<xsl:with-param name="part" select="$part"/>
						<xsl:with-param name="part-mask" select="$part-mask"/>
						<xsl:with-param name="level-max" select="$level-max"/>
					</xsl:apply-templates>
				</li>
			</xsl:when>
			<xsl:otherwise>
				<xsl:apply-templates select="child::section[not(title/@toc) or title/@toc = 'true'][1]" mode="TOC">
					<xsl:with-param name="part" select="$part"/>
					<xsl:with-param name="part-mask" select="$part-mask"/>
					<xsl:with-param name="level-max" select="$level-max"/>
				</xsl:apply-templates>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	<!--
		Manages titles: tiltes in the table of content or titles as header of a document section.
    -->
	<xsl:template match="title">
		<xsl:param name="toc" select="'false'"/> <!-- true for table of content generation, false otherwise -->
		<xsl:variable name="level" select="count(ancestor::section)"/> <!-- the title level corresponds to its absolute section level : number of parent section -->
		<xsl:variable name="num"> <!-- hierarchical numbering of section -->
			<xsl:apply-templates select="parent::section" mode="NUM"/>
		</xsl:variable>
		<xsl:variable name="id"> <!-- the attribute id of the section or a generated id (based on the title's number) -->
			<xsl:choose>
				<xsl:when test="parent::section/@id"><xsl:value-of select="parent::section/@id"/></xsl:when>
				<xsl:otherwise><xsl:value-of select="concat('_', translate($num, '.', '_'))"/></xsl:otherwise>
			</xsl:choose>
		</xsl:variable>
		<xsl:choose>
			<xsl:when test="$toc = 'true'"> <!-- table of content part -->
				<xsl:if test="not(@numbering) or @numbering = 'true'"><xsl:value-of select="concat($num, ' ')"/></xsl:if>
				<a href="{concat('#', $id)}" class="dashed"><xsl:apply-templates select="text[(not(@lang) and $lang='en') or @lang = $lang or @lang = 'multilang']"/></a>
			</xsl:when>
			<xsl:when test="$toc = 'false'"> <!-- content part -->
				<!--xsl:message>LEVEL = <xsl:value-of select="$level"/></xsl:message-->
				<!--xsl:if test="$level > 1">
					<xsl:if test="$level &lt;= 4">
						<div><br /></div>
					</xsl:if>
					<xsl:variable name="back"><xsl:call-template name="label"><xsl:with-param name="key" select="'back.toc'"/></xsl:call-template></xsl:variable>
					<div class="top"><a href="#toc" title="{$back}"><span><xsl:value-of select="$back"/></span></a></div>
				</xsl:if-->
				<xsl:element name="{concat('h', $level)}">
					<xsl:attribute name="id"><xsl:value-of select="$id"/></xsl:attribute>
					<xsl:apply-templates select="." mode="TEXT">
						<xsl:with-param name="num" select="$num"/>					
					</xsl:apply-templates>
				</xsl:element>
			</xsl:when>
		</xsl:choose>
	</xsl:template>
	<!-- Gets the text of a title : its number following by its text of the current language -->
	<xsl:template match="title" mode="TEXT">
		<xsl:param name="num"/>
		<xsl:if test="not(@numbering) or @numbering = 'true'"><xsl:value-of select="concat($num, ' ')"/></xsl:if>
		<xsl:apply-templates select="text[(not(@lang) and $lang='en') or @lang = $lang or @lang = 'multilang']"/>
	</xsl:template>
	<!--
		Paragraphe generation.
    -->
	<xsl:template match="para">
		<p>
			<xsl:apply-templates select="@*" mode="COPY"/>
			<xsl:apply-templates select="text[(not(@lang) and $lang='en') or @lang = $lang or @lang = 'multilang']"/>
		</p>
	</xsl:template>
	<!--
		pre generation.
    -->
	<xsl:template match="pre">
		<pre class="code"><xsl:apply-templates select="text[(not(@lang) and $lang='en') or @lang = $lang or @lang = 'multilang']"/></pre>
	</xsl:template>
	<!--
		Figure management : it's a spacial paragraphe containing images.
		Like text element, image element depends on language.
	-->
	<xsl:template match="figure">
		<p class="figure">
			<xsl:apply-templates select="@*" mode="COPY"/>
			<xsl:apply-templates select="image[(not(@lang) and $lang = 'en') or @lang = $lang or @lang = 'multilang']|space"/>
			<xsl:apply-templates select="caption"/>
		</p>
	</xsl:template>
	<xsl:template match="image">
		<img width="{@width}" height="{@height}" alt="{@title}" title="{@title}">
			<xsl:attribute name="src"><xsl:value-of select="concat($images-base, @src)"/></xsl:attribute>
		</img>
	</xsl:template>
	<xsl:template match="caption">
		<br/>
		<xsl:apply-templates select="text[(not(@lang) and $lang = 'en') or @lang = $lang or @lang = 'multilang']"/>
	</xsl:template>
	<xsl:template match="image" mode="COPY">
		<xsl:apply-templates select="."/>
	</xsl:template>
	<!--
		Listes management: simple list or numbered list
    -->
	<xsl:template match="list">
		<ul>
			<xsl:apply-templates select="@*" mode="COPY"/>
			<xsl:apply-templates select="*"/>
		</ul>
	</xsl:template>
	<xsl:template match="n-list">
		<ol>
			<xsl:apply-templates select="@*" mode="COPY"/>
			<xsl:apply-templates select="*"/>
		</ol>
	</xsl:template>
	<xsl:template match="item">
		<li>
			<xsl:apply-templates select="text[(not(@lang) and $lang='en') or @lang = $lang or @lang = 'multilang']|*[not(name() = 'text')]"/>
		</li>
	</xsl:template>
	<!--
		Special section: INFORMATION and WARNING
    -->
	<xsl:template match="info">
		<div class="info">
			<xsl:apply-templates select="@*" mode="COPY"/>
			<span>INFO:</span>
			<xsl:apply-templates select="*"/>
		</div>
	</xsl:template>
	<xsl:template match="warn">
		<div class="warn">
			<xsl:apply-templates select="@*" mode="COPY"/>
			<span>WARNING:</span>
			<xsl:apply-templates select="*"/>
		</div>
	</xsl:template>
	<!--
		A text element is required when the content depnds on a language
		It must be selected by testing if it corresponds to the current language, like this :
			select="text[(not(@lang) and $lang='en') or @lang = $lang  or @lang = 'multilang']"
		The tamplate copies the text content.
    -->
	<xsl:template match="text">
		<xsl:apply-templates select="node()" mode="COPY"/>
	</xsl:template>
	<!--
	-->
	<xsl:template match="space">
		<xsl:text> </xsl:text>
	</xsl:template>
	<!--
		Generic template for generation inside a text element
    -->
	<xsl:template match="@*|node()" mode="COPY">
		<xsl:copy>
			<xsl:apply-templates select="@*|node()" mode="COPY"/>
		</xsl:copy>
	</xsl:template>
	<!--
		Management of link (a element of HTML)
    -->
	<xsl:template match="link" mode="COPY">
		<xsl:choose>
			<xsl:when test="@to"><a href="{concat('#', @to)}" class="dashed" title="{@title}"><xsl:apply-templates select="node()" mode="COPY"/></a></xsl:when>
			<xsl:otherwise><a href="{@href}" class="dashed" title="{@title}"><xsl:apply-templates select="node()" mode="COPY"/></a></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	<!--
		Default management of HTML element outside a text element : copy it !
    -->
	<xsl:template match="br">
		<xsl:apply-templates select="." mode="COPY"/>
	</xsl:template>
	<!--
		Builds a hieracthical title number by a recurtion on parent section. The sections in the hierarchy must have numbering
		directive : numbering = true (true is the default value)
    -->
	<xsl:template match="section" mode="NUM">
		<xsl:param name="separator" select="'.'"/>
		<xsl:apply-templates select="parent::section[not(title/@numbering) or title/@numbering = 'true']" mode="NUM">
			<xsl:with-param name="separator" select="$separator"/>
		</xsl:apply-templates>
		<xsl:if test="parent::section[not(title/@numbering) or title/@numbering = 'true']"><xsl:value-of select="$separator"/></xsl:if>
		<xsl:value-of select="count(preceding-sibling::section[not(title/@numbering) or title/@numbering = 'true']) + 1"/>
	</xsl:template>
	<!-- 
		Builds a hierarchical absolute number by recurtion on parent section (doesn't take into account the numbering attribute)
	-->
	<xsl:template match="section" mode="ABSOLUT-NUM">
		<xsl:param name="separator" select="'.'"/>
		<xsl:apply-templates select="parent::section" mode="ABSOLUT-NUM">
			<xsl:with-param name="separator" select="$separator"/>
		</xsl:apply-templates>
		<xsl:if test="parent::section"><xsl:value-of select="$separator"/></xsl:if>
		<xsl:value-of select="count(preceding-sibling::section) + 1"/>
	</xsl:template>
	<!--
	-->
	<xsl:template name="label">
		<xsl:param name="key"/>
		<xsl:choose>
			<xsl:when test="$lang = 'en'">
				<xsl:choose>
					<xsl:when test="$key = 'back.toc'">Back to the table of content.</xsl:when>
					<xsl:when test="$key = 'go.to.top'">Got to the top of the page.</xsl:when>
					<xsl:otherwise>Unknown label!</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<xsl:when test="$lang = 'fr'">
				<xsl:choose>
					<xsl:when test="$key = 'back.toc'">Retour à la table des matières.</xsl:when>
					<xsl:when test="$key = 'go.to.top'">Aller en haut de la page.</xsl:when>
					<xsl:otherwise>Label inconnu !</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
		</xsl:choose>
	</xsl:template>
	<!--
	-->
</xsl:stylesheet>
