/*
 * @(#)WindowsFileChooserUI.java	1.14 98/07/17
 *
 * Copyright 1997, 1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package com.sun.java.swing.plaf.mac;

import javax.swing.*;
import javax.swing.filechooser.*;
import javax.swing.event.*;
import javax.swing.plaf.*;
import javax.swing.plaf.basic.*;
import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.io.File;
import java.io.IOException;
import java.util.*;

/**
 * Mac L&F implementation of a FileChooser.
 *
 * @version 1.14 07/17/98
 * @author Tom Santos
 */
public class MacFileChooserUI extends BasicFileChooserUI {
    private JFileChooser filechooser;
    private JComboBox comboBox;
    private DefaultComboBoxModel comboBoxModel;
    //private DirectoryComboBoxModel comboBoxModel;
    private JLabel volumeLabel;
    private JList list;
    private JButton ejectButton;
    private JButton desktopButton;
    private JButton approveButton;
    private JButton cancelButton;

    private JPanel leftPanel;
    private JTextField editField;
    private JPanel editPanel;

    private FileSystemView oldFileSystemView;

    private boolean isTemporary = false;

    private Action directoryComboBoxAction = new DirectoryComboBoxAction();

    private FilterComboBoxModel filterComboBoxModel;

    private JTextField filenameTextField;


    private static int PREF_WIDTH = 356;
    private static int PREF_HEIGHT = 178;
    private static Dimension PREF_SIZE = new Dimension(PREF_WIDTH, PREF_HEIGHT);

    private static int MIN_WIDTH = 356;
    private static int MIN_HEIGHT = 178;
    private static Dimension MIN_SIZE = new Dimension(MIN_WIDTH, MIN_HEIGHT);

    private static int LIST_MIN_WIDTH = 400;
    private static int LIST_MIN_HEIGHT = 100;
    private static Dimension LIST_MIN_SIZE = new Dimension(LIST_MIN_WIDTH, LIST_MIN_HEIGHT);

    private int    lookInLabelMnemonic = 0;
    private String lookInLabelText = null;

    private int    fileNameLabelMnemonic = 0;
    private String fileNameLabelText = null;

    private int    filesOfTypeLabelMnemonic = 0;
    private String filesOfTypeLabelText = null;

    private String upFolderToolTipText = null;
    private String upFolderAccessibleName = null;

    private String homeFolderToolTipText = null;
    private String homeFolderAccessibleName = null;

    private String newFolderToolTipText = null;
    private String newFolderAccessibleName = null;

    private String listViewButtonToolTipText = null;
    private String listViewButtonAccessibleName = null;

    private String detailsViewButtonToolTipText = null;
    private String detailsViewButtonAccessibleName = null;

    //
    // ComponentUI Interface Implementation methods
    //
    public static ComponentUI createUI(JComponent c) {
        return new MacFileChooserUI((JFileChooser) c);
    }

    public MacFileChooserUI(JFileChooser filechooser) {
        super(filechooser);
        this.filechooser = filechooser;
    }

    public void installUI(JComponent c) {
        super.installUI(c);
        c.setBackground( Color.white );
        oldFileSystemView = filechooser.getFileSystemView();
        filechooser.setFileSystemView( new MacFileSystemView() );
    }

    public void uninstallComponents(JFileChooser fc) {
        fc.removeAll();
    }

    public void installComponents(JFileChooser fc) {
        // Layout management
        filechooser.setLayout( new BorderLayout() );

        class WhitePanel extends JPanel {
            public WhitePanel() { /*setBackground( Color.white );*/}
        }

        leftPanel = new WhitePanel();
        JPanel rightPanel = new WhitePanel();

        leftPanel.setLayout( new BorderLayout() );
        BoxLayout rightLayout = new BoxLayout( rightPanel, BoxLayout.Y_AXIS );
        rightPanel.setLayout( rightLayout );

        filechooser.add( BorderLayout.CENTER, leftPanel );
        filechooser.add( BorderLayout.EAST, rightPanel );

        // Setting up the combo box of directories
        comboBox = new JComboBox();
        comboBoxModel = new DefaultComboBoxModel();
        comboBox.setModel( comboBoxModel );
        comboBox.setRenderer( createDirectoryComboBoxRenderer( filechooser ) );
        comboBox.addActionListener( new DirectoryComboBoxAction() );
        JPanel comboPanel = new WhitePanel();
        comboPanel.add( comboBox );
        leftPanel.add( BorderLayout.NORTH, comboPanel );

        // Setting up the list of files
        list = new JList( getModel() );
        list.setCellRenderer( new FileRenderer() );
        list.addListSelectionListener( createListSelectionListener( filechooser ) );
        list.addMouseListener( createDoubleClickListener( filechooser, list ) );
        JScrollPane scroller = new JScrollPane( list );
        leftPanel.add( BorderLayout.CENTER, scroller );

        // Setting up the edit field
        editField = new JTextField();
        editPanel = new JPanel();
        if ( filechooser.getDialogType() != JFileChooser.OPEN_DIALOG ) {
            addEditField();
        }

        // Setting up the current volume label
        volumeLabel = new JLabel( "Macintosh HD" );
        volumeLabel.setHorizontalAlignment( JLabel.CENTER );
        MouseListener labelListener = new MouseAdapter() {
            public void mousePressed( MouseEvent e ) {
                goUpOneDirectory();
            }
        };
        volumeLabel.addMouseListener( labelListener );
        rightPanel.add( volumeLabel );

        // Setting up the eject button
        ejectButton = new JButton( "Eject" );
        ejectButton.setEnabled( false );
        JPanel ejectPanel = new WhitePanel();
        ejectPanel.add( ejectButton );
        rightPanel.add( ejectPanel );

        // Setting up the desktop button
        desktopButton = new JButton( "Desktop" );
        desktopButton.setEnabled( false );
        JPanel desktopPanel = new WhitePanel();
        desktopPanel.add( desktopButton );
        rightPanel.add( desktopPanel );

        // Setting up the separator
        JPanel separator = new WhitePanel() {
            public void paintComponent( Graphics g ) {
                super.paintComponent( g );
                int buttonWidth = desktopButton.getWidth();
                int width = getWidth();
                int center = width / 2;
                g.setColor( Color.black );
                g.drawLine( center - buttonWidth / 2, getHeight() / 2,
                            center + buttonWidth / 2, getHeight() / 2 );
            }
        };
        Dimension prefSize = desktopButton.getPreferredSize();
        prefSize.height = prefSize.height / 2;
        separator.setPreferredSize( prefSize );
        rightPanel.add( separator );

        // Setting up the approve button
        approveButton = new JButton( getApproveButtonText( filechooser ) );
        approveButton.setMnemonic( getApproveButtonMnemonic( filechooser ) );
        approveButton.setToolTipText( getApproveButtonToolTipText( filechooser ) );
        approveButton.addActionListener( getApproveSelectionAction() );
        JPanel approvePanel = new WhitePanel();
        approvePanel.add( approveButton );
        rightPanel.add( approvePanel );

        // Setting up the cancel button
        cancelButton = new JButton( "Cancel" );
        cancelButton.addActionListener( getCancelSelectionAction() );
        JPanel cancelPanel = new WhitePanel();
        cancelPanel.add( cancelButton );
        rightPanel.add( cancelPanel );
    }

    protected void installIcons(JFileChooser fc) {
	directoryIcon    = UIManager.getIcon("FileView.directoryIcon");
	fileIcon         = UIManager.getIcon("FileView.fileIcon");
	computerIcon     = UIManager.getIcon("FileView.computerIcon");
	hardDriveIcon    = UIManager.getIcon("FileView.hardDriveIcon");
	floppyDriveIcon  = UIManager.getIcon("FileView.floppyDriveIcon");
    }

    void initializeComboBox() {
        comboBoxModel = new DefaultComboBoxModel();

        File dir = filechooser.getCurrentDirectory();

        if ( dir.toString().equals( "/" ) ) {
            Thread.dumpStack();
        }

        isTemporary = true;
        buildParents( dir );
        isTemporary = false;
        comboBox.setModel( comboBoxModel );
    }

    void buildParents( File dir ) {
        comboBoxModel.addElement( dir );
        if ( dir.getParent() != null && !dir.getParent().equals( "/" ) ) {
            buildParents( new File( dir.getParent() ) );
        }
        else {
            volumeLabel.setText( dir.toString() );
        }
    }

    void initializeList() {
        //getModel().removeAllElements();
        File dir = filechooser.getCurrentDirectory();
        String[] files = dir.list();
        for ( int i = 0; i < files.length; ++i ) {
            //listModel.addElement( files[i] );
        }
    }

    void directoryChanged( File directory ) {
        filechooser.setCurrentDirectory( directory );
    }

    void fileChanged( File file ) {
    }

    void goUpOneDirectory() {
        File dir = filechooser.getCurrentDirectory();
        if ( dir.getParent() != null &&
             !dir.getParent().equals( "/" ) ) {
            FileSystemView fileSystem = filechooser.getFileSystemView();
            File parent = new File( dir.getParent() );
            filechooser.setCurrentDirectory( parent );
        }
    }

    protected void installStrings(JFileChooser fc) {
        super.installStrings(fc);

        lookInLabelMnemonic = UIManager.getInt("FileChooser.lookInLabelMnemonic"); 
        lookInLabelText = UIManager.getString("FileChooser.lookInLabelText");

        fileNameLabelMnemonic = UIManager.getInt("FileChooser.fileNameLabelMnemonic");  
        fileNameLabelText = UIManager.getString("FileChooser.fileNameLabelText"); 

        filesOfTypeLabelMnemonic = UIManager.getInt("FileChooser.filesOfTypeLabelMnemonic"); filesOfTypeLabelText = UIManager.getString("FileChooser.filesOfTypeLabelText"); 

        upFolderToolTipText =  UIManager.getString("FileChooser.upFolderToolTipText");
        upFolderAccessibleName = UIManager.getString("FileChooser.upFolderAccessibleName"); 

        homeFolderToolTipText =  UIManager.getString("FileChooser.homeFolderToolTipText");
        homeFolderAccessibleName = UIManager.getString("FileChooser.homeFolderAccessibleName"); 

        newFolderToolTipText = UIManager.getString("FileChooser.newFolderToolTipText");
        newFolderAccessibleName = UIManager.getString("FileChooser.newFolderAccessibleName"); 

        listViewButtonToolTipText = UIManager.getString("FileChooser.listViewButtonToolTipText"); 
        listViewButtonAccessibleName = UIManager.getString("FileChooser.listViewButtonAccessibleName"); 

        detailsViewButtonToolTipText = UIManager.getString("FileChooser.detailsViewButtonToolTipText"); 
        detailsViewButtonAccessibleName = UIManager.getString("FileChooser.detailsViewButtonAccessibleName"); 
    }

    protected JPanel createList(JFileChooser fc) {
        JPanel p = new JPanel(new BorderLayout());
        list = new JList();
        list.setCellRenderer(new FileRenderer());
        list.setModel(getModel());
        list.addListSelectionListener(createListSelectionListener(fc));
        list.addMouseListener(createDoubleClickListener(fc, list));
        JScrollPane scrollpane = new JScrollPane(list);
        scrollpane.setBorder(BorderFactory.createLoweredBevelBorder());
        p.add(scrollpane, BorderLayout.CENTER);
        return p;
    }

    protected class FileRenderer extends DefaultListCellRenderer {

        public Component getListCellRendererComponent(JList list, Object value,
                                                      int index, boolean isSelected,
                                                      boolean cellHasFocus) {

            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            File file = (File) value;
            String fileName = getFileChooser().getName(file);
            setText(fileName);

            Icon icon = getFileChooser().getIcon(file);
            setIcon(icon);

            return this;
        }
    }


    public void uninstallUI(JComponent c) {
        // Remove listeners
        c.removePropertyChangeListener(filterComboBoxModel);
        filechooser.setFileSystemView( oldFileSystemView );

        super.uninstallUI(c);
    }

    public Dimension getPreferredSize(JComponent c) {
        return PREF_SIZE;
    }

    public Dimension getMinimumSize(JComponent c) {
        return MIN_SIZE;
    }

    public Dimension getMaximumSize(JComponent c) {
        return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
    }

    void addEditField() {
        leftPanel.add( BorderLayout.SOUTH, editField );
    }

    void removeEditField() {
        leftPanel.remove( editField );
    }

    /*
     * Listen for filechooser property changes, such as
     * the selected file changing, or the type of the dialog changing.
     */
    public PropertyChangeListener createPropertyChangeListener(JFileChooser fc) {
        return new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent e) {
                String prop = e.getPropertyName();
                if ( prop.equals(JFileChooser.SELECTED_FILE_CHANGED_PROPERTY) ) {
                    File f = (File) e.getNewValue();
                    if ( f != null ) {
                        setFileName(getFileChooser().getName(f));
                        if ( getModel().contains(f) ) {
                            list.setSelectedIndex(getModel().indexOf(e.getNewValue()));
                            list.ensureIndexIsVisible(list.getSelectedIndex());
                        }
                    }
                }
                else if ( prop.equals(JFileChooser.DIALOG_TYPE_CHANGED_PROPERTY ) ) {
                    int oldValue = ((Integer)e.getOldValue()).intValue();
                    int newValue = ((Integer)e.getNewValue()).intValue();

                    if ( newValue == JFileChooser.OPEN_DIALOG ) {
                        removeEditField();
                    }
                    else if ( oldValue == JFileChooser.OPEN_DIALOG ) {
                        addEditField();
                    }
                }
                else if ( prop.equals(JFileChooser.DIRECTORY_CHANGED_PROPERTY) ) {
                    clearIconCache();
                    list.clearSelection();
                    initializeComboBox();
/*
                    File currentDirectory = getFileChooser().getCurrentDirectory();
                     if ( currentDirectory != null ) {
                         comboBox.addItem(currentDirectory);
                         // Enable the newFolder action if the current directory
                         // is writable.
                         // PENDING(jeff) - broken - fix
                         getNewFolderAction().setEnabled(currentDirectory.canWrite());
                     }
                       */
                }
                else if ( prop.equals(JFileChooser.FILE_SELECTION_MODE_CHANGED_PROPERTY) ) {
                    clearIconCache();
                    list.clearSelection();
                }
                else if ( prop == JFileChooser.ACCESSORY_CHANGED_PROPERTY ) {
                    if ( getAccessoryPanel() != null ) {
                        if ( e.getOldValue() != null ) {
                            getAccessoryPanel().remove((JComponent) e.getOldValue());
                        }
                        JComponent accessory = (JComponent) e.getNewValue();
                        if ( accessory != null ) {
                            getAccessoryPanel().add(accessory, BorderLayout.CENTER);
                        }
                    }
                }
                else if ( prop == JFileChooser.APPROVE_BUTTON_TEXT_CHANGED_PROPERTY ||
                          prop == JFileChooser.DIALOG_TYPE_CHANGED_PROPERTY ) {
                    JFileChooser chooser = getFileChooser();
                    approveButton.setText(getApproveButtonText(chooser));
                    approveButton.setToolTipText(getApproveButtonToolTipText(chooser));
                    approveButton.setMnemonic(getApproveButtonMnemonic(chooser));
                }
                else if ( prop.equals(JFileChooser.APPROVE_BUTTON_MNEMONIC_CHANGED_PROPERTY) ) {
                    approveButton.setMnemonic(getApproveButtonMnemonic(getFileChooser()));
                }
            }
        };
    }

    public void ensureFileIsVisible(JFileChooser fc, File f) {
        if ( getModel().contains(f) ) {
            list.ensureIndexIsVisible(getModel().indexOf(f));
        }
    }

    public void rescanCurrentDirectory(JFileChooser fc) {
        getModel().invalidateFileCache();
        getModel().validateFileCache();
    }

    public String getFileName() {
        if ( filechooser.getDialogType() != JFileChooser.OPEN_DIALOG ) {
	    return editField.getText();
	}
	else {
            if ( list.getSelectedValue() != null ) {
                return list.getSelectedValue().toString();
	    }
	    else {
	        return null;
	    }
	}
    }
  
    public void setFileName(String filename) {
        if ( filenameTextField != null ) {
            filenameTextField.setText(filename);
        }
    }

    public String getDirectoryName() {
        // PENDING(jeff) - get the name from the directory combobox
        return null;
    }

    public void setDirectoryName(String dirname) {
        // PENDING(jeff) - set the name in the directory combobox
    }

    protected DirectoryComboBoxRenderer createDirectoryComboBoxRenderer(JFileChooser fc) {
        return new DirectoryComboBoxRenderer();
    }

    //
    // Renderer for DirectoryComboBox
    //
    class DirectoryComboBoxRenderer extends DefaultListCellRenderer {
        public Component getListCellRendererComponent(JList list, Object value,
                                                      int index, boolean isSelected,
                                                      boolean cellHasFocus) {

            super.getListCellRendererComponent(list, value, index,
                                               isSelected, cellHasFocus);
            File directory = (File) value;
            if ( directory == null ) {
                setText("");
                return this;
            }
            String fileName = getFileChooser().getName(directory);
            setText(fileName);

            // REMIND(jeff) - pay attention to parent heirarchy and
            // indent accordingly
            Icon icon = getFileChooser().getIcon(directory);
            setIcon(icon);

            return this;
        }
    }



    //
    // Renderer for Types ComboBox
    //
    protected FilterComboBoxRenderer createFilterComboBoxRenderer() {
        return new FilterComboBoxRenderer();
    }

    /**
     * Render different type sizes and styles.
     */
    public class FilterComboBoxRenderer extends DefaultListCellRenderer {
        public Component getListCellRendererComponent(JList list,
                                                      Object value, int index, boolean isSelected,
                                                      boolean cellHasFocus) {

            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);

            FileFilter filter = (FileFilter) value;
            if ( filter != null ) {
                setText(filter.getDescription());
            }

            return this;
        }
    }

    //
    // DataModel for Types Comboxbox
    //
    protected FilterComboBoxModel createFilterComboBoxModel() {
        return new FilterComboBoxModel();
    }

    /**
     * Data model for a type-face selection combo-box.
     */
    protected class FilterComboBoxModel extends AbstractListModel implements ComboBoxModel, PropertyChangeListener {
        protected FileFilter[] filters;
        protected FilterComboBoxModel() {
            super();
            filters = getFileChooser().getChoosableFileFilters();
        }

        public void propertyChange(PropertyChangeEvent e) {
            String prop = e.getPropertyName();
            if ( prop == JFileChooser.CHOOSABLE_FILE_FILTER_CHANGED_PROPERTY ) {
                filters = (FileFilter[]) e.getNewValue();
                fireContentsChanged(this, -1, -1);
            }
        }

        public void setSelectedItem(Object filter) {
            if ( filter != null ) {
                getFileChooser().setFileFilter((FileFilter) filter);
                fireContentsChanged(this, -1, -1);
            }
        }

        public Object getSelectedItem() {
            // Ensure that the current filter is in the list.
            // NOTE: we shouldnt' have to do this, since JFileChooser adds
            // the filter to the choosable filters list when the filter
            // is set. Lets be paranoid just in case someone overrides
            // setFileFilter in JFileChooser.
            FileFilter currentFilter = getFileChooser().getFileFilter();
            boolean found = false;
            if ( currentFilter != null ) {
                for ( int i=0; i < filters.length; i++ ) {
                    if ( filters[i] == currentFilter ) {
                        found = true;
                    }
                }
                if ( found == false ) {
                    getFileChooser().addChoosableFileFilter(currentFilter);
                }
            }
            return getFileChooser().getFileFilter();
        }

        public int getSize() {
            if ( filters != null ) {
                return filters.length;
            }
            else {
                return 0;
            }
        }

        public Object getElementAt(int index) {
            if ( index > getSize() - 1 ) {
                // This shouldn't happen. Try to recover gracefully.
                return getFileChooser().getFileFilter();
            }
            if ( filters != null ) {
                return filters[index];
            }
            else {
                return null;
            }
        }
    }

    public void valueChanged(ListSelectionEvent e) {
        File f = getFileChooser().getSelectedFile();
        if ( !e.getValueIsAdjusting() && f != null && !getFileChooser().isTraversable(f) ) {
            setFileName(getFileChooser().getName(f));
        }
    }

    /**
     * Acts when DirectoryComboBox has changed the selected item.
     */
    protected class DirectoryComboBoxAction extends AbstractAction {
        protected DirectoryComboBoxAction() {
            super("DirectoryComboBoxAction");
        }

        public void actionPerformed(ActionEvent e) {
            getFileChooser().setCurrentDirectory((File) comboBox.getSelectedItem());
        }
    }

    protected JButton getApproveButton(JFileChooser fc) {
        return approveButton;
    }

}

