(* ==================================================================== *)
(*									*)
(*  State Module for the Gardens Point Component Pascal Compiler.	*)
(*	Copyright (c) John Gough 1999, 2000.				*)
(*									*)
(*  Note that since this module is likely to be imported by most other  *)
(*  modules, it is important to ensure that it does not import others,  *)
(*  to avoid import cycles. 						*)
(*									*)
(* ==================================================================== *)

MODULE CompState;

  IMPORT 
	GPCPcopyright,
	RTS,
	Error,
	GPText,
	Symbols,
	IdDesc,
	Console, 
	CPascalS,
	NameHash,
	FileNames,
	CPascalErrors;

  CONST	prefix     = "#gpcp: ";
	millis     = "mSec";

  CONST netV1_0* = 0;
        netV1_1* = 1;
        netV2_0* = 2;

(* ==================================================================== *)
(*		     State Variables of this compilation		*)
(* ==================================================================== *)

  VAR
    ntvObj* : Symbols.Type;     (* native Object type          	*)
    ntvStr* : Symbols.Type;     (* native String type          	*)
    ntvExc* : Symbols.Type;     (* native Exceptions type       *)
    ntvTyp* : Symbols.Type;     (* native System.Type type      *)
    ntvEvt* : Symbols.Type;     (* native MulticastDelegate     *)
    rtsXHR* : Symbols.Type;     (* native XHR type descriptor   *)
    ntvVal* : Symbols.Type;     (* native ValueType type        *)

    objId*  : Symbols.Idnt;
    strId*  : Symbols.Idnt;
    excId*  : Symbols.Idnt;
    clsId*  : Symbols.Idnt;
    xhrId*  : IdDesc.FldId;     (* descriptor of RTS.XHR.prev   *)
    rtsBlk* : IdDesc.BlkId;
    prgArg* : IdDesc.BlkId;
    argLst* : IdDesc.VarId;     (* descriptor of RTS.argList    *)
    srcBkt* : INTEGER;          (* hashtable bucket of "src"    *)

  VAR
    modNam*   : FileNames.NameString;    (* name of the _MODULE_        *)
    basNam-,                             (* base name of source _FILE_  *)
    srcNam-,                             (* name of the source file     *)
    lstNam-   : FileNames.NameString;    (* name of the listing file    *)

    target-   : ARRAY 4 OF CHAR;

    binDir-,                             (* PE-file directory .NET only *)
    symDir-   : FileNames.NameString;    (* Symbol file directory       *)

    strict-,
    special-,
    warning-,
    verbose-,
    extras-,
    unsafe-,
    doStats-,
    doHelp-,
    ovfCheck-,
    debug-,
    doneHelp,
    doVersion-,
    doneVersion,
    doSym-,
    doAsm-,
    doJsmn-,
    doIlasm-,
    doCode-,
    system-    : BOOLEAN;
    netRel-,
    listLevel-,
    hashSize-  : INTEGER;

    thisMod-   : IdDesc.BlkId;           (* Desc. of compiling module.  *)
    sysMod-    : IdDesc.BlkId;           (* Desc. of compiling module.  *)

    impSeq*    : Symbols.ScpSeq;

    totalS*    : LONGINT;
    parseS*    : LONGINT;
    parseE*    : LONGINT;
    attrib*    : LONGINT;
    symEnd*    : LONGINT;
    asmEnd*    : LONGINT;
    totalE*    : LONGINT;
    import1*   : LONGINT;
    import2*   : LONGINT;

    impMax*    : INTEGER;

  VAR
    expectedNet : BOOLEAN;         (* A .NET specific option was parsed *)
    expectedJvm : BOOLEAN;         (* A JVM specific option was parsed  *)

(* ==================================================================== *)
(*				Utilities				*)
(* ==================================================================== *)

    PROCEDURE targetIsNET*() : BOOLEAN;
    BEGIN
      RETURN target = "net";
    END targetIsNET;

    PROCEDURE targetIsJVM*() : BOOLEAN;
    BEGIN
      RETURN target = "jvm";
    END targetIsJVM;

    PROCEDURE Message*(IN mss : ARRAY OF CHAR);
    BEGIN
      Console.WriteString(prefix);
      Console.WriteString(mss);
      Console.WriteLn;
    END Message;

    PROCEDURE PrintLn*(IN mss : ARRAY OF CHAR);
    BEGIN
      Console.WriteString(mss);
      Console.WriteLn;
    END PrintLn;

    PROCEDURE ErrMesg*(IN mss : ARRAY OF CHAR);
    BEGIN
      Console.WriteString(prefix);
      Error.WriteString(mss);
      Error.WriteLn;
    END ErrMesg;

    PROCEDURE Abort*(IN mss : ARRAY OF CHAR);
    BEGIN
      ErrMesg(mss); ASSERT(FALSE);
    END Abort;

    PROCEDURE isForeign*() : BOOLEAN;
    BEGIN
      RETURN 
	(Symbols.rtsMd IN thisMod.xAttr) OR
	(Symbols.frnMd IN thisMod.xAttr);
    END isForeign;

    PROCEDURE TimeMsg(IN mss : ARRAY OF CHAR; tim : LONGINT);
    BEGIN
      IF (tim < 0) OR (tim >= totalS) THEN tim := 0 END;
      Console.WriteString(prefix);
      Console.WriteString(mss);
      Console.WriteInt(SHORT(tim), 5);
      Console.WriteString(millis);
      Console.WriteLn;
    END TimeMsg;

(* ==================================================================== *)

    PROCEDURE Usage;
    BEGIN
      PrintLn("gardens point component pascal: " + GPCPcopyright.verStr);
      Message("Usage from the command line ...");
      IF RTS.defaultTarget = "net" THEN
PrintLn("       $ gpcp [cp-options] file {file}");
PrintLn("# CP Options ...");
PrintLn("       /bindir=XXX  ==> Place binary files in directory XXX");
PrintLn("       /copyright   ==> Display copyright notice");
PrintLn("       /debug       ==> Generate debugging information (default)");
PrintLn("       /nodebug     ==> Give up debugging for maximum speed");
PrintLn("       /dostats     ==> Give a statistical summary");
PrintLn("       /extras      ==> Enable experimental compiler features");
PrintLn("       /help        ==> Write out this usage message");
PrintLn("       /hsize=NNN   ==> Set hashtable size >= NNN (0 .. 65000)");
PrintLn("       /ilasm       ==> Force compilation via ILASM");
PrintLn("       /list        ==> (default) Create *.lst file if errors");
PrintLn("       /list+       ==> Unconditionally create *.lst file");
PrintLn("       /list-       ==> Don't create error *.lst file");
PrintLn("       /noasm       ==> Don't create asm (or object) files");
PrintLn("       /nocode      ==> Don't create any object files");
PrintLn("       /nocheck     ==> Don't perform arithmetic overflow checks");
PrintLn("       /nosym       ==> Don't create *.sym (or asm or object) files");
PrintLn("       /strict      ==> Disallow non-standard constructs");
PrintLn("       /special     ==> Compile dummy symbol file");
PrintLn("       /symdir=XXX  ==> Place symbol files in directory XXX");
PrintLn("       /target=XXX  ==> Emit (jvm|net|dcf) assembly");
PrintLn("       /unsafe      ==> Allow unsafe code generation");
PrintLn("       /vX.X        ==> (v1.0 | v1.1 | v2.0) CLR target version");
PrintLn("       /verbose     ==> Emit verbose diagnostics");
PrintLn("       /version     ==> Write out version number");
PrintLn("       /vserror     ==> Print error messages in Visual Studio format");
PrintLn("       /warn-       ==> Don't emit warnings");
PrintLn("       /nowarn      ==> Don't emit warnings");
PrintLn("       /whidbey     ==> Target code for Whidbey Beta release");
PrintLn("       /xmlerror    ==> Print error messages in XML format");
PrintLn('	Unix-style options: "-option" are recognized also');
      ELSE
        IF RTS.defaultTarget = "jvm" THEN
PrintLn("       $ cprun gpcp [cp-options] file {file}, OR");
PrintLn("       $ java [java-options] CP.gpcp.gpcp [cp-options] file {file}");
        ELSIF RTS.defaultTarget = "dcf" THEN
	  PrintLn("       $ gpcp [cp-options] file {file}");
	END;
PrintLn("# CP Options ...");
PrintLn("       -clsdir=XXX  ==> Set class tree root in directory XXX");
PrintLn("       -copyright   ==> Display copyright notice");
PrintLn("       -dostats     ==> Give a statistical summary");
PrintLn("       -extras      ==> Enable experimental compiler features");
PrintLn("       -help        ==> Write out this usage message");
PrintLn("       -hsize=NNN   ==> Set hashtable size >= NNN (0 .. 65000)");
PrintLn("       -jasmin      ==> Ceate asm files and run Jasmin");
PrintLn("       -list        ==> (default) Create *.lst file if errors");
PrintLn("       -list+       ==> Unconditionally create *.lst file");
PrintLn("       -list-       ==> Don't create error *.lst file");
PrintLn("       -nocode      ==> Don't create any object files");
PrintLn("       -noasm       ==> Don't create asm (or object) files");
PrintLn("       -nosym       ==> Don't create *.sym (or asm or object) files");
PrintLn("       -special     ==> Compile dummy symbol file");
PrintLn("       -strict      ==> Disallow non-standard constructs");
PrintLn("       -symdir=XXX  ==> Place symbol files in directory XXX");
PrintLn("       -target=XXX  ==> Emit (jvm|net|dcf) assembly");
PrintLn("       -verbose     ==> Emit verbose diagnostics");
PrintLn("       -version     ==> Write out version number");
PrintLn("       -warn-       ==> Don't emit warnings");
PrintLn("       -nowarn      ==> Don't emit warnings");
PrintLn("       -xmlerror    ==> Print error messages in XML format");
        IF RTS.defaultTarget = "jvm" THEN
PrintLn("# Java Options ...");
PrintLn("       -D<name>=<value>  pass <value> to JRE as system property <name>");
PrintLn("       -DCPSYM=$CPSYM    pass value of CPSYM environment variable to JRE");
        END;
      END;
      Message("This program comes with NO WARRANTY");
      Message("Read source/GPCPcopyright for license details");
    END Usage;

(* ==================================================================== *)
(*			     Option Setting				*)
(* ==================================================================== *)

    PROCEDURE ParseOption*(IN opt : ARRAY OF CHAR);
      VAR copy : ARRAY 16 OF CHAR;
	  indx : INTEGER;
     (* ----------------------------------------- *)
      PROCEDURE Unknown(IN str : ARRAY OF CHAR);
      BEGIN
	Message('Unknown option "' + str + '"');
	doHelp := TRUE;
      END Unknown;
     (* ----------------------------------------- *)
      PROCEDURE BadSize();
      BEGIN Message('hsize must be integer in range 0 .. 65000') END BadSize;
     (* ----------------------------------------- *)
      PROCEDURE ParseSize(IN opt : ARRAY OF CHAR);
        VAR ix : INTEGER;
            nm : INTEGER;
            ch : CHAR;
      BEGIN
        nm := 0;
        ix := 7;
        WHILE opt[ix] # 0X DO
          ch := opt[ix];
          IF (ch >= '0') & (ch <= '9') THEN
            nm := nm * 10 + ORD(ch) - ORD('0');
            IF nm > 65521 THEN BadSize; hashSize := nm; RETURN END;
          ELSE
            BadSize; doHelp := TRUE; hashSize := nm; RETURN;
          END;
          INC(ix);
        END;
        hashSize := nm;
      END ParseSize;
     (* ----------------------------------------- *)
      PROCEDURE GetDirectory(IN  opt : ARRAY OF CHAR;
                             OUT dir : ARRAY OF CHAR);
        VAR idx : INTEGER;
            chr : CHAR;
      BEGIN
        ASSERT(opt[7] = "=");
        idx := 8;
        chr := opt[idx];
        WHILE (chr # 0X) & (idx < LEN(opt)) DO
          dir[idx - 8] := chr;
          INC(idx); chr := opt[idx];
        END;
      END GetDirectory;
     (* ----------------------------------------- *)
    BEGIN
      indx := 1;
      WHILE (indx < 16) & (indx < LEN(opt)) DO
	copy[indx-1] := opt[indx]; INC(indx);
      END;
      copy[15] := 0X;

      CASE copy[0] OF
      | "b" : 
          copy[7] := 0X;
          IF copy = "bindir=" THEN 
            GetDirectory(opt, binDir);
            expectedNet := TRUE;
            IF verbose THEN 
              Message("bin directory set to <" + binDir +">");
            END;
          ELSE 
            Unknown(opt);
          END;
      | "c" : 
          IF copy = "copyright" THEN 
            GPCPcopyright.Write;
          ELSE
            copy[7] := 0X;
            IF copy = "clsdir=" THEN
              GetDirectory(opt, binDir);
              expectedJvm := TRUE;
              IF verbose THEN 
                Message("output class tree rooted at <" + binDir +">");
              END;
            ELSE
              Unknown(opt);
            END;
          END;
      | "d" : 
          IF copy = "dostats" THEN 
            doStats := TRUE;
          ELSIF copy = "debug" THEN
	    debug    := TRUE;
            doIlasm  := TRUE;
            expectedNet := TRUE;
          ELSE 
            Unknown(opt);
          END;
      | "e" : IF copy = "extras" THEN extras := TRUE ELSE Unknown(opt) END;
(*
 *    | "p" : IF copy = "pdc" THEN netRel := pdc ELSE Unknown(opt) END;
 *    | "b" : 
 *        IF    copy = "beta1" THEN netRel := beta1;
 *        ELSIF copy = "beta2" THEN netRel := beta2;
 *        ELSE Unknown(opt);
 *        END;
 *)
      | "h" : 
          copy[6] := 0X;
          IF copy = "help" THEN
            doHelp := TRUE;
          ELSIF copy = "hsize=" THEN
            ParseSize(opt);
          ELSE
            Unknown(opt);
          END;
      | "i" : 
          IF copy = "ilasm" THEN 
            doIlasm := TRUE;
            debug := FALSE;
            expectedNet := TRUE;
          ELSE 
            Unknown(opt);
          END;
      | "j" :
	  IF copy = "jasmin" THEN
	    doCode     := TRUE;
	    doJsmn     := TRUE;
            expectedJvm := TRUE;
	  ELSE 
	    Unknown(opt);
	  END;
      | "l" :
	  IF copy = "list-" THEN
	    listLevel  := CPascalS.listNever;
	  ELSIF copy = "list+" THEN
	    listLevel  := CPascalS.listAlways;
	  ELSIF copy = "list" THEN
	    listLevel  := CPascalS.listErrOnly;
	  ELSE 
	    Unknown(opt);
	  END;
      | "n" :
	  IF copy = "nosym" THEN
	    doSym      := FALSE;
	    doAsm      := FALSE;
	    doCode     := FALSE;
	  ELSIF copy = "noasm" THEN
	    doAsm      := FALSE;
	    doCode     := FALSE;
	  ELSIF copy = "nocode" THEN
	    doCode     := FALSE;
	  ELSIF copy = "nowarn" THEN
	    warning    := FALSE;
            CPascalErrors.nowarn := TRUE;
	  ELSIF copy = "nocheck" THEN
	    ovfCheck   := FALSE;
            expectedNet := TRUE;
	  ELSIF copy = "nodebug" THEN
	    debug    := FALSE;
            doIlasm  := FALSE;
            expectedNet := TRUE;
	  ELSE 
	    Unknown(opt);
	  END;
      | "s" :
	  IF copy = "special" THEN
	    doAsm      := FALSE;
	    special    := TRUE;
	    strict     := FALSE;
	  ELSIF copy = "strict" THEN
	    strict     := TRUE;
  	  ELSE
            copy[7] := 0X;
            IF copy = "symdir=" THEN
              GetDirectory(opt, symDir);
              IF verbose THEN 
                Message("sym directory set to <" + symDir +">");
              END;
            ELSE
	      Unknown(opt);
            END;
	  END;
      | "t" :
	  IF (copy = "target=jvm") OR
	     (copy = "target=JVM") THEN
	    IF RTS.defaultTarget = "jvm" THEN
	      Message("JVM is default target for this build");
	    END;
	    target := "jvm";
	  ELSIF (copy = "target=vos") OR
		(copy = "target=net") OR
		(copy = "target=NET") THEN
	    IF RTS.defaultTarget = "net" THEN
	      Message("NET is default target for this build");
	    END;
	    target := "net";
	  ELSIF copy = "target=dcf" THEN
	    Message('DCode emitter not yet available, using "target=' +
		                                    RTS.defaultTarget + '"');
	  ELSE 
	    Unknown(opt);
	  END;
      | "u" :
	  IF copy = "unsafe" THEN
	    unsafe := TRUE;
            expectedNet := TRUE;
	  ELSE 
	    Unknown(opt);
	  END;
      | "v" :
	  IF copy = "version" THEN
	    doVersion  := TRUE;
	  ELSIF copy = "verbose" THEN
	    verbose    := TRUE;
	    doStats    := TRUE;
	    CPascalErrors.prompt := TRUE;
	  ELSIF copy = "vserror" THEN
            CPascalErrors.forVisualStudio := TRUE;
            expectedNet := TRUE;
	  ELSIF copy = "v1.0" THEN
            netRel := netV1_0;
            expectedNet := TRUE;
	  ELSIF copy = "v1.1" THEN
            netRel := netV1_1;
            expectedNet := TRUE;
	  ELSIF copy = "v2.0" THEN
            netRel := netV2_0;
            expectedNet := TRUE;
	  ELSE 
	    Unknown(opt);
	  END;
      | "w" :
	  IF copy = "warn-" THEN
	    warning    := FALSE;
            CPascalErrors.nowarn := TRUE;
	  ELSIF copy = "whidbey" THEN
            netRel := netV2_0;
            expectedNet := TRUE;
	  ELSE 
	    Unknown(opt);
	  END;
      | "x" :
	  IF copy = "xmlerror" THEN
            CPascalErrors.xmlErrors := TRUE;
	  ELSE 
	    Unknown(opt);
	  END;
      ELSE
	Unknown(opt);
      END;
      IF doVersion & ~doneVersion THEN 
	Message(target + GPCPcopyright.verStr); 
	doneVersion := TRUE;
      END;
      IF doHelp & ~doneHelp THEN Usage; doneHelp := TRUE END;
    END ParseOption;

(* ==================================================================== *)

    PROCEDURE CheckOptionsOK*;
    BEGIN
      IF target = "net" THEN
        IF expectedJvm THEN Message
          ("WARNING - a JVM-specific option was specified for .NET target");
          expectedJvm := FALSE;
        END;
      ELSIF target = "jvm" THEN
        IF expectedNet THEN Message
          ("WARNING - a .NET-specific option was specified for JVM target");
          expectedNet := FALSE;
        END;
      END;
    END CheckOptionsOK;

(* ==================================================================== *)

    PROCEDURE InitCompState*(IN nam : ARRAY OF CHAR);
    BEGIN
      IF verbose THEN Message("opened local file <" + nam + ">") END;
      GPText.Assign(nam, srcNam);
      CPascalErrors.SetSrcNam(nam);
      FileNames.StripExt(nam, basNam);
      FileNames.AppendExt(basNam, "lst", lstNam);
      NEW(thisMod); 
      thisMod.SetKind(IdDesc.modId);
      thisMod.ovfChk := ovfCheck;

      xhrId := IdDesc.newFldId();
      xhrId.hash := NameHash.enterStr("prev");

      srcBkt     := NameHash.enterStr("src");

      NEW(sysMod); 
      sysMod.SetKind(IdDesc.impId);
    END InitCompState;

(* ==================================================================== *)

  PROCEDURE Report*;
    VAR str1 : ARRAY 8 OF CHAR;
	str2 : ARRAY 8 OF CHAR;
  BEGIN
    Message(target + GPCPcopyright.verStr); 
    GPText.IntToStr(CPascalS.line, str1);
    Message(str1 + " source lines");
    GPText.IntToStr(impMax, str1);
    Message("import recursion depth " + str1);
    GPText.IntToStr(NameHash.size, str2);
    GPText.IntToStr(NameHash.entries, str1);
    Message(str1 + " entries in hashtable of size " + str2);
    TimeMsg("import time   ", import2 - import1);
    TimeMsg("source time   ", parseS  - totalS);
    TimeMsg("parse time    ", parseE  - parseS - import2 + import1);
    TimeMsg("analysis time ", attrib  - parseE);
    TimeMsg("symWrite time ", symEnd  - attrib);
    TimeMsg("asmWrite time ", asmEnd  - symEnd);
    TimeMsg("assemble time ", totalE  - asmEnd);
    TimeMsg("total time    ", totalE  - totalS);
  END Report;

(* ==================================================================== *)

  PROCEDURE InitOptions*;
  BEGIN
    warning	:= TRUE;
    verbose	:= FALSE;
    doHelp	:= FALSE; doneHelp    := FALSE;
    doVersion	:= FALSE; doneVersion := FALSE;
    ovfCheck	:= TRUE;
    debug	:= TRUE;
    netRel 	:= netV1_1; (* probably should be from RTS? *)
    doSym	:= TRUE;
    extras 	:= FALSE;
    unsafe 	:= FALSE;
    doStats	:= FALSE;
    doJsmn 	:= FALSE;
    doIlasm 	:= TRUE;
    doCode	:= TRUE;
    doAsm 	:= TRUE;
    special 	:= FALSE;
    strict 	:= FALSE;
    system 	:= FALSE;
    listLevel   := CPascalS.listErrOnly;
    hashSize    := 5000;	(* gets default hash size *)
    expectedNet := FALSE;
    expectedJvm := FALSE;
  END InitOptions;

(* ==================================================================== *)
BEGIN
  GPText.Assign(RTS.defaultTarget, target);
END CompState.
(* ==================================================================== *)

