// copyright 2001-2002 by The Mind Electric

package electric.util;

import java.util.*;
import java.io.*;
import electric.util.io.*;

/**
 * <tt>Lex</tt> is a lexical analyzer.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public final class Lex
  {
  public final static int SKIP_WS = 1;
  public final static int CONSUME = 2;
  public final static int INCLUDE = 4;
  public final static int EOF_OK = 8;
  public final static int QUOTES = 16;
  public final static int STRIP = 32;
  public final static int HTML = 64;
  public final static int COMMENTS = 128;

  final static int BUFFER_SIZE = 60;
  final static Hashtable substitutions = new Hashtable();

  private Reader reader;
  private String defaultDelimiters;
  private int defaultFlags;
  private int index = 0;
  private boolean eol;
  private int lineNumber = 1;
  private int charNumber = 0;
  private StringBuffer comment;
  private int[] window = null;

  // ********** INITIALIZATION **********************************************

  static
    {
    initSubstitutions();
    }

  // ********** CONSTRUCTION ************************************************

  /**
   * @param string
   */
  public Lex( String string )
    {
    this( new FastReader( string ) );
    }

  /**
   * @param reader
   */
  public Lex( Reader reader )
    {
    this( reader, "", SKIP_WS );
    }

  /**
   * @param string
   * @param defaultDelimiters
   * @param defaultFlags
   */
  public Lex( String string, String defaultDelimiters, int defaultFlags )
    {
    this( new FastReader( string ), defaultDelimiters, defaultFlags );
    }

  /**
   * @param reader
   * @param defaultDelimiters
   * @param defaultFlags
   */
  public Lex( Reader reader, String defaultDelimiters, int defaultFlags )
    {
    this.reader = reader;
    this.defaultDelimiters = defaultDelimiters;
    this.defaultFlags = defaultFlags;
    setWindow( BUFFER_SIZE );
    }

  // ********** COMMENTS ****************************************************

  /**
   *
   */
  public String getComment()
    {
    return (comment == null ? "" : comment.toString());
    }

  /**
   *
   */
  public void clearComment()
    {
    comment = null;
    }

  // ********** READING TO PATTERN ******************************************

  /**
   * @param pattern
   * @param flags
   * @throws IOException
   */
  public String readToPattern( String pattern, int flags )
    throws IOException
    {
    StringBuffer buffer = new StringBuffer();

    if( (flags & SKIP_WS) != 0 )
      skipWhitespace( flags );

    int ch = pattern.charAt( 0 );
    int length = pattern.length();

    while( true )
      {
      int next = peek();

      if( next == -1 )
        {
        if( (flags & EOF_OK) == 0 )
          throw new IOException( "unexpected EOF while searching for '" + pattern + "'" );

        read();
        return buffer.toString();
        }

      if( next == ch ) // got first character of pattern
        {
        if( length == 1 )
          {
          if( (flags & INCLUDE) != 0 )
            buffer.append( (char) next );

          if( (flags & CONSUME) != 0 )
            read();

          return buffer.toString();
          }
        else
          {
          int[] peekBuffer = new int[ length ];
          peek( peekBuffer );
          boolean match = true;

          for( int i = 1; i < length; i++ )
            if( peekBuffer[ i ] != pattern.charAt( i ) )
              {
              match = false;
              break;
              }

          if( match )
            {
            if( (flags & CONSUME) != 0 )
              for( int i = 0; i < length; i++ )
                read();

            if( (flags & INCLUDE) != 0 )
              for( int i = 0; i < length; i++ )
                buffer.append( (char) peekBuffer[ i ] );

            return buffer.toString();
            }
          }
        }

      if( next == '&' && (flags & HTML) != 0)
        readHTML( buffer );
      else if( (next == '\'' || next == '"') && (flags & QUOTES) != 0)
        readQuoted( buffer, next, flags );
      else
        buffer.append( (char) read() );
      }
    }

  // ********** READING TO DELIMITER ****************************************

  /**
   * @param delimiters
   * @param flags
   * @throws IOException
   */
  public String readToDelimiter( String delimiters, int flags )
    throws IOException
    {
    if( (flags & SKIP_WS) != 0 )
      skipWhitespace( flags );

    StringBuffer buffer = new StringBuffer();

    while( true )
      {
      int next = peek();

      if( next == -1 )
        {
        if( buffer.length() <= 0 )
          throw new IOException( "unexpected EOF" );

        read();
        return buffer.toString();
        }

      if( delimiters.indexOf( next ) != -1 )
        {
        if( buffer.length() == 0 )
          buffer.append( (char) read() );

        return buffer.toString();
        }

      if( Character.isWhitespace( (char) next ) )
        return buffer.toString();

      if( next == '&' && (flags & HTML) != 0)
        readHTML( buffer );
      else if( (next == '\'' || next == '"') && (flags & QUOTES) != 0)
        readQuoted( buffer, next, flags );
      else
        buffer.append( (char) read() );
      }
    }

  /**
   * @param delimiters
   * @throws IOException
   */
  public String readToDelimiter( String delimiters )
    throws IOException
    {
    return readToDelimiter( delimiters, defaultFlags );
    }

  /**
   * @throws IOException
   */
  public String readToken()
    throws IOException
    {
    return readToDelimiter( defaultDelimiters, defaultFlags );
    }

  /**
   * @param expected
   * @throws IOException
   */
  public void readToken( String expected )
    throws IOException
    {
    String got = readToken();

    if( !expected.equals( got ) )
      throw new IOException( "expected \"" + expected + "\", got \"" + got + "\"" );
    }

  // ********** READING AN HTML SPECIAL CHARACTER ***************************

  /**
   * @param buffer
   * @throws IOException
   */
  public void readHTML( StringBuffer buffer )
    throws IOException
    {
    read(); // &
    String special = readTo( ';' );

    try
      {
      if( special.startsWith( "#x" ) )
        {
        buffer.append( (char) Integer.parseInt( special.substring( 2 ), 16 ) );
        }
      else if( special.startsWith( "#" ) )
        {
        buffer.append( (char) Integer.parseInt( special.substring( 1 ) ) );
        }
      else
        {
        String replacement = (String) substitutions.get( special );

        if( replacement != null )
          {
          buffer.append( replacement );
          }
        else
          {
          buffer.append( '&' );
          buffer.append( special );
          buffer.append( ';' );
          }
        }
      }
    catch( NumberFormatException exception )
      {
      throw new IOException( "number format exception while parsing &" + special + ";" );
      }
    }

  // ********** READING A QUOTED STRING *************************************

  /**
   * @param buffer
   * @param quote
   * @param flags
   * @throws IOException
   */
  private void readQuoted( StringBuffer buffer, int quote, int flags )
    throws IOException
    {
    int c = read();

    if( (flags & STRIP) == 0 )
      buffer.append( (char) c );

    while( true )
      {
      c = read();

      if( c == -1 )
        break;

      if( c == quote )
        {
        if( (flags & STRIP) == 0 )
          buffer.append( (char) c );

        break;
        }

      buffer.append( (char) c );
      }
    }

  // ********** READING CHARACTERS ******************************************

  /**
   * @throws IOException
   */
  public int readChar()
    throws IOException
    {
    skipWhitespace();
    int ch = read();

    if( ch == -1 )
      throw new IOException( "unexpected EOF" );

    return ch;
    }

  /**
   * @param expected
   * @throws IOException
   */
  public void readChar( int expected )
    throws IOException
    {
    int got = readChar();

    if( got != expected )
      throw new IOException( "expected '" + (char) expected + "', got '" + (char) got + "'" );
    }

  /**
   * @param ch
   * @throws IOException
   */
  public String readTo( int ch )
    throws IOException
    {
    StringBuffer special = new StringBuffer();

    while( true )
      {
      int c = read();

      if( c == -1 )
        throw new IOException( "could not find " + ch );
      else if( c == ch )
        break;
      else
        special.append( (char) c );
      }

    return special.toString();
    }

  // ********** WHITESPACE **************************************************

  /**
   * @throws IOException
   */
  public StringBuffer readWhitespace()
    throws IOException
    {
    return readWhitespace( defaultFlags );
    }

  /**
   * @param flags
   * @throws IOException
   */
  public StringBuffer readWhitespace( int flags )
    throws IOException
    {
    StringBuffer buffer = null;

    while( true )
      {
      reader.mark( 2 );
      int ch = reader.read();

      if( Character.isWhitespace( (char) ch ) )
        {
        if( buffer == null )
          buffer = new StringBuffer();

        reader.reset();
        buffer.append( (char) read() );
        }
      else
        {
        if( ch == '/' && (flags & COMMENTS) != 0 )
          {
          ch = reader.read();
          reader.reset();

          if( ch == '/' )
            readOneLineComment();
          else if( ch == '*' )
            readMultiLineComment();
          else
            return buffer;
          }
        else
          {
          reader.reset();
          return buffer;
          }
        }
      }
    }

  /**
   * @throws IOException
   */
  public void skipWhitespace()
    throws IOException
    {
    skipWhitespace( defaultFlags );
    }

  /**
   * @param flags
   * @throws IOException
   */
  public void skipWhitespace( int flags )
    throws IOException
    {
    while( true )
      {
      reader.mark( 2 );
      int ch = reader.read();

      if( Character.isWhitespace( (char) ch ) )
        {
        reader.reset();
        read();
        }
      else
        {
        if( ch == '/' && (flags & COMMENTS) != 0 )
          {
          ch = reader.read();
          reader.reset();

          if( ch == '/' )
            readOneLineComment();
          else if( ch == '*' )
            readMultiLineComment();
          else
            return;
          }
        else
          {
          reader.reset();
          return;
          }
        }
      }
    }

  // ********** COMMENTS ****************************************************

  /**
   * @throws IOException
   */
  private void readOneLineComment()
    throws IOException
    {
    if( comment == null )
      comment = new StringBuffer();

    comment.append( readToPattern( "\n", (CONSUME | EOF_OK) ) );
    }

  /**
   * @throws IOException
   */
  private void readMultiLineComment()
    throws IOException
    {
    if( comment == null )
      comment = new StringBuffer();

    while( true )
      {
      int ch = read();

      if( ch == -1 )
        throw new IOException( "missing */ on comment" );

      comment.append( (char) ch );

      if( ch == '*' && peek() == '/' )
        {
        ch = read();
        comment.append( (char) ch );
        break;
        }
      }
    }

  // ********** PEEKING *****************************************************

  /**
   * @param length
   * @throws IOException
   */
  public String peekString( int length )
    throws IOException
    {
    int[] array = new int[ length ];
    peek( array );
    StringBuffer buffer = new StringBuffer();

    for( int i = 0; i < length; i++ )
      if( array[ i ] != -1 )
        buffer.append( (char) array[ i ] );
      else
        break;

    return buffer.toString();
    }

  /**
   * @param string
   * @throws IOException
   */
  public boolean peekString( String string )
    throws IOException
    {
    return string.equals( peekString( string.length() ) );
    }

  /**
   * @param buffer
   * @throws IOException
   */
  public void peek( int[] buffer )
    throws IOException
    {
    reader.mark( buffer.length );

    for( int i = 0; i < buffer.length; i++ )
      buffer[ i ] = reader.read();

    reader.reset();
    }

  /**
   * @throws IOException
   */
  public int peek()
    throws IOException
    {
    reader.mark( 1 );
    int next = reader.read();
    reader.reset();
    return next;
    }

  /**
   * @param length
   */
  public void mark( int length )
    throws IOException
    {
    reader.mark( length );
    }

  /**
   * @throws IOException
   */
  public void reset()
    throws IOException
    {
    reader.reset();
    }

  // ********** EOF *********************************************************

  /**
   * @throws IOException
   */
  public boolean eof()
    throws IOException
    {
    skipWhitespace();
    return (peek() == -1);
    }

  // ********** SKIPPING ****************************************************

  /**
   * @param length
   * @throws IOException
   */
  public void skip( int length )
    throws IOException
    {
    for( int i = 0; i < length; i++ )
      read();
    }

  // ********** CURRENT LOCATION ********************************************

  /**
   *
   */
  public String getLocation()
    {
    StringBuffer buffer = new StringBuffer();
    buffer.append( "line " ).append( lineNumber ).append( ", char " ).append( charNumber );

    if( window == null )
      return buffer.toString();

    buffer.append( ": ..." );
    StringBuffer line = new StringBuffer();
    int count = 0;
    int start = index;

    while( count++ < charNumber )
      {
      if( start-- == 0 ) // go back one
        start = BUFFER_SIZE - 1;

      if( window[ start ] == '\n' || start == index )
        break;

      line.append( (char) window[ start ] );
      }

    return buffer.append( line.reverse() ).toString();
    }

  /**
   *
   */
  public int getLineNumber()
    {
    return lineNumber;
    }

  // ********** READ PRIMITIVE **********************************************

  /**
   * @throws IOException
   */
  public int peekRead()
    throws IOException
    {
    return reader.read();
    }

  /**
   * @throws IOException
   */
  public int read()
    throws IOException
    {
    if( eol )
      {
      ++lineNumber;
      charNumber = 0;
      }

    int ch = reader.read();
    ++charNumber;
    eol = (ch == '\n');

    if( window != null )
      {
      window[ index ] = ch;

      if( ++index == window.length )
        index = 0;
      }

    return ch;
    }

  // ********** WINDOW ******************************************************

  /**
   * @param length
   */
  public void setWindow( int length )
    {
    window = new int[ length ];
    }

  // ********** SUBSTITUTIONS ***********************************************

  /**
   *
   */
  private static void initSubstitutions()
    {
    // alternatively, pass subsitutions array in during construction
    // w/ start and stop markers (&, ;)
    substitutions.put( "apos", "'" );
    substitutions.put( "lt", "<" );
    substitutions.put( "gt", ">" );
    substitutions.put( "quot", "\"" );
    substitutions.put( "amp", "&" );
    }
  }