// copyright 2001-2002 by The Mind Electric

package electric.util;

import java.io.*;
import java.lang.reflect.*;
import electric.util.encoding.*;

/**
 * <tt>Strings</tt> defines a set of static methods for manipulating strings.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public final class Strings
  {
  /**
   * @param flag
   * @param args
   * @param index
   */
  public static String getString( String flag, String[] args, int index )
    {
    if( index >= args.length )
      throw new IllegalArgumentException( "missing argument to -" + flag );

    return args[ index ];
    }

  /**
   * @param string
   */
  public static String asJavaName( String string )
    {
    StringBuffer buffer = new StringBuffer();

    for( int i = 0; i < string.length(); i++ )
      {
      char ch = string.charAt( i );

      if( (i == 0 && Character.isJavaIdentifierStart( ch )) || Character.isJavaIdentifierPart( ch ) )
        buffer.append( ch );
      }

    return buffer.toString();
    }

  /**
   * @param name
   */
  public static String getJavaPackage( String name )
    {
    int dot = name.lastIndexOf( '.' );
    return (dot != -1 ? name.substring( 0, dot ) : null);
    }

  /**
   * @param name
   */
  public static String getLocalJavaName( String name )
    {
    return tail( name, '.' );
    }

  /**
   * @param bytes
   */
  public static boolean isUTF16( byte[] bytes )
    {
    if( bytes.length < 2 )
      return false;

    return (bytes[ 0 ] == (byte) 0xFF && bytes[ 1 ] == (byte) 0xFE) || (bytes[ 0 ] == (byte) 0xFE && bytes[ 1 ] == (byte) 0xFF);
    }

  /**
   * @param encoding
   */
  public static boolean isUTF8( String encoding )
    {
    return (encoding == null || encoding.equalsIgnoreCase( "UTF-8" ) || encoding.equalsIgnoreCase( "UTF8" ));
    }

  /**
   * @param bytes
   */
  public static String toString( byte[] bytes )
    throws UnsupportedEncodingException
    {
    byte[] header = bytes;

    if( bytes.length > 100 )
      {
      header = new byte[ 100 ];
      System.arraycopy( bytes, 0, header, 0, 100 );
      }

    return new String( bytes, Encodings.getJavaEncoding( header ) );
    }

  /**
   * @param path
   */
  public static String extension( String path )
    {
    int slash = path.lastIndexOf( '/' ); // what about '\'?
    int dot = path.lastIndexOf( '.' );
    return (dot > slash ? path.substring( dot + 1 ) : null);
    }

  /**
   * @param string
   * @param delimiter
   */
  public static String tail( String string, char delimiter )
    {
    int index = string.lastIndexOf( delimiter );
    return (index == -1 ? string : string.substring( index + 1 ));
    }

  /**
   * @param string
   * @param delimiter
   */
  public static String head( String string, char delimiter )
    {
    int index = string.lastIndexOf( delimiter );
    return (index == -1 ? string : string.substring( 0, index ));
    }

  /**
   * @param path1
   * @param path2
   */
  public static String splice( String path1, String path2 ) // assumes /, not \
    {
    if( path2 == null || path2.length() == 0 )
      return path1;
    else if( path1.endsWith( "/" ) && path2.startsWith( "/" ) )
      return path1 + path2.substring( 1 );
    else if( path1.endsWith( "/" ) || path2.startsWith( "/" ) )
      return path1 + path2;
    else if( path1.length() > 0 )
      return path1 + "/" + path2;
    else
      return path2;
    }

  /**
   * @param string
   * @param before
   * @param after
   */
  public static String replace( String string, String before, String after )
    {
    if( string.indexOf( before.charAt( 0 ) ) == -1 )
      return string;

    StringBuffer buffer = new StringBuffer();
    int index = 0;

    while( true )
      {
      int match = string.indexOf( before, index );

      if( match == -1 )
        {
        buffer.append( string.substring( index, string.length() ) );
        return buffer.toString();
        }
      else
        {
        buffer.append( string.substring( index, match ) ).append( after );
        index = match + before.length();
        }
      }
    }

  /**
   * @param string
   * @param charToStrip
   */
  public static String stripChar( String string, char charToStrip )
    {
    StringBuffer newString = new StringBuffer();
    char[] chars = string.toCharArray();

    for( int i = 0; i < chars.length; i++ )
      if( chars[ i ] != charToStrip )
        newString.append( chars[ i ] );

    return newString.toString();
    }

  /**
   * @param array
   */
  public static String arrayToString( Object array )
    {
    int length = Array.getLength( array );

    if( length == 0 )
      return "()";

    StringBuffer buffer = new StringBuffer();
    buffer.append( "( " );

    for( int i = 0; i < length; i++ )
      {
      if( i > 0 )
        buffer.append( ", " );

      buffer.append( Array.get( array, i ) );
      }

    return buffer.append( " )" ).toString();
    }

  /**
   * @param string
   */
  public static String getURN( String string )
    {
    int index = string.lastIndexOf( "/" );
    return (index == -1 ? string : string.substring( index + 1 ));
    }

  /**
   * @param string
   */
  public static String getEndpoint( String string )
    {
    int index = string.lastIndexOf( "/" );
    return (index == -1 ? "" : string.substring( 0, index )); // don't include final "/"
    }

  /**
   * @param name
   */
  public static String toFilename( String name )
    {
    StringBuffer buffer = new StringBuffer();

    for( int i = 0; i < name.length(); i++ )
      {
      char ch = name.charAt( i );

      if( Character.isLetterOrDigit( ch ) || ch == '.' )
        {
        buffer.append( ch );
        }
      else
        {
        buffer.append( '_' );
        buffer.append( Integer.toString( (int) ch ) );
        buffer.append( '_' );
        }
      }

    return buffer.toString();
    }

  /**
   * @param name
   */
  public static String fromFilename( String name )
    {
    int start = name.indexOf( '_' );

    if( start == -1 )
      return name;

    int stop = name.indexOf( '_', start + 1 );
    char ch = (char) Integer.parseInt( name.substring( start + 1, stop ) );
    return name.substring( 0, start ) + ch + fromFilename( name.substring( stop + 1 ) );
    }

  /**
   * @param query
   * @param post
   */
  public static void substitute( String[][] query, String[][] post )
    throws IOException
    {
    for( int i = 0; i < query.length; i++ )
      {
      String value = query[ i ][ 1 ];

      if( value.length() > 0 && value.charAt( 0 ) == '$' ) // argument
        {
        String key = value.substring( 1, value.length() ); // what if exception here?

        for( int j = 0; j < post.length; j++ )
          if( post[ j ][ 0 ].equals( key ) )
            {
            query[ i ][ 1 ] = post[ j ][ 1 ];
            break;
            }
        }
      }
    }

  /**
   * @param string
   */
  public static String getCapitalized( String string )
    {
    return Character.toUpperCase( string.charAt( 0 ) ) + string.substring( 1 );
    }

  /**
   * Returns true if the string is "yes" or "true", false if "no" or "false", otherwise
   * it returns defaultValue.
   * @param string
   * @param defaultValue
   */
  public static boolean getBoolean( String string, boolean defaultValue )
    {
    if( string == null )
      return defaultValue;
    else if( string.equalsIgnoreCase( "yes" ) || string.equalsIgnoreCase( "true" ) )
      return true;
    else if( string.equalsIgnoreCase( "no" ) || string.equalsIgnoreCase( "false" ) )
      return false;
    else
      return defaultValue;
    }

  /**
   * Remove the quotes from around a string.
   * @param quotedString
   */
  public static String stripQuotes( String quotedString )
    {
    if ( quotedString != null && quotedString.length() > 2 && quotedString.startsWith( "\"" ) )
      return quotedString.substring( 1, quotedString.length() - 1 );

    return quotedString;
    }

  /**
   * @param string
   */
  public static byte[] getASCII( String string )
    {
    try
      {
      return string.getBytes( "ASCII" );
      }
    catch( Exception e )
      {
      }

    return null;
    }
  }
