// copyright 2001-2002 by The Mind Electric

package electric.util.io;

import java.io.*;

/**
 * <tt>FastBufferedReader</tt> is a fast version of BufferedReader.
 */

public final class FastBufferedReader extends Reader
  {
  private static final int INVALIDATED = -2;
  private static final int UNMARKED = -1;
  private static int defaultCharBufferSize = 8192;
  private static int defaultExpectedLineLength = 80;

  private Reader reader;
  private char buffer[];
  private int nChars;
  private int nextChar;
  private int markedChar = UNMARKED;
  private int readAheadLimit = 0; // valid only when markedChar > 0
  private boolean skipLF = false; // if the next character is a line feed, skip it
  private boolean markedSkipLF = false; // the skipLF flag when the mark was set

  // ********** CONSTRUCTION ************************************************

  /**
   * @param reader
   * @param size
   */
  public FastBufferedReader( Reader reader, int size )
    {
    super( reader );

    if( size <= 0 )
      throw new IllegalArgumentException( "buffer size <= 0" );

    this.reader = reader;
    buffer = new char[ size ];
    nextChar = nChars = 0;
    }

  /**
   * @param reader
   */
  public FastBufferedReader( Reader reader )
    {
    this( reader, defaultCharBufferSize );
    }

  // ********** I/O *********************************************************

  /**
   * @throws IOException
   */
  private void fill()
    throws IOException
    {
    int dst;

    if( markedChar <= UNMARKED ) // no mark
      {
      dst = 0;
      }
    else // marked
      {
      int delta = nextChar - markedChar;

      if( delta >= readAheadLimit ) // gone past read-ahead limit: Invalidate mark
        {
        markedChar = INVALIDATED;
        readAheadLimit = 0;
        dst = 0;
        }
      else
        {
        if( readAheadLimit <= buffer.length ) // shuffle in the current buffer
          {
          System.arraycopy( buffer, markedChar, buffer, 0, delta );
          markedChar = 0;
          dst = delta;
          }
        else // Reallocate buffer to accomodate read-ahead limit
          {
          char ncb[] = new char[ readAheadLimit ];
          System.arraycopy( buffer, markedChar, ncb, 0, delta );
          buffer = ncb;
          markedChar = 0;
          dst = delta;
          }

        nextChar = nChars = delta;
        }
      }

    int n;

    do
      {
      n = reader.read( buffer, dst, buffer.length - dst );
      }
    while( n == 0 );

    if( n > 0 )
      {
      nChars = dst + n;
      nextChar = dst;
      }
    }

  /**
   *
   */
  public int read()
    throws IOException
    {
    for(;;)
      {
      if( nextChar >= nChars )
        {
        fill();

        if( nextChar >= nChars )
          return -1;
        }

      if( skipLF )
        {
        skipLF = false;

        if( buffer[ nextChar ] == '\n' )
          {
          nextChar++;
          continue;
          }
        }

      return buffer[ nextChar++ ];
      }
    }

  /**
   * @param cbuf
   * @param offset
   * @param length
   * @throws IOException
   */
  private int read1( char[] cbuf, int offset, int length )
    throws IOException
    {
    if( nextChar >= nChars )
      {
      if( length >= buffer.length && markedChar <= UNMARKED && !skipLF )
        return reader.read( cbuf, offset, length );

      fill();
      }

    if( nextChar >= nChars )
      return -1;

    if( skipLF )
      {
      skipLF = false;

      if( buffer[ nextChar ] == '\n' )
        {
        nextChar++;

        if( nextChar >= nChars )
          fill();

        if( nextChar >= nChars )
          return -1;
        }
      }

    int n = Math.min( length, nChars - nextChar );
    System.arraycopy( buffer, nextChar, cbuf, offset, n );
    nextChar += n;
    return n;
    }

  /**
   * @param cbuf
   * @param offset
   * @param length
   * @throws IOException
   */
  public int read( char cbuf[], int offset, int length )
    throws IOException
    {
    if( length == 0 )
      return 0;

    int n = read1( cbuf, offset, length );

    if( n <= 0 )
      return n;

    while( (n < length) && reader.ready() )
      {
      int n1 = read1( cbuf, offset + n, length - n );

      if( n1 <= 0 )
        break;

      n += n1;
      }

    return n;
    }

  /**
   * @param n
   * @throws IOException
   */
  public long skip( long n )
    throws IOException
    {
    long r = n;

    while( r > 0 )
      {
      if( nextChar >= nChars )
        fill();

      if( nextChar >= nChars ) /* EOF */
	break;

      if( skipLF )
        {
        skipLF = false;

        if( buffer[ nextChar ] == '\n' )
          nextChar++;
        }

      long d = nChars - nextChar;

      if( r <= d )
        {
        nextChar += r;
        r = 0;
        break;
        }
      else
        {
        r -= d;
        nextChar = nChars;
        }
      }

    return n - r;
    }

  /**
   * @throws IOException
   */
  public boolean ready()
    throws IOException
    {
    return (nextChar < nChars) || reader.ready();
    }

  /**
   *
   */
  public boolean markSupported()
    {
    return true;
    }

  /**
   * @param readAheadLimit
   */
  public void mark( int readAheadLimit )
    {
    this.readAheadLimit = readAheadLimit;
    markedChar = nextChar;
    markedSkipLF = skipLF;
    }

  /**
   * @throws IOException
   */
  public void reset()
    throws IOException
    {
    if( markedChar < 0 )
      throw new IOException( (markedChar == INVALIDATED ? "Mark invalid" : "Stream not marked") );

    nextChar = markedChar;
    skipLF = markedSkipLF;
    }

  /**
   * @throws IOException
   */
  public void close()
    throws IOException
    {
    if( reader == null )
      return;

    reader.close();
    reader = null;
    buffer = null;
    }
  }