// copyright 2001-2002 by The Mind Electric

package electric.xml;

import java.io.*;
import electric.util.encoding.Encodings;
import electric.util.io.*;

/**
 * <tt>Child</tt> is the abstract root of all nodes that have a parent,
 * and includes method for adding/inserting siblings
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public abstract class Child extends Node implements Cloneable
  {
  protected Parent parent; // my parent, or null if I have none

  // ********** CONSTRUCTION ************************************************

  /**
   * Construct a Child with no parent.
   */
  protected Child()
    {
    }

  /**
   * Construct a Child with the specified parent.
   * @param parent The parent.
   */
  protected Child( Parent parent )
    {
    parent.addChild( this );
    }

  // ********** STANDARD METHODS ********************************************

  /**
   * Write myself to a String, starting with indent level 0, and then
   * return the String.
   */
  public String toString()
    {
    return toString( getStartIndent() );
    }

  /**
   * Write myself to a String, starting at the specified indent level, and
   * then return the String. If the indent level is -1, no indentation will
   * occur, otherwise the indent level increases by two at each child node.
   * @param indent The initial indent level.
   */
  public String toString( int indent )
    {
    StringWriter stringWriter = new StringWriter();
    NodeWriter nodeWriter = new NodeWriter( stringWriter, indent, false );

    try
      {
      nodeWriter.write( this );
      }
    catch( IOException exception )
      {
      }

    return stringWriter.toString();
    }

  // ********** CLONING *****************************************************

  /**
   * Return a clone of this Node.
   */
  abstract public Object clone();

  // ********** PARENT *****************************************************

  /**
   * Set my parent.
   * @param parent The new parent value.
   */
  public void setParent( Parent parent )
    {
    this.parent = parent;
    }

  /**
   * Return my parent.
   */
  public Parent getParent()
    {
    return parent;
    }

  // ********** REPLACING ****************************************************

  /**
   * Replace myself with the specified Child.
   * @param child The Child.
   */
  public void replaceWith( Child child )
    {
    parent.replaceChild( this, child );
    }

  // ********** SIBLINGS ****************************************************

  /**
   * Return my next sibling, or null if I have none.
   */
  public Child getNextSiblingChild()
    {
    return (Child) getNextSiblingNode();
    }

  /**
   * Set my next sibling to be the specified child.
   * If the child was already in a document, remove it first.
   * @param child The child to add
   * @return Myself, to allow cascading.
   */
  public Child setNextSiblingChild( Child child )
    {
    child.setParent( parent );
    setNextSiblingNode( child );
    return this;
    }

  /**
   * Return my previous sibling, or null if I have none.
   */
  public Child getPreviousSiblingChild()
    {
    return (Child) getPreviousSiblingNode();
    }

  /**
   * Set my previous sibling to be the specified child.
   * If the child was already in a document, remove it first.
   * @param child The child to add
   * @return Myself, to allow cascading.
   */
  public Child setPreviousSiblingChild( Child child )
    {
    child.setParent( parent );
    setPreviousSiblingNode( child );
    return this;
    }

  // ********** DOCUMENT ****************************************************

  /**
   * Return my parent's document, or null if I have no parent.
   */
  public Document getDocument()
    {
    return (parent == null ? null : parent.getDocument());
    }

  // ********** ROOT ********************************************************

  /**
   * Return my parent's root, or null if I have no parent.
   */
  public Element getRoot()
    {
    return (parent == null ? null : parent.getRoot());
    }

  // ********** INDENTATION *************************************************

  /**
   *
   */
  private int getStartIndent()
    {
    Document document = getDocument();
    return (document == null || document.isStripped() ? 0 : -1);
    }

  // ********** WRITING *****************************************************

  /**
   * Write myself, starting at indent level 0, to the specified Writer.
   * @param writer The Writer.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( Writer writer )
    throws IOException
    {
    NodeWriter nodeWriter = new NodeWriter( writer, getStartIndent(), false );
    nodeWriter.write( this );
    }

  /**
   * Write myself, starting at indent level 0, to the specified File.
   * The File is written in either UTF-8 or UTF-16 format, depending on the
   * encoding style of the document.
   * @param file The File to write to.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( File file )
    throws IOException
    {
    Writer fileWriter = new BufferedWriter( Streams.getWriter( file, getDocument().getEncoding() ) );

    try
      {
      write( fileWriter );
      }
    finally
      {
      fileWriter.close();
      }
    }

  /**
   * Write myself, starting at indent level 0, to the specified OutputStream.
   * The stream is written in either UTF-8 or UTF-16 format, depending on the
   * encoding style of the document.
   * @param stream The OutputStream to write to.
   * @throws IOException If an I/O exception occurs.
   */
  public void write( OutputStream stream )
    throws IOException
    {
    Writer streamWriter = Streams.getWriter( stream, getDocument().getEncoding() );
    write( streamWriter );
    streamWriter.flush();
    }

  // ********** CONVERTING TO BYTES *****************************************

  /**
   * Write myself with no indentation to a byte array and then return the
   * byte array. The bytes are generated in either UTF-8 or UTF-16 format,
   * depending on the encoding style of the document.
   * @throws UnsupportedEncodingException
   */
  public byte[] getBytes()
    throws UnsupportedEncodingException
    {
    return getBytes( false );
    }

  /**
   * Write myself with no indentation to a byte array and then return the
   * byte array. The bytes are generated in either UTF-8 or UTF-16 format,
   * depending on the encoding style of the document.
   * @param expandEmptyElements Expand empty elements?
   * @throws UnsupportedEncodingException
   */
  public byte[] getBytes( boolean expandEmptyElements )
    throws UnsupportedEncodingException
    {
    StringWriter stringWriter = new StringWriter();

    try
      {
      NodeWriter nodeWriter = new NodeWriter( stringWriter, -1, expandEmptyElements );
      nodeWriter.write( this );
      }
    catch( IOException exception )
      {
      }

    return getBytes( stringWriter.toString() );
    }

  /**
   * @param xmlString
   */
  public byte[] getBytes( String xmlString )
    throws UnsupportedEncodingException
    {
    Document document = getDocument();

    if( document == null )
      {
      // use default encoding
      return xmlString.getBytes();
      }
    else
      {
      String encoding = Encodings.getJavaEncoding( document.getEncoding() );
      return xmlString.getBytes( encoding );
      }
    }

  // ********** DOM *********************************************************

  /**
   * Return a duplicate of this node, i.e., serve as a generic copy constructor
   * for nodes. The duplicate node has no parent; ( parentNode is null.).
   * Cloning an Element copies all attributes and their values, including those
   * generated by the XML processor to represent defaulted attributes, but
   * this method does not copy any text it contains unless it is a deep clone,
   * since the text is contained in a child Text node. Cloning an Attribute
   * directly, as opposed to be cloned as part of an Element cloning operation,
   * returns a specified attribute (specified is true). Cloning any other type
   * of node simply returns a copy of this node. Note that cloning an immutable
   * subtree results in a mutable copy, but the children of an EntityReference
   * clone are readonly. In addition, clones of unspecified Attr nodes are
   * specified. And, cloning Document, DocumentType, Entity, and Notation nodes
   * is implementation dependent.
   * @param deep If true, recursively clone the subtree under the specified node;
   * if false, clone only the node itself (and its attributes, if it is an Element).
   */
  public org.w3c.dom.Node cloneNode( boolean deep )
    {
    // ignore deep parameter for now
    return (org.w3c.dom.Node) clone();
    }

  /**
   * Return the Document object associated with this node. This is also the
   * Document object used to create new nodes. When this node is a Document
   * or a DocumentType which is not used with any Document yet, this is null.
   */
  public org.w3c.dom.Document getOwnerDocument()
    {
    return getDocument();
    }

  /**
   * Return the parent of this node. All nodes, except Attr, Document,
   * DocumentFragment, Entity, and Notation may have a parent. However, if
   * a node has just been created and not yet added to the tree, or if it
   * has been removed from the tree, this is null.
   */
  public org.w3c.dom.Node getParentNode()
    {
    return parent;
    }
  }