/**
 * Title:        Comedia Utils
 * Description:  Project contains some general purpose non-visual beans.
 * Beans do not require any special libraies.
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.util;

import java.util.*;
import java.io.*;
import java.net.*;
import java.text.*;

/**
 * Emulates Windows INI file interface. It loads and entire ini file into
 * memory and allows all operations to be performed on the memory image.
 * The image can then be written out to the disk file.
 * <p>
 * Example of usage:
 * <pre>
 * CIniFile iniFile = new CIniFile("test.ini");
 *
 * boolean p1 = iniFile.readBool("Section 1", "P1", false);
 * System.out.println("P1 = " + p1);
 * iniFile.writeBool("Section 1", "P1", !p1);
 *
 * System.out.println("Section 2, Parameter 1 = " 
 *   + iniFile.readString("Other X", "Param1", ""));
 *
 * System.out.println("Section 2, Parameter 2 = " 
 *   + iniFile.readInteger("Section 2", "P2", 123));
 *
 * System.out.println("Section 2, Parameter 3 = " 
 *  + iniFile.readDateTime("Section 2", "P3", new Date()));
 *
 * iniFile.writeString("Section 2", "P1", "..\"xxx\"...");
 * iniFile.writeDateTime("Section 2", "P3", new Date());
 *
 * iniFile.flush();
 * </pre>
 */
public class CIniFile {
  // Ini file contents
  private CItemArray sections = new CItemArray();
  // Url which points to destination file.
  private URL url = null;

  /**
   * Constructs an empty class with default properties.
   */
  public CIniFile() {
  }

  /**
   * Constructs this class with specified class name.
   * @param fileName a name of a ini file.
   */
  public CIniFile(String fileName) throws IOException {
    this(new File(fileName));
  }

  /**
   * Constructs this class with specified class name.
   * @param file a file object of a ini file.
   */
  public CIniFile(File file) throws IOException {
    this(file.toURL());
  }

  /**
   * Constructs this class with specified class name.
   * @param url an url which points to file.
   */
  public CIniFile(URL url) throws IOException {
    this.url = url;
    loadFromStream(url.openStream());
  }

  /**
   * Creates output stream for the current url.
   * @result a created output stream.
   */
  private OutputStream createOutputStream() throws IOException {
    if (url.getProtocol().equals("file"))
      return new FileOutputStream(url.getFile());
    return url.openConnection().getOutputStream();
  }

  /**
   * Adds new empty section to the and of the file.
   * @param section the name of the new section.
   * @result the new section class.
   */
  private CItemArray addSection(String section) {
    CItemArray a = new CItemArray();
    sections.putValue(section, a);
    return a;
  }

  /**
   * Removes all sections from this ini file.
   */
  public void clear() {
    sections.clear();
  }

  /**
   * Removes the specified key from the section.
   * @param section a name of the section.
   * @param ident the name of removed key.
   */
  public void deleteKey(String section, String ident) {
    CItemArray a = (CItemArray) sections.getValue(section);
    if (a != null)
      a.removeValue(ident);
  }

  /**
   * Removes specified section from this ini file.
   * @param section the name of removed section.
   */
  public void eraseSection(String section) {
    sections.removeValue(section);
  }

  /**
   * Tests if key exists in the specified section in this ini file.
   * @param section the name of the section.
   * @param ident the name of the tested key.
   * @result <code>TRUE</code> if key exists, <code>FALSE</code> otherwise.
   */
  public boolean valueExists(String section, String ident) {
    CItemArray a = readSection(section);
    return (a != null && a.getValue(ident) != null);
  }

  /**
   * Tests if section exists in this ini file.
   * @param section teh name of the tested section.
   * @result <code>TRUE</code> if section exists, <code>FALSE</code> otherwise.
   */
  public boolean isSectionExists(String section) {
    return (readSection(section) != null);
  }

  /**
   * Reads line from the stream.
   * @param stream an input stream.
   * @result a line string or <code>null</code> if end of stream archived.
   */
  private String readLine(InputStream stream) throws IOException {
    String line = "";
    int c;
    do {
      c = stream.read();
      if (c >= 0 && c != '\n' && c != '\r')
        line += (char) c;
    } while (c >= 0 && c != '\n');
    return (c >= 0 || line.length() > 0)? line.trim(): null;
  }

  /**
   * Splits a line into three parts: key, value and commet.
   * @param line a string line to split.
   * @result an array contained three splitted parts.
   */
  private String[] splitLine(String line) {
    String[] result = new String[3];
    int part = 0;
    boolean inString = false;
    result[0] = "";
    result[1] = "";
    result[2] = "";
    for (int i = 0; i < line.length(); i++) {
      char c = line.charAt(i);
      if (c == '=' && part == 0)
        part = 1;
      else if (c == ';' && !inString && part < 2)
        part = 2;
      else {
        if (c == '\"' && part == 1)
          inString = !inString;
        result[part] += c;
      }
    }
    return result;
  }

  /**
   * Loads the content from the specified input stream.
   * @param stream the input stream to read the ini file.
   */
  public void loadFromStream(InputStream stream) throws IOException {
    sections.clear();
    String s = readLine(stream);
    if (s == null) return;
    CItemArray section = new CItemArray();
    sections.putValue("", section);
    while (s != null) {
      if (s.startsWith("[")) {
        s = s.substring(1);
        if (s.endsWith("]"))
          s = s.substring(0, s.length()-1);
        section = new CItemArray();
        sections.putValue(s, section);
      } else {
        String[] parts = splitLine(s);
        section.putValue(parts[0].toUpperCase(), s);
      }
      s = readLine(stream);
    }
  }

  /**
   * Gets specified section from this ini file.
   * @param section teh name of the section.
   * @result the section with specified name.
   */
  private CItemArray readSection(String section) {
    return (CItemArray) sections.getValue(section);
  }

  /**
   * Gets list of all sections of this ini file.
   * @result the list of all sections.
   */
  private CItemArray readSections() {
    return (CItemArray) sections;
  }

  /**
   * Converts a string from escape format limited
   * with quotes into oridinary (local) presentation.
   * @param s a string in escape format.
   * @result a result string in ordinary (local) presentation.
   */
  public String unwrapString(String s) {
    s = s.trim();
    if (s.startsWith("\"") && s.length() > 0) {
      String result = "";
      for (int i = 1; i < s.length(); i++) {
        if (s.charAt(i) == '\\')
          i++;
        if (s.charAt(i) != '\"' || s.charAt(i-1) == '\\')
          result += s.charAt(i);
      }
      return result;
    }
    return s;
  }

  /**
   * Reads string value from this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param def a default value, if this key is not exists.
   * @result a string value to read from ini file, or default value
   *   if the key is not exists.
   */
  public String readString(String section, String ident, String def) {
    CItemArray a = (CItemArray) sections.getValue(section);
    if (a == null) return def;
    String line = (String) a.getValue(ident);
    if (line == null) return def;
    String[] parts = splitLine(line);
    return unwrapString(parts[1]);
  }

  /**
   * Renames or reload this ini file.
   * @param fileName the new file name.
   * @param reload if <code>TRUE</code> the file will be reloaded with new
   *   contents, otherwise it will be stored in the new file with old contents.
   */
  public void rename(String fileName, boolean reload) throws IOException {
    rename(new File(fileName), reload);
  }

  /**
   * Renames or reload this ini file.
   * @param file object the new file name.
   * @param reload if <code>TRUE</code> the file will be reloaded with new
   *   contents, otherwise it will be stored in the new file with old contents.
   */
  public void rename(File file, boolean reload) throws IOException {
    rename(file.toURL(), reload);
  }

  /**
   * Renames or reload this ini file.
   * @param url a new url of the file name.
   * @param reload if <code>TRUE</code> the file will be reloaded with new
   *   contents, otherwise it will be stored in the new file with old contents.
   */
  public void rename(URL url, boolean reload) throws IOException {
    this.url = url;
    if (reload) loadFromStream(url.openStream());
  }

  /**
   * Writes a string line into output stream.
   * @param stream an output stream.
   * @param line a string line to write in the stream.
   */
  private void writeLine(OutputStream stream, String line) throws IOException {
    stream.write(line.getBytes());
  }

  /**
   * Saves ini file into output stream.
   * @param ouput stream to save this ini file.
   */
  public void saveToStream(OutputStream stream) throws IOException {
    boolean first = true;
    for(int i=0; i<sections.size(); i++) {
      String s = (String) sections.getKey(i);
      String line = "[" + s + "]";
      if (!line.equals("[]")) {
        if (!first) line = "\r\n" + line;
        writeLine(stream, line);
        first  = false;
      }
      CItemArray a = (CItemArray) sections.getValue(i);
      for(int j=0; j<a.size(); j++) {
        line = (String) a.getValue(j);
        if (!first) line = "\r\n" + line;
        writeLine(stream, line);
        first  = false;
      }
    }
  }

  /**
   * Flushes this ini file to disk file.
   */
  public void flush() throws IOException {
    saveToStream(createOutputStream());
  }

  /**
   * Converts a string from ordinary into Pescape format
   * limited with quotes.
   * @param s a string in ordinary (local) presentation.
   * @result a result string in escape format.
   */
  private String wrapString(String s) {
    if (s.indexOf(' ') >= 0 || s.indexOf('\t') >= 0
      || s.indexOf('\"') >= 0 || s.indexOf(';') >= 0) {
      String result = "\"";
      for (int i = 0; i < s.length(); i++) {
        if (s.charAt(i) == '\"' || s.charAt(i) == '\\')
          result += '\\';
        result += s.charAt(i);
      }
      return result + "\"";
    }
    return s;
  }

  /**
   * Writes string value with specified key into the section in this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param value a value of the key.
   */
  public void writeString(String section, String ident, String value) {
    CItemArray a = (CItemArray) sections.getValue(section);
    if (a == null)
      a = addSection(section);
    String line = (String) a.getValue(ident);
    if (line == null) {
      line = ident + "=" + wrapString(value);
    } else {
      String[] parts = splitLine(line);
      line = parts[0] + "=" + wrapString(value);
      if (parts[2].length() > 0)
        line += ";" + parts[2];
    }
    a.putValue(ident, line);
  }

  /**
   * Reads integer value from this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param def a default value, if this key is not exists.
   * @result an integer value read from ini file, or default value
   *   if the key is not exists.
   */
  public int readInteger(String section, String ident, int def) {
    String s = readString(section, ident, new Integer(def).toString());
    try {
      return new Integer(s).intValue();
    }
    catch (Exception e) {
      return def;
    }
  }

  /**
   * Writes integer value with specified key into the section in this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param value a value of the key.
   */
  public void writeInteger(String section, String ident, int value) {
    writeString(section, ident, new Integer(value).toString());
  }

  /**
   * Reads boolean value from this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param def a default value, if this key is not exists.
   * @result a boolean value read from ini file, or default value
   *   if the key is not exists.
   */
  public boolean readBool(String section, String ident, boolean def) {
    return (readInteger(section, ident, (def)? 1: 0) != 0);
  }

  /**
   * Writes boolean value with specified key into the section in this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param value a value of the key.
   */
  public void writeBool(String section, String ident, boolean value) {
    writeString(section, ident, (value)? "1": "0");
  }

  /**
   * Reads float value from this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param def a default value, if this key is not exists.
   * @result a float value read from ini file, or default value
   *   if the key is not exists.
   */
  public double readFloat(String section, String ident, double def) {
    String s = readString(section, ident, new Double(def).toString());
    try {
      return new Double(s).doubleValue();
    }
    catch (Exception e) {
      return def;
    }
  }

  /**
   * Writes float value with specified key into the section in this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param value a value of the key.
   */
  public void writeFloat(String section, String ident, double value) {
    writeString(section, ident, new Double(value).toString());
  }

  /**
   * Reads dateTime value from this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param def a default value, if this key is not exists.
   * @result a dateTime value read from ini file, or default value
   *   if the key is not exists.
   */
  public Date readDateTime(String section, String ident, Date def) {
    String s = readString(section, ident, def.toString());
    try {
      return new SimpleDateFormat().parse(s);
    }
    catch (Exception e) {
      return def;
    }
  }

  /**
   * Writes dateTime value with specified key into the section in this ini file.
   * @param section the name of the section.
   * @param ident the key name of the value.
   * @param value a value of the key.
   */
  public void writeDateTime(String section, String ident, Date value) {
    writeString(section, ident, new SimpleDateFormat().format(value));
  }

  /**
   * Special class for storing inifile items.
   */
  private class CItemArray extends ArrayList {
    private class CItemHolder {
      public String key;
      public Object value;

      public CItemHolder(String key, Object value) {
        this.key = key;
        this.value = value;
      }
    }

    private String normalizeKey(String key) {
      boolean last = false;
      String s = "";
      for (int i=0; i<key.length(); i++) {
        if (!last || (key.charAt(i) != ' ' && key.charAt(i) != '\t'))
          s += key.charAt(i);
        last = (key.charAt(i) == ' ' || key.charAt(i) == '\t');
      }
      return s;
    }

    public String getKey(int index) {
      return ((CItemHolder) get(index)).key;
    }

    public Object getValue(int index) {
      return ((CItemHolder) get(index)).value;
    }

    public Object getValue(String key) {
      key = normalizeKey(key);
      for (int i=0; i<size(); i++) {
        CItemHolder current = ((CItemHolder) get(i));
        if (current.key.equalsIgnoreCase(key))
          return current.value;
      }
      return null;
    }

    public void putValue(String key, Object value) {
      key = normalizeKey(key);
      if (key.length() > 0) {
        for (int i=0; i<size(); i++) {
          CItemHolder current = ((CItemHolder) get(i));
          if (current.key.equalsIgnoreCase(key)) {
            current.value = value;
            set(i, current);
            return;
          }
        }
      }
      add(new CItemHolder(key, value));
    }

    public void removeValue(int index) {
      remove(index);
    }

    public void removeValue(String key) {
      key = normalizeKey(key);
      for (int i=0; i<size(); i++) {
        CItemHolder current = ((CItemHolder) get(i));
        if (current.key.equalsIgnoreCase(key)) {
          remove(i);
          return;
        }
      }
    }
  }

  /**
   * The main procedure for test purposes.
   * @param args the command line arguments.
   */
  public static void main(String[] args) {
    try {
      CIniFile iniFile = new CIniFile("test.ini");
      boolean boolVal = iniFile.readBool("Main", "BoolVal", false);
      System.out.println("BoolVal = " + boolVal);
      iniFile.writeBool("Main", "BoolVal", !boolVal);
      System.out.println("Param1 = " + iniFile.readString("Other X", "Param1", ""));
      System.out.println("Param2 = " + iniFile.readInteger("Other X", "param2", 123));
      System.out.println("Param3 = " + iniFile.readDateTime("Other X", "param3", new Date()));

      iniFile.writeString("Other x", "param1", "..\"xxx\"...");
      iniFile.writeDateTime("Other x", "param3", new Date());
      iniFile.flush();
    }
    catch (Exception e) {
      System.out.println("Ooopps..");
      System.out.println(e.getMessage());
    }
  }

}