/**
 * Title:        Comedia Utils
 * Description:  Project contains some general purpose non-visual beans.
 * Beans do not require any special libraies.
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.util;

/**
 * Implements regular expressions algorithms.
 * It allows unix grep-like pattern comparisons, for instance:
 * <ul>
 * <li>? - Matches any single characer
 * <li>* - Matches any contiguous characters
 * <li>[abc] - Matches a or b or c at that position
 * <li>[^abc] - Matches anything but a or b or c at that position
 * <li>[!abc] - Ditto
 * <li>[a-e] - Matches a through e at that position
 * </ul>
 * Usage examples:
 * <pre>
 * System.out.println("Match result: "
 *   + CRegExp.isMatch("[A-Z,_]*[A-Z,0-9,_]*0?7*", "LAB_001"));
 * </pre>
 */
public class CRegExp {
  /**
   * The match defines.
   */
  private final static int MATCH_PATTERN = 6;
  private final static int MATCH_LITERAL = 5;
  private final static int MATCH_RANGE = 4;
  private final static int MATCH_ABORT = 3;
  private final static int MATCH_END = 2;
  private final static int MATCH_VALID = 1;

  /**
   * The pattern defines.
   */
  private final static int PATTERN_VALID = 0;
  private final static int PATTERN_ESC = -1;
  private final static int PATTERN_RANGE = -2;
  private final static int PATTERN_CLOSE = -3;
  private final static int PATTERN_EMPTY = -4;

  /**
   * The character defines
   */
  private final static char MATCH_CHAR_SINGLE = '?';
  private final static char MATCH_CHAR_KLEENE_CLOSURE = '*';
  private final static char MATCH_CHAR_RANGE_OPEN = '[';
  private final static char MATCH_CHAR_RANGE = '-';
  private final static char MATCH_CHAR_RANGE_CLOSE = ']';
  private final static char MATCH_CHAR_CARAT_NEGATE = '^';
  private final static char MATCH_CHAR_EXCLAMATION_NEGATE = '!';

  /**
   * Matches a text after star (any sequence).
   * @param pattern a regular expression pattern.
   * @param text a text to match.
   * @result a position of stopping match.
   */
  public static boolean isMatch(String pattern, String text) {
    return matche(pattern, text) == MATCH_VALID;
  }

  /**
   * Matches a text to regular expression.
   * @param pattern a regular expression pattern.
   * @param text a text to match.
   * @result a match result (see predefined constants).
   */
  private static int matche(String pattern, String text) {
    int rangeStart, rangeEnd, p = 0, t = 0, plen, tlen, result = 0;
    boolean invert, memberMatch, loop;

    plen = pattern.length();
    tlen = text.length();
    while ((result == 0) && (p < plen)) {
      if (t >= tlen) {
        if ((pattern.charAt(p) == MATCH_CHAR_KLEENE_CLOSURE) && (p >= plen-1))
          return MATCH_VALID;
        else
          return MATCH_ABORT;
      } else {
        switch (pattern.charAt(p)) {
          case MATCH_CHAR_KLEENE_CLOSURE:
            result = matchAfterStar(pattern.substring(p), text.substring(t));
            break;
          case MATCH_CHAR_RANGE_OPEN:
            {
              p++;
              invert = false;
              if ((pattern.charAt(p) == MATCH_CHAR_EXCLAMATION_NEGATE)
                || (pattern.charAt(p) == MATCH_CHAR_CARAT_NEGATE)) {
                invert = true;
                p++;
              }
              if (pattern.charAt(p) == MATCH_CHAR_RANGE_CLOSE)
                return MATCH_PATTERN;
              memberMatch = false;
              loop = true;
              while ((loop) && (pattern.charAt(p) != MATCH_CHAR_RANGE_CLOSE)) {
                rangeStart = rangeEnd = p;
                p++;
                if (p >= plen) return MATCH_PATTERN;
                if (pattern.charAt(p) == MATCH_CHAR_RANGE) {
                  p++;
                  rangeEnd = p;
                  if ((p >= plen)
                    || (pattern.charAt(rangeEnd) == MATCH_CHAR_RANGE_CLOSE))
                    return MATCH_PATTERN;
                  p++;
                }
                if (p >= plen) return MATCH_PATTERN;
                if (rangeStart < rangeEnd) {
                  if ((text.charAt(t) >= pattern.charAt(rangeStart)) &&
                    (text.charAt(t) <= pattern.charAt(rangeEnd))) {
                    memberMatch = true;
                    loop = false;
                  }
                } else {
                  if ((text.charAt(t) >= pattern.charAt(rangeEnd))
                    && (text.charAt(t) <= pattern.charAt(rangeStart))) {
                    memberMatch = true;
                    loop = false;
                  }
                }
              }
              if ((invert && memberMatch) || (!(invert || memberMatch)))
                return MATCH_RANGE;
              if (memberMatch)
                while ((p < plen) && (pattern.charAt(p) != MATCH_CHAR_RANGE_CLOSE))
                  p++;
              if (p >= plen) return MATCH_PATTERN;
                break;
            } // MATCH_CHAR_RANGE_OPEN:
          default:
            if (pattern.charAt(p) != MATCH_CHAR_SINGLE)
            if (pattern.charAt(p) != text.charAt(t)) result = MATCH_LITERAL;
        } // case
      }
      p++;
      t++;
    } // while
    if (result == 0) {
      if (t < tlen) return MATCH_END;
      else return MATCH_VALID;
    } else return result;
  }

  /**
   * Matches a text after star (any sequence).
   * @param pattern a regular expression pattern.
   * @param text a text to match.
   * @result a match result (see predefined constants).
   */
  private static int matchAfterStar(String pattern, String text) {
    int p = 0, t = 0, plen, tlen, result = 0;

    if (matche(pattern.substring(1), text) == MATCH_VALID)
      return MATCH_VALID;

    plen = pattern.length();
    tlen = text.length();
    while ((t < tlen) && (p < plen) && ((pattern.charAt(p) == MATCH_CHAR_SINGLE)
      || (pattern.charAt(p) == MATCH_CHAR_KLEENE_CLOSURE))) {
      if (pattern.charAt(p) == MATCH_CHAR_SINGLE) t++;
      p++;
    }
    if (t >= tlen) return MATCH_ABORT;
    if (p >= plen) return MATCH_VALID;
    do {
      if ((pattern.charAt(p) == text.charAt(t))
        || (pattern.charAt(p) == MATCH_CHAR_RANGE_OPEN)) {
        pattern = pattern.substring(p);
        text = text.substring(t);
        plen = pattern.length();
        tlen = text.length();
        p = t = 0;
        result = matche(pattern, text);
        if (result == MATCH_VALID) return result;
        else result = 0;
      }
      t++;
      if (t >= tlen) return MATCH_ABORT;
    } while (result == 0);
    return (result != 0)? result : MATCH_ABORT;
  }

  /**
   * Runs the test for this class.
   * @param args a command line arguments.
   */
  public static void main(String args[]) {
    System.out.println("The match result (1): "
      + CRegExp.isMatch("BLA", "BLA"));
    System.out.println("The match result (2): "
      + CRegExp.isMatch("A*", "ABBA"));
    System.out.println("The match result (3): "
      + CRegExp.isMatch("[A-Z,_]*[A-Z,0-9,_]*0?7*", "AGENT_007"));
  }

}