/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import javax.swing.*;
import java.awt.*;
import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.Iterator;

/**
 * <p>Title: uifactory</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2001</p>
 * <p>Company: Abacus Research</p>
 *
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */

public class DesignCockpitPreferences {
    final SuperDesignCockpit theDesignCockpit;
    public static final int METAL_LAF = 0;   // javax.swing.plaf.metal.MetalLookAndFeel
    public static final int MOTIF_LAF = 1;   // com.sun.java.swing.plaf.motif.MotifLookAndFeel
    public static final int MAC_LAF = 2;     // com.sun.java.swing.plaf.mac.MacLookAndFeel
    public static final int WINDOWS_LAF = 3; // com.sun.java.swing.plaf.windows.WindowsLookAndFeel

    public static final int DEFAULT_LOOK = 0;   //
    public static final int PLASTICXP_LOOK = 1;   //
    public static final int PLASTIC3D_LOOK = 2;  //
    public static final int PLASTIC_LOOK = 3;   //

    private static final int MINIMUM_WINDOW_SIZE = 100; // the minimum width/height of a window
    public String sFontChooserName = "ch.abacus.designcockpit.ide.IDEObjectFontChooser";
    public String sLookAndFeelName = "com.sun.java.swing.plaf.windows.WindowsLookAndFeel";
    public String sGenLookFeelName = "Default";
    public String sCodeGenerationDocument = "metadata.meta";
    public String sAuthor = "The unknown programmer";
    public String theWorkspace = "";
    public String sTransientDirectory = "transient";
    public String sStickyDelimiters = "";
    public String sDelimiters = "";
    public int iIndentation = 0;
    public int iMaxLineLength = 0;
    public int iSizeBoxHeight = 4;
    public int iSizeBoxWidth = 4;
    public int iViewDisplayerGridX = 8;
    public int iViewDisplayerGridY = 8;
    private int iWidthColumnOne;
    private int iWidthColumnTwo;
    private int iSplitColumnOne;
    private int iSplitColumnTwo;
    private int iSplitColumnThree;
    private String sGeneratedCodeLookFeel;
    public String thePreferencesDocument = null;
    public String sWorkingDirectory = "";
    public boolean bDisplayGrid = true;
    private boolean bFrameMaximized = true; // assume full screen
    private Rectangle framePlacement = new Rectangle(100, 100, 800, 600); // assume reasonable size
    private String sRepositPath = null;
    private String sAdditionalClasspath = null;

    //#ALEX, the Background grid is something specific, I cannot see this in any other IDE
    public boolean bDisplayBackgroundGrid = false;
    public boolean bSnapToGrid = true;

    public DesignCockpitPreferences(int theMode, SuperDesignCockpit designCockpit) {
        theDesignCockpit = designCockpit;
        theWorkspace = "default";
        thePreferencesDocument = theWorkspace + ".conf";
        cfInit();
        load(theMode, theDesignCockpit);
    }

    public DesignCockpitPreferences(int theMode, SuperDesignCockpit designCockpit, String sWorkspace) {
        theDesignCockpit = designCockpit;
        theWorkspace = sWorkspace;
        thePreferencesDocument = theWorkspace + ".conf";
        cfInit();
        load(theMode, theDesignCockpit);
    }

    public void cfInit() {
        // Special initialization for code formatter.
        com.ibm.cf.CodeFormatter theFormatter = new com.ibm.cf.CodeFormatter();
        iIndentation = theFormatter.getIndentationStep();
        iMaxLineLength = theFormatter.getMaxLineLength();
        sStickyDelimiters = theFormatter.getStickyDelimiters();
        sDelimiters = theFormatter.getDelimiters();
    }

    public boolean load(int theMode, SuperDesignCockpit theDesignCockpit) {
        if (theDesignCockpit.isCustomizing()) {
            setDefaultWindowSizes();
            theDesignCockpit.theCustomizer.loadPreferences(this);
        } else {
            // Now load the XML document and do the settings.
            electric.xml.Document xmlPreferencesDocument = null;
            try {
                File fileInput = new File(thePreferencesDocument);
                //   Load it as an XML document.
                if (fileInput.exists())
                    xmlPreferencesDocument = new electric.xml.Document(fileInput);
                else
                    return true;
            } catch (electric.xml.ParseException e1) {
                e1.printStackTrace();
                return false;
            }
            electric.xml.Element xmlRoot = xmlPreferencesDocument.getRoot();
            electric.xml.Element xmlFontChooser = xmlRoot.getElement("FONT_CHOOSER");
            if (xmlFontChooser != null)
                sFontChooserName = xmlFontChooser.getTextString();
            electric.xml.Element xmlLookAndFeel = xmlRoot.getElement("PLAF");
            if (xmlLookAndFeel != null)
                sLookAndFeelName = xmlLookAndFeel.getTextString();
            electric.xml.Element xmlMetafile = xmlRoot.getElement("Metafile");
            if (xmlMetafile != null)
                sCodeGenerationDocument = xmlMetafile.getTextString();
            electric.xml.Element xmlAuthor = xmlRoot.getElement("Author");
            if (xmlAuthor != null)
                sAuthor = xmlAuthor.getTextString();
            electric.xml.Element xmlTransient = xmlRoot.getElement("Transient");
            if (xmlTransient != null)
                sTransientDirectory = xmlTransient.getTextString();
            electric.xml.Element xmlWorking = xmlRoot.getElement("Working");
            if (xmlWorking != null)
                sWorkingDirectory = xmlWorking.getTextString();
            electric.xml.Element xmlSticky = xmlRoot.getElement("StickyDelims");
            if (xmlSticky != null)
                sStickyDelimiters = xmlSticky.getTextString();
            electric.xml.Element xmlDelims = xmlRoot.getElement("Delims");
            if (xmlDelims != null)
                sDelimiters = xmlDelims.getTextString();
            electric.xml.Element xmlIndentation = xmlRoot.getElement("Indentation");
            if (xmlIndentation != null)
                iIndentation = (new Integer(xmlIndentation.getTextString())).intValue();
            electric.xml.Element xmlMaxLineLength = xmlRoot.getElement("MaxLineLength");
            if (xmlMaxLineLength != null)
                iMaxLineLength = (new Integer(xmlMaxLineLength.getTextString())).intValue();
            electric.xml.Element xmlSizeBoxHeight = xmlRoot.getElement("SizeBoxHeight");
            if (xmlSizeBoxHeight != null)
                iSizeBoxHeight = (new Integer(xmlSizeBoxHeight.getTextString())).intValue();
            electric.xml.Element xmlSizeBoxWidth = xmlRoot.getElement("SizeBoxWidth");
            if (xmlSizeBoxWidth != null)
                iSizeBoxWidth = (new Integer(xmlSizeBoxWidth.getTextString())).intValue();

            electric.xml.Element xmlGridX = xmlRoot.getElement("GridWidth");
            if (xmlGridX != null)
                iViewDisplayerGridX = (new Integer(xmlGridX.getTextString())).intValue();

            electric.xml.Element xmlGridY = xmlRoot.getElement("GridHeight");
            if (xmlGridY != null)
                iViewDisplayerGridY = (new Integer(xmlGridY.getTextString())).intValue();

            readFramePosition(xmlRoot);
            setDefaultWindowSizes();

            electric.xml.Element xmlColumnOneWidth = xmlRoot.getElement("ColumnOneWidth");
            if (xmlColumnOneWidth != null)
                iWidthColumnOne = getSizeOrMinimum(xmlColumnOneWidth.getTextString());

            electric.xml.Element xmlColumnTwoWidth = xmlRoot.getElement("ColumnTwoWidth");
            if (xmlColumnTwoWidth != null)
                iWidthColumnTwo = getSizeOrMinimum(xmlColumnTwoWidth.getTextString());

            //electric.xml.Element xmlColumnThreeWidth = xmlRoot.getElement("ColumnThreeWidth");
            //if (xmlColumnThreeWidth != null)
            //    iWidthColumnThree = getSizeOrMinimum(xmlColumnThreeWidth.getTextString());

            electric.xml.Element xmlColumnOneSplit = xmlRoot.getElement("ColumnOneSplit");
            if (xmlColumnOneSplit != null)
                iSplitColumnOne = getSizeOrMinimum(xmlColumnOneSplit.getTextString());

            electric.xml.Element xmlColumnTwoSplit = xmlRoot.getElement("ColumnTwoSplit");
            if (xmlColumnTwoSplit != null)
                iSplitColumnTwo = getSizeOrMinimum(xmlColumnTwoSplit.getTextString());

            electric.xml.Element xmlColumnThreeSplit = xmlRoot.getElement("ColumnThreeSplit");
            if (xmlColumnThreeSplit != null)
                iSplitColumnThree = getSizeOrMinimum(xmlColumnThreeSplit.getTextString());

            if ((theDesignCockpit != null) && (theDesignCockpit.theProgramMode == SuperDesignCockpit.JAVABUILDER))
            {
                electric.xml.Element xmlGeneratedCodeLookFeel = xmlRoot.getElement("GeneratedCodeLookFeel");
                if (xmlGeneratedCodeLookFeel != null)
                    sGeneratedCodeLookFeel = xmlGeneratedCodeLookFeel.getTextString();
            }

            electric.xml.Element xmlDisplayGrid = xmlRoot.getElement("DisplayGrid");
            if (xmlDisplayGrid != null)
                bDisplayGrid = xmlDisplayGrid.getBoolean();

            electric.xml.Element xmlSnapToGrid = xmlRoot.getElement("SnapToGrid");
            if (xmlSnapToGrid != null)
                bSnapToGrid = xmlSnapToGrid.getBoolean();

            electric.xml.Element xmlRepositPath = xmlRoot.getElement("RepositoryPath");
            if (xmlRepositPath != null)
                sRepositPath = xmlRepositPath.getTextString();
            electric.xml.Element xmlAdditionalClassPath = xmlRoot.getElement("AdditionalClassPath");
            if (xmlAdditionalClassPath != null)
                sAdditionalClasspath= xmlAdditionalClassPath.getTextString();


            // Load the MRU...
            electric.xml.Element xmlMruRoot = xmlRoot.getElement("MRU");
            if (xmlMruRoot != null) {
                int iFiles = xmlMruRoot.getElements().size();

                for (int iFile = iFiles; iFile >= 1; iFile--) {
                    electric.xml.Element xmlMruFile = xmlMruRoot.getElement(new Integer(iFile).toString());

                    String sFile = xmlMruFile.getTextString();
                    theDesignCockpit.mruList.fileClosed(sFile); // inform the mru list that this was a file recently closed
                }
            }
        }
        return true;
    }

    private void setDefaultWindowSizes() {
        Dimension frameSize;
        if (bFrameMaximized) {
            Toolkit kit = Toolkit.getDefaultToolkit();
            frameSize = kit.getScreenSize();
        } else {
            frameSize = framePlacement.getSize();
        }

        iWidthColumnOne = frameSize.width /4;
        iWidthColumnTwo = frameSize.width /2 + iWidthColumnOne;
        iSplitColumnOne = iSplitColumnTwo = iSplitColumnThree = frameSize.height /2;
    }


    public boolean save(SuperDesignCockpit theDesignCockpit) {
 //iWidthColumnOne = theDesignCockpit.paneLeft.getDividerLocation(); //theObjectTreeViewPane.getSize().getWidth();
        iWidthColumnTwo = theDesignCockpit.paneRight.getDividerLocation(); //theViewDisplayerScrollPane.getSize().getWidth();
        //iWidthColumnThree = (int) theDesignCockpit.thePropertyTable.getSize().getWidth();
        iSplitColumnOne = theDesignCockpit.paneOne.getDividerLocation(); //theObjectTreeViewPane.getSize().getHeight();
        iSplitColumnTwo = theDesignCockpit.paneTwo.getDividerLocation(); //theViewDisplayer.getSize().getHeight();
        iSplitColumnThree = theDesignCockpit.paneThree.getDividerLocation(); //thePropertyTableScrollPane.getSize().getHeight();

        if (theDesignCockpit.isCustomizing()) {
            theDesignCockpit.theCustomizer.savePreferences(this);
        } else {
            try {
                File fileOutput = new File(thePreferencesDocument);
                PrintWriter pwOutput = new PrintWriter(new FileWriter(fileOutput));

                electric.xml.Document xmlPreferencesDocument = new electric.xml.Document();
                electric.xml.Element xmlRoot = new electric.xml.Element("Preferences");
                electric.xml.Element xmlValue;
                xmlPreferencesDocument.addChild(xmlRoot);
                if (sLookAndFeelName != null) {
                    xmlValue = new electric.xml.Element("PLAF");
                    xmlValue.setText(sLookAndFeelName);
                    xmlRoot.addElement(xmlValue);
                }
                if (sCodeGenerationDocument != null) {
                    xmlValue = new electric.xml.Element("Metafile");
                    xmlValue.setText(sCodeGenerationDocument);
                    xmlRoot.addElement(xmlValue);
                }
                if (sAuthor != null) {
                    xmlValue = new electric.xml.Element("Author");
                    xmlValue.setText(sAuthor);
                    xmlRoot.addElement(xmlValue);
                }
                if (sTransientDirectory != null) {
                    xmlValue = new electric.xml.Element("Transient");
                    xmlValue.setText(sTransientDirectory);
                    xmlRoot.addElement(xmlValue);
                }
                if (sWorkingDirectory != null) {
                    xmlValue = new electric.xml.Element("Working");
                    xmlValue.setText(sWorkingDirectory);
                    xmlRoot.addElement(xmlValue);
                }
                if (sStickyDelimiters != null) {
                    xmlValue = new electric.xml.Element("StickyDelims");
                    xmlValue.setText(sStickyDelimiters);
                    xmlRoot.addElement(xmlValue);
                }
                if (sDelimiters != null) {
                    xmlValue = new electric.xml.Element("Delims");
                    xmlValue.setText(sDelimiters);
                    xmlRoot.addElement(xmlValue);
                }
                if (sGenLookFeelName!= null) {
                    xmlValue = new electric.xml.Element("sGenLookFeelName");
                    xmlValue.setText(sGenLookFeelName);
                    xmlRoot.addElement(xmlValue);
                }
                if (sRepositPath!= null) {
                    xmlValue = new electric.xml.Element("RepositoryPath");
                    xmlValue.setText(sRepositPath);
                    xmlRoot.addElement(xmlValue);
                }
                if (sAdditionalClasspath!= null) {
                    xmlValue = new electric.xml.Element("AdditionalClassPath");
                    xmlValue.setText(sAdditionalClasspath);
                    xmlRoot.addElement(xmlValue);
                }


                xmlValue = new electric.xml.Element("Indentation");
                xmlValue.setText((new Integer(iIndentation)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("MaxLineLength");
                xmlValue.setText((new Integer(iMaxLineLength)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("SizeBoxHeight");
                xmlValue.setText((new Integer(iSizeBoxHeight)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("SizeBoxWidth");
                xmlValue.setText((new Integer(iSizeBoxWidth)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("GridHeight");
                xmlValue.setText((new Integer(iViewDisplayerGridY)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("GridWidth");
                xmlValue.setText((new Integer(iViewDisplayerGridX)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("ColumnOneWidth");
                xmlValue.setText((new Integer(iWidthColumnOne)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("ColumnTwoWidth");
                xmlValue.setText((new Integer(iWidthColumnTwo)).toString());
                xmlRoot.addElement(xmlValue);
                //xmlValue = new electric.xml.Element("ColumnThreeWidth");
                //xmlValue.setText((new Integer(iWidthColumnThree)).toString());
                //xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("ColumnOneSplit");
                xmlValue.setText((new Integer(iSplitColumnOne)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("ColumnTwoSplit");
                xmlValue.setText((new Integer(iSplitColumnTwo)).toString());
                xmlRoot.addElement(xmlValue);
                xmlValue = new electric.xml.Element("ColumnThreeSplit");
                xmlValue.setText((new Integer(iSplitColumnThree)).toString());
                xmlRoot.addElement(xmlValue);

                // Write the Frame position...
                writeFramePosition(xmlRoot);

                xmlValue = new electric.xml.Element("DisplayGrid");
                xmlValue.setBoolean(theDesignCockpit.thePreferences.bDisplayGrid);
                xmlRoot.addElement(xmlValue);

                xmlValue = new electric.xml.Element("SnapToGrid");
                xmlValue.setBoolean(theDesignCockpit.thePreferences.bSnapToGrid);
                xmlRoot.addElement(xmlValue);

                // Write the MRU List
                electric.xml.Element xmlMru = new electric.xml.Element("MRU");
                xmlRoot.addChild(xmlMru);

                Iterator li = theDesignCockpit.mruList.getArrayList().iterator();
                String sFile;
                int iPosition = 1;
                while (li.hasNext()) {
                    sFile = (String) li.next();

                    xmlValue = new electric.xml.Element(new Integer(iPosition).toString());
                    xmlValue.setText(sFile);
                    xmlMru.addElement(xmlValue);
                    iPosition++;
                }

                // Write the document.
                xmlPreferencesDocument.write(pwOutput);
                pwOutput.close();
            } catch (java.io.IOException e) {
                return false;
            }
        }
        return true;
    }

    private void writeFramePosition(electric.xml.Element xmlRoot) {
        boolean bIsFrameMaximized = theDesignCockpit.theFramePlacement.isMaximized();
        Dimension frameSize = theDesignCockpit.theFramePlacement.getNormalSize();
        Point frameLocation = theDesignCockpit.theFramePlacement.getNormalLocation();

        electric.xml.Element xmlValue;
        electric.xml.Element xmlFramePosition = new electric.xml.Element("FramePlacement");

        xmlValue = new electric.xml.Element("Maximized");
        xmlValue.setText((new Boolean(bIsFrameMaximized)).toString());
        xmlFramePosition.addElement(xmlValue);

        xmlValue = new electric.xml.Element("Height");
        xmlValue.setText((new Integer(frameSize.height)).toString());
        xmlFramePosition.addElement(xmlValue);

        xmlValue = new electric.xml.Element("Width");
        xmlValue.setText((new Integer(frameSize.width)).toString());
        xmlFramePosition.addElement(xmlValue);

        xmlValue = new electric.xml.Element("Top");
        xmlValue.setText((new Integer(frameLocation.y)).toString());
        xmlFramePosition.addElement(xmlValue);

        xmlValue = new electric.xml.Element("Left");
        xmlValue.setText((new Integer(frameLocation.x)).toString());
        xmlFramePosition.addElement(xmlValue);

        xmlRoot.addElement(xmlFramePosition);
    }

    private void readFramePosition(electric.xml.Element xmlRoot) {
        try {
            electric.xml.Element xmlFramePlacement = xmlRoot.getElement("FramePlacement");
            if (xmlFramePlacement != null) {
                bFrameMaximized = xmlFramePlacement.getBoolean("Maximized");
                framePlacement.height = xmlFramePlacement.getInt("Height");
                framePlacement.width = xmlFramePlacement.getInt("Width");
                framePlacement.y = xmlFramePlacement.getInt("Top");
                framePlacement.x = xmlFramePlacement.getInt("Left");
            }
        } catch (NumberFormatException e) {
            // if any errors occur, just use defaults
        }
    }

    public int getPLAF(String sPLAF) {
        if (sPLAF.equals("javax.swing.plaf.metal.MetalLookAndFeel"))
            return METAL_LAF;
        if (sPLAF.equals("com.sun.java.swing.plaf.motif.MotifLookAndFeel"))
            return MOTIF_LAF;
        if (sPLAF.equals("com.sun.java.swing.plaf.mac.MacLookAndFeel"))
            return MAC_LAF;
        if (sPLAF.equals("com.sun.java.swing.plaf.windows.WindowsLookAndFeel"))
            return WINDOWS_LAF;
        return WINDOWS_LAF;
    }

    public String getRepositPath()
    {
       return sRepositPath;
    }

    public void setRepositPath(String sValue)
    {
        sRepositPath=sValue;
    }

    public String getAdditionalClasspath()
    {
       return sAdditionalClasspath;
    }

    public void setAdditionalClasspath(String sValue)
    {
        sAdditionalClasspath=sValue;
    }

    public int getPLAF() {
        return getPLAF(sLookAndFeelName);
    }


    public String getPLAF(int iPLAF) {
        sLookAndFeelName = "com.sun.java.swing.plaf.windows.WindowsLookAndFeel";  // default
        switch (iPLAF) {
            case METAL_LAF:
                sLookAndFeelName = "javax.swing.plaf.metal.MetalLookAndFeel";
                break;
            case MOTIF_LAF:
                sLookAndFeelName = "com.sun.java.swing.plaf.motif.MotifLookAndFeel";
                break;
            case MAC_LAF:
                sLookAndFeelName = "com.sun.java.swing.plaf.mac.MacLookAndFeel";
                break;
        }
        return sLookAndFeelName;
    }

    public void setPLAF(int iChoice, Component topComponent) {
        String sOldLookAndFeelName = sLookAndFeelName;
        sLookAndFeelName = getPLAF(iChoice);
        try {
            UIManager.setLookAndFeel(sLookAndFeelName);
            SwingUtilities.updateComponentTreeUI(topComponent);  // The SuperDesignCockpit or top rendering object
        } catch (Exception evt) {
            JOptionPane.showMessageDialog(null, "Cannot change the look and feel to " + sLookAndFeelName,
                    "Swing Failure", JOptionPane.INFORMATION_MESSAGE);
            sLookAndFeelName = sOldLookAndFeelName;
        }
    }


    public String getGenCodeLookFeel(int iGenCodeLAF) {
        sGenLookFeelName = "Default";  // default
        switch (iGenCodeLAF) {
            case PLASTICXP_LOOK:
                sGenLookFeelName = "PlasticXPLookAndFeel";
                break;
            case PLASTIC3D_LOOK:
                sGenLookFeelName = "Plastic3DLookAndFeel";
                break;
            case PLASTIC_LOOK:
                sGenLookFeelName = "PlasticLookAndFeel";
                break;
        }
        return sGenLookFeelName;
    }

    public int getGenCodeLookFeel() {
        return theDesignCockpit.theCurrentDesignProject.theMetaProject.getGenCodeLookFeel(sGenLookFeelName);
    }

    private int getSizeOrMinimum(String sSize) {
        int iSize = MINIMUM_WINDOW_SIZE;

        try {
            iSize = Integer.parseInt(sSize);
        } catch (NumberFormatException e) {/* leave iSize as the default */
        }

        if (iSize < MINIMUM_WINDOW_SIZE)
            iSize = MINIMUM_WINDOW_SIZE;

        return iSize;
    }

    public int getFrameWidth() {
        return framePlacement.width;
    }

    public int getFrameHeight() {
        return framePlacement.height;
    }

    public int getFrameTop() {
        return framePlacement.y;
    }

    public int getFrameLeft() {
        return framePlacement.x;
    }

    public boolean isFrameMaximized() {
        return bFrameMaximized;
    }

    public void setFrameMaximized(boolean bMaximized) {
        bFrameMaximized = bMaximized;
    }

    public void setFrameWidth (int iWidth) {
        framePlacement.width = iWidth;
    }

    public void setFrameHeight (int iHeight) {
        framePlacement.height = iHeight;
    }
    public void setFrameLeft (int iLeft) {
        framePlacement.x = iLeft;
    }
    public void setFrameTop (int iTop) {
        framePlacement.y = iTop;
    }

    public int getWidthColumnOne() {
        return iWidthColumnOne;
    }

    public int getWidthColumnTwo() {
        return iWidthColumnTwo;
    }

    public int getSplitColumnOne() {
        return iSplitColumnOne;
    }

    public int getSplitColumnTwo() {
        return iSplitColumnTwo;
    }

    public int getSplitColumnThree() {
        return iSplitColumnThree;
    }

    public void setWidthColumnOne(int iWidthColumnOne) {
        this.iWidthColumnOne = iWidthColumnOne;
    }

    public void setWidthColumnTwo(int iWidthColumnTwo) {
        this.iWidthColumnTwo = iWidthColumnTwo;
    }

    public void setSplitColumnOne(int iSplitColumnOne) {
        this.iSplitColumnOne = iSplitColumnOne;
    }

    public void setSplitColumnTwo(int iSplitColumnTwo) {
        this.iSplitColumnTwo = iSplitColumnTwo;
    }

    public void setSplitColumnThree(int iSplitColumnThree) {
        this.iSplitColumnThree = iSplitColumnThree;
    }
}