/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.lib.ui.JATextArea;
import ch.abacus.lib.ui.JATextField;
import ch.abacus.lib.ui.renderer.common.*;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.PrintWriter;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * User: Stuart Booth
 * Date: Jul 22, 2004
 *
 * Encapsulates all interface-related actions together. A single object of this class is instantiated in the IDE.
 * Each Action is exposed as an object that you can access to associate with components.
 */
public class IDEActionManager {
    private final SuperDesignCockpit theCockpit;
    private final HammerImageIconFactory theIconFactory;
    ResourceBundle theResourceBundle = null;

    private AboutAction aboutAction;
    private BackgroundColorChooserAction backgroundColorChooserAction;
    private ForegroundColorChooserAction foregroundColorChooserAction;
    private FontChooserAction fontChooserAction;
    private UndoAction undoAction;
    private NewAction newAction;
    private PasteAction pasteAction;
    private PasteSelectedObjectAction pasteSelectedObjectAction;
    private RedoAction redoAction;
    private SaveAction saveAction;
    private SaveWithCompileAction saveWithCompileAction;
    private OpenAction openAction;
    private SaveAllAction saveAllAction;
    private RunAction runAction;
    private CompileAction compileAction;
    private CutAction cutAction;
    private CopyAction copyAction;
    private DatabaseSelectionAction databaseSelectionAction;
    private ImporterAction importerAction;
    private ProjectPreferencesAction projectPreferencesAction;
    private PointModeAction pointModeAction;
    private SaveAsAction saveAsAction;
    private SaveMetadataAction saveMetadataAction;
    private CloseAction closeAction;
    private ExitAction exitAction;
    private ClassEditorAction classEditorAction;
    private DeleteAction deleteAction;
    private DuplicateObjectNamesAction duplicateObjectNamesAction;
    private LanguageChooserAction languageDeutschAction;
    private LanguageChooserAction languageFrancaisAction;
    private LanguageChooserAction languageItalianAction;
    private LanguageChooserAction languageEnglishAction;
    private MultiObjectAlignmentAction multiObjectAlignmentAction;
    private MultiObjectSizeAction multiObjectSizeAction;
    private InsertHorzSplitterAction insertHorzSplitterAction;
    private InsertVertSplitterAction insertVertSplitterAction;

    public IDEActionManager (SuperDesignCockpit theCockpit) {
        this.theCockpit = theCockpit;
        theIconFactory = new HammerImageIconFactory(theCockpit.theClassLoader);
        theResourceBundle = theCockpit.theLanguageManager.getActionsBundle();

        initializeActions();
    }

    private void initializeActions () {
        ActionInitializer initializer = new ActionInitializer();

        aboutAction = initializer.initAboutAction();
        undoAction = initializer.initUndoAction();
        pasteAction = initializer.initPasteAction();
        pasteSelectedObjectAction = initializer.initPasteSelectedObjectAction();
        redoAction = initializer.initRedoAction();
        saveAction = initializer.initSaveAction();
        cutAction = initializer.initCutAction();
        copyAction = initializer.initCopyAction();
        exitAction = initializer.initExitAction();
        classEditorAction = initializer.initClassEditorAction();
        deleteAction = initializer.initDeleteAction();
        languageDeutschAction = initializer.initLanguageDeutschAction();
        languageFrancaisAction = initializer.initLanguageFrancaisAction();
        languageItalianAction = initializer.initLanguageItalianAction();
        languageEnglishAction = initializer.initLanguageEnglishAction();
        multiObjectAlignmentAction = initializer.initMultiObjectAlignmentAction();
        multiObjectSizeAction = initializer.initMultiObjectSizeAction();
        insertHorzSplitterAction = initializer.initInsertHorzSplitterAction();
        insertVertSplitterAction = initializer.initInsertVertSplitterAction();
        runAction = initializer.initRunAction();

        if (theCockpit.isCustomizing() == false) {
            // These actions are only used if not Customizing...
            saveWithCompileAction = initializer.initSaveWithCompileAction();
            duplicateObjectNamesAction = initializer.initDuplicateObjectNamesAction();
            newAction = initializer.initNewAction();
            saveAsAction = initializer.initSaveAsAction();
            saveMetadataAction = initializer.initSaveMetadataAction();
            openAction = initializer.initOpenAction();
            saveAllAction = initializer.initSaveAllAction();
            closeAction = initializer.initCloseAction();
            databaseSelectionAction = initializer.initDatabaseSelectionAction();
            pointModeAction = initializer.initPointModeAction();
            backgroundColorChooserAction = initializer.initBackgroundColorChooserAction();
            foregroundColorChooserAction = initializer.initForegroundColorChooserAction();
            fontChooserAction = initializer.initFontChooserAction();
            projectPreferencesAction = initializer.initProjectPreferencesAction();

            if (theCockpit.getProgramMode() == SuperDesignCockpit.JAVABUILDER) {
                importerAction = initializer.initImporterAction();
                compileAction = initializer.initCompileAction();
            }
        }

    }

    /**
     * Used to determine if the focus is a component that is text-oriented, meaning cut/copy/paste operations should
     * go straight to the clipboard.<p>
     * If the focus is the menu, special logic is implemented to detect if the focus was thed editor before the menu.
     * If it was, then this function returns true
     *
     * @param theFocus
     * @return true if the focus is a text-oriented component
     */
    private boolean isFocusTextOriented(Component theFocus) {
        return (theFocus instanceof JATextField) ||
               (theFocus instanceof JATextArea) ||
               (theFocus instanceof IDESourceEditor) ||
               (theCockpit.getLastFocussedComponent() instanceof IDESourceEditor);
    }

    private Icon getIcon (String sIconName) {
        if (sIconName == null)
            return null;

        Icon icon = theIconFactory.getHammerImageIcon(theCockpit.sImageDirectory + sIconName);

        //if (icon == null)
        //    theCockpit.theLogFile.doLogEntry("Image not found", sIconName);

        return icon;
    }

    public AboutAction getAboutAction () {
        return aboutAction;
    }

    public BackgroundColorChooserAction getBackgroundColorChooserAction () {
        return backgroundColorChooserAction;
    }

    public ForegroundColorChooserAction getForegroundColorChooserAction () {
        return foregroundColorChooserAction;
    }

    public FontChooserAction getFontChooserAction () {
        return fontChooserAction;
    }

    public UndoAction getUndoAction () {
        return undoAction;
    }

    public NewAction getNewAction () {
        return newAction;
    }

    public PasteAction getPasteAction () {
        return pasteAction;
    }

    public PasteSelectedObjectAction getPasteSelectedObjectAction () {
        return pasteSelectedObjectAction;
    }

    public RedoAction getRedoAction () {
        return redoAction;
    }

    public SaveAction getSaveAction () {
        return saveAction;
    }

    public SaveAsAction getSaveAsAction () {
        return saveAsAction;
    }

    public SaveMetadataAction getSaveMetadataAction () {
        return saveMetadataAction;
    }

    public SaveWithCompileAction getSaveWithCompileAction () {
        return saveWithCompileAction;
    }

    public OpenAction getOpenAction () {
        return openAction;
    }

    public SaveAllAction getSaveAllAction () {
        return saveAllAction;
    }

    public RunAction getRunAction () {
        return runAction;
    }

    public CompileAction getCompileAction () {
        return compileAction;
    }

    public CutAction getCutAction () {
        return cutAction;
    }

    public CopyAction getCopyAction () {
        return copyAction;
    }

    public DatabaseSelectionAction getDatabaseSelectionAction () {
        return databaseSelectionAction;
    }

    public ImporterAction getImporterAction () {
        return importerAction;
    }

    public ProjectPreferencesAction getProjectPreferencesAction () {
        return projectPreferencesAction;
    }

    public PointModeAction getPointModeAction () {
        return pointModeAction;
    }

    public CloseAction getCloseAction () {
        return closeAction;
    }

    public ExitAction getExitAction () {
        return exitAction;
    }

    public ClassEditorAction getClassEditorAction () {
        return classEditorAction;
    }

    public DeleteAction getDeleteAction () {
        return deleteAction;
    }

    public InsertHorzSplitterAction getInsertHorzSplitterAction () {
        return insertHorzSplitterAction;
    }

    public InsertVertSplitterAction getInsertVertSplitterAction () {
        return insertVertSplitterAction;
    }

    public MultiObjectAlignmentAction getMultiObjectAlignmentAction() {
        return multiObjectAlignmentAction;
    }

    public MultiObjectSizeAction getMultiObjectSizeAction() {
        return multiObjectSizeAction;
    }

    public DuplicateObjectNamesAction getDuplicateObjectNamesAction () {
        return duplicateObjectNamesAction;
    }

    AbstractAction createCopySelectedObjectAction (String sLabel, SuperDesignCockpit theCockpit, int iProgramMode) {
        return new CopySelectedObjectAction(sLabel, getIcon("glp_copy.png"), theCockpit, iProgramMode);
    }

    public void updateUndoStatus (String sLabel, boolean bEnabled) {
        getUndoAction().setEnabled(bEnabled);
        getUndoAction().putValue(Action.NAME, sLabel);
    }

    public void updateRedoStatus (String sLabel, boolean bEnabled) {
        getRedoAction().setEnabled(bEnabled);
        getRedoAction().putValue(Action.NAME, sLabel);
    }

    public LanguageChooserAction getLanguageDeutschAction () {
        return languageDeutschAction;
    }

    public LanguageChooserAction getLanguageFrancaisAction () {
        return languageFrancaisAction;
    }

    public LanguageChooserAction getLanguageItalianAction () {
        return languageItalianAction;
    }

    public LanguageChooserAction getLanguageEnglishAction () {
        return languageEnglishAction;
    }

    /**
     * The base class to use for all IDE interface Actions.<p>
     */
    private abstract class IDEInterfaceAction extends AbstractAction {

        public IDEInterfaceAction (String sLabel, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(sLabel, getIcon(sIconName));

            if (mnemonic != '\0')
                setMnemonicKey(mnemonic);

            if ((sDescription != null) && (sDescription.length() > 0))
                setShortDescription(sDescription);

            if (keyStroke != null)
                setAcceleratorKey(keyStroke);

        }

        public IDEInterfaceAction (ActionDescription desc) {
            this(desc.getLabel(), desc.getMnemonic(), desc.getAccelerator(), desc.getImageName(), desc.getTooltip());
        }

        private void setMnemonicKey (char mnemonic) {
            putValue(Action.MNEMONIC_KEY, new Integer(mnemonic));
        }

        private void setShortDescription (String sDescription) {
            putValue(Action.SHORT_DESCRIPTION, sDescription);
        }

        protected void setAcceleratorKey (KeyStroke keyStroke) {
            putValue(Action.ACCELERATOR_KEY, keyStroke);
        }

        public String getLabel() {
            return (String) getValue(Action.NAME);
        }

        public char getMnemonicKey() {
            return (char)((Integer) getValue(Action.MNEMONIC_KEY)).intValue();
        }


    }

    class AboutAction extends IDEInterfaceAction {
        public AboutAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            new IDEAboutBox(theCockpit);
        }
    }

    class BackgroundColorChooserAction extends IDEInterfaceAction {
        public BackgroundColorChooserAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            if (theCockpit.getSelectedObject() != null) {
                new IDEObjectColorChooser(theCockpit, false);
            }
        }
    }

    class ForegroundColorChooserAction extends IDEInterfaceAction {
        public ForegroundColorChooserAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            if (theCockpit.getSelectedObject() != null) {
                new IDEObjectColorChooser(theCockpit, true);
            }
        }
    }

    class FontChooserAction extends IDEInterfaceAction {
        public FontChooserAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            String sError = "Error displaying the Font Dialog";
            if (theCockpit.getSelectedObject() != null) {
                String sFontChooserClass = theCockpit.getConfigurationManager().getConfigurationVariable("FONT_CHOOSER", theCockpit.thePreferences.sFontChooserName);
                try {
                    Class cls = theCockpit.getClassLoader().getLoader().loadClass(sFontChooserClass);
                    if (cls != null) {
                        Object[] parameters = new Object[1];
                        parameters[0] = theCockpit;
                        Class[] parameterTypes = new Class[1];
                        parameterTypes[0] = SuperDesignCockpit.class;

                        Constructor constructor = cls.getConstructor(parameterTypes);
                        constructor.newInstance(parameters);
                    }
                } catch (ClassNotFoundException e) {
                    JOptionPane.showMessageDialog(theCockpit, "Component missing (" + sFontChooserClass + ")", sError, JOptionPane.ERROR_MESSAGE);
                } catch (NoSuchMethodException e) {
                    JOptionPane.showMessageDialog(theCockpit, "Missing Constructor.\n\nThe class does not have a constructor with only one parameter (SuperDesignCockpit - needed as the parent)", sError, JOptionPane.ERROR_MESSAGE);
                } catch (Exception e) {
                    JOptionPane.showMessageDialog(theCockpit, e.getMessage(), sError, JOptionPane.ERROR_MESSAGE);
                    e.printStackTrace();
                }
            }
        }
    }

    class UndoAction extends IDEInterfaceAction {
        public UndoAction (ActionDescription description) {
            super(description);
            enabled = false;
        }

        public void actionPerformed (ActionEvent evt) {
            theCockpit.getDesignProject().theUndoRedoController.undo();
            MetaObject theSelectedFrame = theCockpit.getSelectedFrame();
            MetaObject theSelectedObject = theCockpit.getSelectedObject();
            theCockpit.SelectObject(null, false);
            if (theSelectedFrame != null)
                theSelectedFrame.refresh();
            if (theSelectedObject.bIsDeleted == false)
                theCockpit.SelectObject(theSelectedObject, false);
        }
    }

    class NewAction extends IDEInterfaceAction {
        public NewAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            DesignProject theDesignProject = null;
            try {
                theDesignProject = theCockpit.loadProject((String) null, null, theCockpit, MetaProject.FROM_XML_DOCUMENT);
            } catch (HammerException e1) {
            }
            if (theDesignProject != null) {
                theDesignProject.setInputType(MetaProject.FROM_XML_DOCUMENT);
                SelectProjectAction actionProject = new SelectProjectAction(theDesignProject, theCockpit);
                // Add to the window menu.
                theCockpit.getIDEMenuBar().addProjectItem(actionProject);
                theDesignProject.theAssociatedAction.actionPerformed(
                        new ActionEvent(this, 0, theDesignProject.theMetaProject.sName));
                theDesignProject.theMetaProject.setProjectChangedState(false);
            }
        }
    }

    class PasteAction extends IDEInterfaceAction {
        public PasteAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
//            MetaObject theSelectedContainer = theDesignCockpit.getSelectedContainer();
            Component theFocus = KeyboardFocusManager.getCurrentKeyboardFocusManager().getFocusOwner();
            if (isFocusTextOriented(theFocus)) {
                if (theCockpit.theClipboardController.getClipboardContent() == IDEClipboardController.CLIPBOARD_TEXT)
                    theCockpit.editorAction("paste-from-clipboard", evt);
                // clipboard is essentially empty - wrong stuff to paste.
            } else {
                if (theCockpit.theClipboardController.getClipboardContent() == IDEClipboardController.CLIPBOARD_OBJECT) {
                    if ((theCockpit.theLastPopupMenu != null) && (theCockpit.theLastPopupMenu.theTarget != null))
                        pasteSelectedObjectAction.actionPerformed(evt);
                    else {
                        MetaObject theTarget = theCockpit.theFirstDesignProject.getSelectedFrame();
                        theCockpit.theLastPopupMenu.theTarget = theTarget;
                        pasteSelectedObjectAction.actionPerformed(evt);
                    }
                }
            }
        }
    }

    /**
     * Only used by the Context Menu of the View Displayer!
     */
    class PasteSelectedObjectAction extends IDEInterfaceAction {
        public PasteSelectedObjectAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            DesignProject theSelectedProject = theCockpit.getDesignProject();
            if (theSelectedProject == null)
                return;
            theCockpit.SelectObject(theCockpit.theLastPopupMenu.theTarget, false);
            Point ptLocation = theCockpit.theLastPopupMenu.getLastPopupLocation();
            Point ptActualLocation = theCockpit.theLastPopupMenu.getActualStrikePoint();
            Integer X = new Integer(ptLocation.x);
            Integer Y = new Integer(ptLocation.y);
            Integer ActualX = new Integer(ptActualLocation.x);
            Integer ActualY = new Integer(ptActualLocation.y);
            try {
                if (theCockpit.theClipboardController.getClipboardObjectCount() != 0) {  // something needs to be copied first
                    MetaObject theTarget = theCockpit.theLastPopupMenu.theTarget;
                    ArrayList objsPasted = new ArrayList(0);
                    AbstractAction theAction = theCockpit.theClipboardController.getClipboardBufferingAction();
                    boolean bPreserveObjectNames = false;
                    if (theAction instanceof IDEObjectPopupMenu.CutSelectedObjectAction)
                        bPreserveObjectNames = true;
                    for (int i = 0; i < theCockpit.theClipboardController.getClipboardObjectCount(); i++) {
                        MetaObject theClipboardObject = theCockpit.theClipboardController.getClipboardObject(i);
                        MetaObject theNewObject = null;
                        if (theCockpit.theLastPopupMenu.theTarget != null) {
                            if (theCockpit.theLastPopupMenu.equals(theCockpit.theObjectTreeViewPane.thePopupMenu)) {
                                if ((theClipboardObject.isAbalet() == false) && (theClipboardObject.isFrame() == false))  // cannot paste these inside frames
                                    theNewObject = theCockpit.PasteObject(theCockpit.theLastPopupMenu.theTarget, theClipboardObject, theCockpit.getDesignProject().theMetaProject, bPreserveObjectNames);
                            } else {
                                if ((theClipboardObject.isAbalet() == false) && (theClipboardObject.isFrame() == false))  // cannot paste these inside frames
                                    theNewObject = theCockpit.PasteObject(theCockpit.theLastPopupMenu.theTarget, theClipboardObject, theCockpit.getDesignProject().theMetaProject, X, Y, ActualX, ActualY, i, bPreserveObjectNames);
                            }
                        } else
                            theNewObject = theCockpit.theCurrentDesignProject.PasteObject(theClipboardObject, theCockpit, bPreserveObjectNames);
                        if (theNewObject != null) {
                            objsPasted.add(theNewObject);
                        }
                    }
                    theSelectedProject.theUndoRedoController.createPasteEvent(theTarget, objsPasted);
                    // make sure no objects are selected after pasting.
                    theCockpit.theCurrentDesignProject.deselectObjects();
                    MetaObject testObject = theCockpit.theLastPopupMenu.theTarget;
                    while ((testObject != null) && (testObject.isFrame() == false) && (testObject.isAbalet() == false)) {
                        testObject = testObject.theParentObject;
                    }
                    if ((theTarget != null) && (theTarget.isWithinTabPane() == false))
                        theCockpit.FrameSelection(theTarget, true);

                    // 20050915, acg: Select pasted objects
                    if ( objsPasted.size()>0 )
                    {
                        MetaObject theSelectedFrame = theCockpit.getSelectedFrame();
                        theCockpit.SelectObject(null, false);
                        if (theSelectedFrame != null)
                            theSelectedFrame.refresh();
                        for ( int iobj=0; iobj<objsPasted.size(); iobj++ )
                            theCockpit.SelectObject( (MetaObject)objsPasted.get(iobj), true );
                    }
                }
            } catch (Exception e1) {
                e1.printStackTrace();
            }
//            theDesignCockpit.clipboardBufferedObject = theDesignCockpit.getSelectedObject().Clone(new MetaCloneMode(theDesignCockpit.theClipboardController.theCloneMode));
        }
    }

    class RedoAction extends IDEInterfaceAction {
        public RedoAction (ActionDescription description) {
            super(description);
            enabled = false;
        }

        public void actionPerformed (ActionEvent evt) {
            theCockpit.getDesignProject().theUndoRedoController.redo();
            MetaObject theSelectedFrame = theCockpit.getSelectedFrame();
            MetaObject theSelectedObject = theCockpit.getSelectedObject();
            theCockpit.SelectObject(null, false);
            if (theSelectedFrame != null)
                theSelectedFrame.refresh();
            if (theSelectedObject.bIsDeleted == false)
                theCockpit.SelectObject(theSelectedObject, false);
        }
    }

    class SaveAction extends IDEInterfaceAction {
        public SaveAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            boolean bShouldCompile = (theCockpit.isCustomizing()? true: false);
            theCockpit.SaveProject(bShouldCompile); // do not compile it, unless Customizing
        }

    }

    class SaveWithCompileAction extends IDEInterfaceAction {
        public SaveWithCompileAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            theCockpit.SaveProject(true); // compile it
        }

    }

    class OpenAction extends IDEInterfaceAction {
        public OpenAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            String newProjectName;  // Name of project opened.

            JFileChooser theFileChooser = new JFileChooser(theCockpit.getCurrentDirectory());
            theFileChooser.addChoosableFileFilter(new ProjFileFilter(theCockpit));
            theFileChooser.setFileView(new ProjFileView(theCockpit));
            int iOptionSelected = theFileChooser.showOpenDialog(theCockpit.getContentPane());
            String sFullPath = null;
            if (iOptionSelected == JFileChooser.APPROVE_OPTION) {
                File file = theFileChooser.getSelectedFile();
                try {
                    sFullPath = file.getCanonicalPath();
                    String sSeparator = theCockpit.props.getProperty("file.separator");
                    int iSeparator = sFullPath.lastIndexOf(sSeparator);
                    if (iSeparator != -1) {
                        String sDirectoryName = sFullPath.substring(0, iSeparator + 1);
                        theCockpit.setCurrentDirectory(sDirectoryName);
                    }
                } catch (java.io.IOException e1) {
                }

                boolean bFromJar = false;
                // Check from jar.
                HammerJarAccess theJarAccess = null;
                String sTestName = file.getName();

                if ((sTestName != null) && (sTestName.toLowerCase().endsWith("jar"))) {
                    sTestName = file.getAbsolutePath();
                    JarOpenFileDialog theJarDialog = new JarOpenFileDialog(sTestName);
                    // wait till they do something.

                    theJarDialog.jarOpenDialog.show();
                    JarContentsTreeNode theComponent = theJarDialog.getSelection();
                    if (theComponent == null)
                        return;
                    newProjectName = theComponent.theAccess.sKey;
                    if (newProjectName == null)
                        return;
                    theJarAccess = theJarDialog.getSelection().theAccess.theAccess;
                    if (theCockpit.getOpenSourceState() == false)
                        bFromJar = true;
                } else
                    newProjectName = file.getName();

                // Before loading check to see if it is open.
                // Switch to project if open already.
                Component[] theProjectWindowMenuItems = theCockpit.getIDEMenuBar().getWindowMenu().getMenuComponents();
                for (int i = 0; i < theProjectWindowMenuItems.length; i++) {
                    if (theProjectWindowMenuItems[i] instanceof JMenuItem) {
                        JMenuItem theMenuItem = (JMenuItem) theProjectWindowMenuItems[i];
                        if (theMenuItem.getText().equals(newProjectName)) {
                            SelectProjectAction reopenAction = (SelectProjectAction) theMenuItem.getAction();
                            reopenAction.actionPerformed(new ActionEvent(theCockpit, 0, reopenAction.theDesignProject.theMetaProject.sName));
                            return;
                        }
                    }
                }

                // Otherwise open new project and add it to the window menu.
                DesignProject newProject = null;
                try {
                    if (bFromJar)
                        newProject = theCockpit.loadProject(theJarAccess, newProjectName, theCockpit, MetaProject.FROM_RESOURCE);
                    else
                        newProject = theCockpit.loadProject(sFullPath, file.getName(), theCockpit, MetaProject.FROM_XML_DOCUMENT);
                } catch (HammerException e1) {
                }
                if (newProject != null) {
                    SelectProjectAction actionProject = new SelectProjectAction(newProject, theCockpit);
                    // Add to the window menu.
                    theCockpit.getIDEMenuBar().addProjectItem(actionProject);
                    actionProject.actionPerformed(new ActionEvent(theCockpit, 0, actionProject.theDesignProject.theMetaProject.sName));
                    newProject.theMetaProject.setProjectChangedState(false);
                }
            }
        }
    }

    class SaveAllAction extends IDEInterfaceAction {
        public SaveAllAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            DesignProject theDesignProject = theCockpit.theFirstDesignProject;
            while (theDesignProject != null) {
                boolean bTest = theDesignProject.store(true);
                if (bTest == false)
                    break;
                theDesignProject = theDesignProject.theNextDesignProject;
            }
        }
    }

    class RunAction extends IDEInterfaceAction {
        public RunAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            theCockpit.runProject();
        }
    }

    class CompileAction extends IDEInterfaceAction {
        public CompileAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            // This works on a copy of the class list.
            MetaProjectHeader theProgram = null;
            try {
                theProgram = theCockpit.getProgramMetadata();
            } catch (HammerException e1) {
                return;  // no point in going on.
            }
            // add classes.
            if ((theCockpit.getDesignProject() == null) || (theCockpit.getDesignProject().getFirstObject() == null))
                return;
            // Save nls documents.
            theCockpit.getDesignProject().theMetaProject.getNLS().writeAllChangedNlsDocuments();

            // add imports for top level classes.  Subclasses don't need to import.
//            MetaClass theClass = getDesignProject().theFirstClass;
//            while (theClass != null) {
//                theProgram.AddImport(theClass.theMetadata.getFullPathOfSuperClass());
//                // Clear the pointer to next to not have crossed references.
//               // We use designclass tree to get to classes here, not metadata.
//              theClass = theClass.theNextSibling;
//           }
            theProgram.theMainObject = theCockpit.getDesignProject().getFirstObject();
            // Name the mainclass.
            theProgram.sMainClass = theCockpit.getDesignProject().getFirstObject().theClass.getMetadata().sClassName;
            // Name the package.
            theProgram.sPackageName = "ch.abacus.designcockpit.ide";
            // Generate the output document.
            UIFactoryAssemblyLine roastThis = null;

            theCockpit.setEnabled(false);

            try {
                roastThis = new UIFactoryAssemblyLine(theCockpit.getDesignProject().theMetaProject, theCockpit.thePreferences.sWorkingDirectory);
                roastThis.CreateUI();
                if (theCockpit.theSourceViewer != null)
                    theCockpit.theSourceViewer.theMessagesEditor.setText("Compilation Successful!\n");
            } catch (Exception e1) {
                ExceptionWriter writerTemp = new ExceptionWriter(theCockpit);
                PrintWriter pwTemp = new PrintWriter(writerTemp);
                e1.printStackTrace(pwTemp);
                if (theCockpit.theSourceViewer != null)
                    theCockpit.theSourceViewer.setMessagesText(writerTemp.getBuffer());
            } catch (Throwable e2) {
                ExceptionWriter writerTemp = new ExceptionWriter(theCockpit);
                PrintWriter pwTemp = new PrintWriter(writerTemp);
                e2.printStackTrace(pwTemp);
                if (theCockpit.theSourceViewer != null)
                    theCockpit.theSourceViewer.setMessagesText(writerTemp.getBuffer());
            } finally {
                theCockpit.setEnabled(true);
            }

            // The errors can be found using the code jumper.
            if (roastThis != null) {
                theCockpit.theCodeJumper = roastThis.theCodeJumper;
                if (theCockpit.theSourceViewer != null) {
                    theCockpit.theSourceViewer.theListener.setCodeJumper(theCockpit.theCodeJumper);
                    theCockpit.theSourceViewer.showMessageEditor();
                }
            }
        }
    }

    class CutAction extends IDEInterfaceAction {
        public CutAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            Component theFocus = KeyboardFocusManager.getCurrentKeyboardFocusManager().getFocusOwner();
            if (isFocusTextOriented(theFocus)) {
                theCockpit.theClipboardController.setClipboardContent(IDEClipboardController.CLIPBOARD_TEXT);
                theCockpit.editorAction("cut-to-clipboard", evt);
            } else {
                if (theCockpit.theLastPopupMenu == null)
                    theCockpit.theLastPopupMenu = theCockpit.theObjectTreeViewPane.thePopupMenu;
                theCockpit.theLastPopupMenu.cutSelectedObjectAction.actionPerformed(evt);
            }
        }
    }

    class CopyAction extends IDEInterfaceAction {
        public CopyAction (ActionDescription description) {
            super(description);
        }

        public void actionPerformed (ActionEvent evt) {
            Component theFocus = KeyboardFocusManager.getCurrentKeyboardFocusManager().getFocusOwner();
            if (isFocusTextOriented(theFocus)) {
                theCockpit.theClipboardController.setClipboardContent(IDEClipboardController.CLIPBOARD_TEXT);
                theCockpit.editorAction("copy-to-clipboard", evt);
            } else {
                if (theCockpit.theLastPopupMenu == null)
                    theCockpit.theLastPopupMenu = theCockpit.theObjectTreeViewPane.thePopupMenu;
                theCockpit.theLastPopupMenu.copySelectedObjectAction.actionPerformed(evt);
            }
        }
    }

    class DatabaseSelectionAction extends IDEInterfaceAction {

        public DatabaseSelectionAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            if (theCockpit.getSelectedObject() != null) {

                String theDataBaseSelectorClass = theCockpit.getConfigurationManager().getConfigurationVariable("DataBaseSelectorClass");
                try {
                    Class cls = theCockpit.getClassLoader().getLoader().loadClass(theDataBaseSelectorClass);
                    if (cls != null) {
                        Class[] paramClasses = new Class[3];
                        Object[] paramObjects = new Object[3];
                        paramObjects[0] = theCockpit.getSelectedObject();
                        paramObjects[1] = "DummyPropertyName";
                        paramObjects[2] = theCockpit;
                        paramClasses[0] = MetaObject.class;
                        paramClasses[1] = String.class;
                        paramClasses[2] = SuperDesignCockpit.class;
                        try {
                            Constructor constr = cls.getConstructor(paramClasses);
                            constr.newInstance(paramObjects);
                        } catch (NoSuchMethodException e) {
                        } catch (SecurityException e) {
                        } catch (InstantiationException e) {
                        } catch (IllegalAccessException e) {
                        } catch (InvocationTargetException e) {
                        }
                    }
                } catch (ClassNotFoundException e) {

                }
            }
        }
    }

    class ImporterAction extends IDEInterfaceAction {
        public ImporterAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            new IDEImporter(theCockpit);
        }
    }

    class ProjectPreferencesAction extends IDEInterfaceAction {
        public ProjectPreferencesAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            MetaProject theMetaProject;
            try {
                if (theCockpit.getDesignProject() == null) {
                    theMetaProject = new MetaProject(AbaMetaDataUser.DESIGN_MODE, theCockpit.theGlobalInterface, (theCockpit.theClassLoader), theCockpit.getMetadataDispenser());
                    DesignProject theDesignProject = new DesignProject(theMetaProject);
                    theCockpit.addDesignProject(theDesignProject);
                    theCockpit.SetDesignProject(theDesignProject);
                } else
                    theMetaProject = theCockpit.getDesignProject().theMetaProject;

                new IDEProjectTool(theCockpit, theMetaProject);
            } catch (HammerException e1) {
                System.err.println("Exception: " + e1.getMessage());
            }
        }
    }

    class PointModeAction extends IDEInterfaceAction {
        public PointModeAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            theCockpit.setMouseSelectState(true);
        }
    }

    class CopySelectedObjectAction extends AbstractAction {
        SuperDesignCockpit theCockpit = null;
        MetaCloneMode theMode = new MetaCloneMode(MetaCloneMode.deepCopy);

        public CopySelectedObjectAction (String label, Icon icon, SuperDesignCockpit objDesignCockpit, int mode) {
            super(label, icon);
            theCockpit = objDesignCockpit;
            theMode = new MetaCloneMode(mode);
        }

        public void actionPerformed (ActionEvent evt) {
            DesignProject theSelectedProject = theCockpit.getDesignProject();
            if (theSelectedProject == null)
                return;
            theCockpit.theClipboardController.removeAllClipboardObjects();
            for (int i = 0; i < theSelectedProject.getSelectedObjectCount(); i++) {
                MetaObject theMetaObject = theSelectedProject.getSelectedObject(i);
                if (theMetaObject != null) {
                    // if the component name starts with the class name, then we will assign a new name,
                    // otherwise we will retain the existing name
                    String sClassName = theMetaObject.theClass.getMetadata().sClassName;
                    String sObjectName = theMetaObject.getName();
                    boolean bPreserveNames = !(sObjectName.startsWith(sClassName));

                    // OpenSource needs to pass the events
                    if(theCockpit.getOpenSourceState() && theMode.theMode==MetaCloneMode.shallowCopyWithoutCode)
                        theMode.theMode=MetaCloneMode.shallowCopyWithoutCodeOpenSource;

                    theCockpit.theClipboardController.addClipboardObject(theMetaObject.Clone(theMode, null, bPreserveNames));
                }
            }
            theCockpit.theClipboardController.setClipboardContent(IDEClipboardController.CLIPBOARD_OBJECT);
            theCockpit.theClipboardController.setClipboardBufferingAction(this);
            theCockpit.theClipboardController.theCloneMode = theMode.theMode;
            theSelectedProject.deselectObjects();
        }
    }

    class SaveAsAction extends IDEInterfaceAction {

        public SaveAsAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            String sProjectName = theCockpit.theProjectName;
            if ((theCockpit.getDesignProject() == null) || (theCockpit.getDesignProject().getFirstObject() == null))
                return;

            theCockpit.theProjectName = null; // Make sure we are prompted for a filename.
            boolean bSavedOk = theCockpit.SaveProject(false); // save, but do not compile, a newly-created project
            if (bSavedOk == false)
                theCockpit.theProjectName = sProjectName; // reset the name, as the user canceled the Save As
        }
    }

    class SaveMetadataAction extends IDEInterfaceAction {

        public SaveMetadataAction (String label, char mnemonic, KeyStroke keyStroke, String sIconName, String sDescription) {
            super(label, mnemonic, keyStroke, sIconName, sDescription);

            enabled = theCockpit.isCanSaveMetadata();
        }

        public void actionPerformed (ActionEvent evt) {
            theCockpit.theCurrentDesignProject.theMetaProject.getMetadataDispenser().theMetadataDocument.writeMetaDocument();
        }
    }

    class CloseAction extends IDEInterfaceAction {

        public CloseAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            DesignProject theProject = theCockpit.getDesignProject();
            boolean bTest = theProject.close(true);
            if (bTest == false) // cancel the close.
                return;

            // 2005-09-05, acg: Clear screen
            if (theCockpit.theObjectTreeViewPane != null)
                theCockpit.theObjectTreeViewPane.clear();
            if (theCockpit.theViewDisplayer != null)
                theCockpit.theViewDisplayer.resetContents();
            if (theCockpit.getPropertyInspectorController() != null)
                theCockpit.getPropertyInspectorController().resetContents();
            if (theCockpit.theProgramMode == SuperDesignCockpit.JAVABUILDER) {
                if (theCockpit.theSourceViewer != null)
                    theCockpit.theSourceViewer.setCodeText("");
            }

            // we must add the closed file to the MRU list, but only if it is not a "New Project" and
            // came from a regular project file (eg, not a JAR)
            if ((theProject.theMetaProject.sName != null) &&
                    (!theProject.theMetaProject.sName.startsWith("New Project")) &&
                    (theProject.theMetaProject.sName.trim().length() != 0)) {
                if (theProject.theMetaProject.getInputType() == MetaProject.FROM_XML_DOCUMENT) {
                    if (theCockpit.isCustomizing() == false)
                        theCockpit.mruList.fileClosed(theProject.theMetaProject.getProjectDirectory() + theProject.theMetaProject.sName);
                }
            }
            DesignProject thePrevProject = theProject.thePreviousDesignProject;
            DesignProject theNextProject = theProject.theNextDesignProject;
            theCockpit.removeDesignProject(theProject);
            // End design cockpit if all the projects are closed.
//            if (theDesignCockpit.theFirstDesignProject == null)
//                System.exit(0);
            // Else move on to next.
            DesignProject theChoice = theNextProject;
            if (theChoice == null)
                theChoice = thePrevProject;
            if (theChoice != null)
                theChoice.theAssociatedAction.actionPerformed(new ActionEvent(this, 0,
                        theChoice.theMetaProject.sName));

            Component[] theProjectWindowMenuItems = theCockpit.getIDEMenuBar().getWindowMenu().getMenuComponents();
            for (int i = 0; i < theProjectWindowMenuItems.length; i++) {
                if (theProjectWindowMenuItems[i] instanceof JMenuItem) {
                    JMenuItem theMenuItem = (JMenuItem) theProjectWindowMenuItems[i];
                    if (theMenuItem.getText().equals(theProject.theMetaProject.sName)) {
                        theCockpit.getIDEMenuBar().getWindowMenu().remove(theMenuItem);
                        break;
                    }
                }
            }
            if (theChoice == null) {
                // create new project.
                SelectProjectAction actionProject = null;
                try {
                    theChoice = theCockpit.loadProject((String) null, (String) null, theCockpit, MetaProject.UNDEFINED_INPUT_TYPE);
                    if (theChoice != null) {
                        theChoice.setCurrentLanguage(HammerLanguagePresentation.DEFAULT);
                        actionProject = new SelectProjectAction(theChoice, theCockpit);
                        actionProject.actionPerformed(new ActionEvent(this, 0, actionProject.theDesignProject.theMetaProject.getName()));
                        theChoice.theMetaProject.setProjectChangedState(false);
                        theCockpit.getIDEMenuBar().addProjectItem(actionProject);
                    }
                } catch (HammerException e1) {
                    // This can happen only if it's very unstable.
                    // We can't even count on a message box.
                    // If the log was open, we've already written a message.
                    return;
                }
            }

            theCockpit.setCurrentDesignProject(theChoice);
        }
    }

    class ExitAction extends IDEInterfaceAction {

        public ExitAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            IDEWindowClosingEventHandler exitHandler = new IDEWindowClosingEventHandler(theCockpit);
            exitHandler.windowClosing(new WindowEvent(theCockpit, 0));
        }
    }

    class ClassEditorAction extends IDEInterfaceAction {

        public ClassEditorAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            MetaObject theSelectedObject = theCockpit.getDesignProject().getSelectedObject();
            if (theSelectedObject != null) {
                MetaClass theMetaClass = theSelectedObject.theClass;
                String theEditorClassName = theMetaClass.getMetadata().theDesignEditor;
                if (theEditorClassName != null) {
                    HammerClassLoader clsLoader = theCockpit.theClassLoader;
                    Class clsEditor = null;
                    try {
                        clsEditor = Class.forName(theEditorClassName, true, clsLoader.getLoader());
                    } catch (ClassNotFoundException e1) {
                        System.out.print("Cannot find class editor " + theEditorClassName);
                        return;
                    }
                    // All editors have the following interface.
                    //
                    //  Parameter 1 - MetaObject -> the object being accessed.
                    //
                    Object[] theParams = new Object[1];
                    Class[] theParamClasses = new Class[1];
                    theParams[0] = theSelectedObject;
                    theParamClasses[0] = theSelectedObject.getClass();
                    try {
                        Constructor theConstructor = clsEditor.getConstructor(theParamClasses);
                        try {
                            theConstructor.newInstance(theParams);
                        } catch (Exception e2) {
                            System.out.println(e2.getMessage());
                            return;
                        }
                    } catch (Exception e3) {
                        System.out.println(e3.getMessage());
                        return;
                    }
                }
            }
        }
    }

    class DeleteAction extends IDEInterfaceAction {

        public DeleteAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            DesignProject theSelectedProject = theCockpit.getDesignProject();
            if (theSelectedProject == null)
                return;

            if (theCockpit.isCustomizing()) {
                theCockpit.theCustomizer.selectElementGroupElements();
            }

            ArrayList objsToDelete = new ArrayList(0);
            ArrayList objsParents = new ArrayList(0);
            for (int i = 0; i < theSelectedProject.getSelectedObjectCount(); i++) {
                objsToDelete.add(theSelectedProject.getSelectedObject(i));
                objsParents.add(theSelectedProject.getSelectedObject(i).theParentObject);
            }

            theSelectedProject.theUndoRedoController.createDeleteObjectsEvent(objsParents, objsToDelete);
            for (int i = 0; i < objsToDelete.size(); i++) {
                MetaObject theSelectedObject = (MetaObject) objsToDelete.get(i);
                if ((theSelectedObject != null) && (theSelectedObject.isFrame() == false) && (theSelectedObject.isAbalet() == false)) {
                    theSelectedProject.deselectObject(theSelectedObject);
                    theSelectedObject.DeleteObject(theCockpit.theCurrentDesignProject.theMetaProject, theCockpit);
                }
            }
            theCockpit.deselectObjects();
            theCockpit.getDesignProject().theMetaProject.setProjectChangedState(true);

        }
    }

    class InsertHorzSplitterAction extends IDEInterfaceAction {

        public InsertHorzSplitterAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            JMenuItem menuItem = (JMenuItem)evt.getSource();
            IDEObjectPopupMenu menu = (IDEObjectPopupMenu)menuItem.getParent();
            MetaObject theTarget = menu.theTarget;
            MetaObject theSplitter = theTarget.createHorizontalSplitter(theCockpit.theViewDisplayer.iMouseY);
            if (theSplitter==null)
                return;
            theCockpit.SelectObject(theSplitter, false);
            JSplitPane theSplitterVisualObject = (JSplitPane) theSplitter.theVisualObject;
            int iDividerLocation = ((Integer)theSplitter.getPropertyValue("DividerLocation", 0).getNativeValue()).intValue();
            theSplitterVisualObject.resetToPreferredSizes();
            theSplitterVisualObject.setDividerLocation(iDividerLocation);
        }
    }

    class InsertVertSplitterAction extends IDEInterfaceAction {

        public InsertVertSplitterAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            JMenuItem menuItem = (JMenuItem)evt.getSource();
            IDEObjectPopupMenu menu = (IDEObjectPopupMenu)menuItem.getParent();
            MetaObject theTarget = menu.theTarget;
            MetaObject theSplitter = theTarget.createVerticalSplitter(theCockpit.theViewDisplayer.iMouseY);
            if (theSplitter==null)
                return;
            theCockpit.SelectObject(theSplitter, false);
            JSplitPane theSplitterVisualObject = (JSplitPane) theSplitter.theVisualObject;
            int iDividerLocation = ((Integer)theSplitter.getPropertyValue("DividerLocation", 0).getNativeValue()).intValue();
            theSplitterVisualObject.resetToPreferredSizes();
            theSplitterVisualObject.setDividerLocation(iDividerLocation);
        }
    }

    class MultiObjectAlignmentAction extends IDEInterfaceAction {

        public MultiObjectAlignmentAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            new IDEMultiSelectAlignmentDialog(theCockpit);
        }
    }

    class MultiObjectSizeAction extends IDEInterfaceAction {

        public MultiObjectSizeAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            new IDEMultiSelectSizeDialog(theCockpit);
        }
    }

    class DuplicateObjectNamesAction extends IDEInterfaceAction {
        public DuplicateObjectNamesAction (ActionDescription actionDescription) {
            super(actionDescription);
        }

        public void actionPerformed (ActionEvent evt) {
            MetaProject currentProject = theCockpit.getDesignProject().theMetaProject;
            if (currentProject != null) {
                ArrayList list = currentProject.getDuplicateObjectNames();
                String sObjects = null;
                String sMessage;
                if ((list == null) || (list.size() == 0))
                    sMessage = "There are no duplicate object names in this project";
                else {
                    for (int iObject = 0; iObject < list.size(); iObject++) {
                        String sObject = (String) list.get(iObject) + "\n";
                        if (sObjects == null)
                            sObjects = sObject;
                        else
                            sObjects += sObject;
                    }
                    sMessage = "The following Objects names are used by more than one object:\n" + sObjects;
                }
                JOptionPane.showMessageDialog(theCockpit, sMessage, "Detect Duplicate Object Names", JOptionPane.INFORMATION_MESSAGE);

            }
        }

    }

    class LanguageChooserAction extends IDEInterfaceAction {
        private int language = HammerLanguagePresentation.DEFAULT;

        public LanguageChooserAction (ActionDescription actionDescription, int iLanguageId) {
            super(actionDescription);
            this.language = iLanguageId;
        }

        public void actionPerformed (ActionEvent evt) {
            theCockpit.getDesignProject().setCurrentLanguage(language);
            theCockpit.theStatusPanel.setLanguage((String)getValue(Action.NAME));
            theCockpit.getIDEToolBar().setLangauge(language);
            MetaObject theSelectedFrame = theCockpit.getSelectedFrame();
            MetaObject theSelectedObject = theCockpit.getSelectedObject();
            theCockpit.SelectObject(null, false);
            if (theSelectedFrame != null)
                theSelectedFrame.refresh();
            theCockpit.SelectObject(theSelectedObject, false);

            if (theCockpit.isCustomizing())
                theCockpit.theCustomizer.languageChanged(language);
        }

        public int getLanguage () {
            return language;
        }
    }

    private class ActionInitializer {

        LanguageChooserAction initLanguageEnglishAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("English");
            actionDescription.setMnemonic('E');
            actionDescription.setImageName("glp_language_english.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "English");

            return new LanguageChooserAction(actionDescription, HammerLanguagePresentation.ENGLISH);
        }

        LanguageChooserAction initLanguageItalianAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Italiano");
            actionDescription.setMnemonic('I');
            actionDescription.setImageName("glp_language_italian.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Italian");

            return new LanguageChooserAction(actionDescription, HammerLanguagePresentation.ITALIAN);
        }

        LanguageChooserAction initLanguageFrancaisAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Francais");
            actionDescription.setMnemonic('F');
            actionDescription.setImageName("glp_language_french.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "French");

            return new LanguageChooserAction(actionDescription, HammerLanguagePresentation.FRANCAIS);
        }

        LanguageChooserAction initLanguageDeutschAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Deutsch (Default)");
            actionDescription.setMnemonic('D');
            actionDescription.setImageName("glp_language_german.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "German");

            return new LanguageChooserAction(actionDescription, HammerLanguagePresentation.DEUTSCH);
        }

        CompileAction initCompileAction () {
            String sLabel = "Compile";
            char mnemonic = 'C';
            KeyStroke keyStroke = null;
            String sImageName = "glp_compile.png";
            String sTooltip = "Compile";

            return new CompileAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        SaveMetadataAction initSaveMetadataAction () {
            String sLabel = "Save Metadata";
            char mnemonic = 'M';
            KeyStroke keyStroke = null;
            String sImageName = "glp_savemetadata.png";
            String sTooltip = null;

            return new SaveMetadataAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        SaveAsAction initSaveAsAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Save As");
            actionDescription.setMnemonic('A');
            actionDescription.overrideWithExternalResources(theResourceBundle, "SaveAs");

            return new SaveAsAction(actionDescription);
        }

        DuplicateObjectNamesAction initDuplicateObjectNamesAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Detect Duplicate Object Names");
            actionDescription.setMnemonic('N');
            actionDescription.overrideWithExternalResources(theResourceBundle, "DuplicateNames");

            return new DuplicateObjectNamesAction(actionDescription);
        }

        CloseAction initCloseAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Close");
            actionDescription.setMnemonic('C');
            actionDescription.setTooltip("Closes the current project");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Close");

            return new CloseAction(actionDescription);
        }

        ExitAction initExitAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Exit");
            actionDescription.setMnemonic('X');
            actionDescription.setImageName("glp_exit.png");
            actionDescription.setTooltip("Exits the program");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Exit");

            return new ExitAction(actionDescription);
        }

        ClassEditorAction initClassEditorAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Object Editor");
            actionDescription.setMnemonic('J');
            actionDescription.setImageName("glp_gear.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "ObjectEditor");

            return new ClassEditorAction(actionDescription);
        }

        DeleteAction initDeleteAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Delete");
            actionDescription.setMnemonic('D');
            actionDescription.setImageName("glp_delete.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Delete");

            return new DeleteAction(actionDescription);
        }

        InsertHorzSplitterAction initInsertHorzSplitterAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Insert Horizontal Splitter");
            actionDescription.setMnemonic('H');
            actionDescription.overrideWithExternalResources(theResourceBundle, "InsertHorizontalSplitter");

            return new InsertHorzSplitterAction(actionDescription);
        }

        InsertVertSplitterAction initInsertVertSplitterAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Insert Vertical Splitter");
            actionDescription.setMnemonic('V');
            actionDescription.overrideWithExternalResources(theResourceBundle, "InsertVerticalSplitter");

            return new InsertVertSplitterAction(actionDescription);
        }

        MultiObjectAlignmentAction initMultiObjectAlignmentAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Alignment");
            actionDescription.setMnemonic('A');
            actionDescription.setTooltip("Configure the alignment of the selected components");
            actionDescription.overrideWithExternalResources(theResourceBundle, "MultiObjectAlignment");

            return new MultiObjectAlignmentAction(actionDescription);
        }

        MultiObjectSizeAction initMultiObjectSizeAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Size");
            actionDescription.setMnemonic('S');
            actionDescription.setTooltip("Configure the sizet of the selected components");
            actionDescription.overrideWithExternalResources(theResourceBundle, "MultiObjectSize");

            return new MultiObjectSizeAction(actionDescription);
        }

        PointModeAction initPointModeAction () {
            String sLabel = "Select Objects";
            char mnemonic = 'S';
            KeyStroke keyStroke = null;
            String sImageName = "glp_arrow.png";
            String sTooltip = "Select";

            return new PointModeAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        ProjectPreferencesAction initProjectPreferencesAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Project Preferences");
            actionDescription.setMnemonic('P');
            actionDescription.setImageName("preferences.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "ProjectPreferences");

            return new ProjectPreferencesAction(actionDescription);


        }

        ImporterAction initImporterAction () {
            String sLabel = "Import Classes";
            char mnemonic = 'I';
            KeyStroke keyStroke = null;
            String sImageName = "import2.png";
            String sTooltip = sLabel;

            return new ImporterAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        DatabaseSelectionAction initDatabaseSelectionAction () {
            String sLabel = "Data Base";
            char mnemonic = 'D';
            KeyStroke keyStroke = null;
            String sImageName = "glp_dbselect.png";
            String sTooltip = "Choose Table and Field Values";

            return new DatabaseSelectionAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        CopyAction initCopyAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Copy");
            actionDescription.setMnemonic('C');
            actionDescription.setImageName("glp_copy.png");
            actionDescription.setTooltip("Copy");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Copy");

            return new CopyAction(actionDescription);
        }

        CutAction initCutAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Cut");
            actionDescription.setMnemonic('T');
            actionDescription.setImageName("glp_cut.png");
            actionDescription.setTooltip("Cut");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Cut");

            return new CutAction(actionDescription);
        }

        RunAction initRunAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Run");
            actionDescription.setMnemonic('R');
            actionDescription.setImageName("media_play.png");
            actionDescription.setTooltip("Render");
            actionDescription.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F9, 0));
            actionDescription.overrideWithExternalResources(theResourceBundle, "Run");

            return new RunAction(actionDescription);
        }

        SaveAllAction initSaveAllAction () {
            String sLabel = "Save All";
            char mnemonic = 'L';
            KeyStroke keyStroke = null;
            String sImageName = "glp_saveall.png";
            String sTooltip = "Save all open projects";

            return new SaveAllAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        OpenAction initOpenAction () {
            String sLabel = "Open";
            char mnemonic = 'O';
            KeyStroke keyStroke = null;
            String sImageName = "folder.png";
            String sTooltip = "Open an existing project";

            return new OpenAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        SaveWithCompileAction initSaveWithCompileAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Save (With Compile)");
            actionDescription.setMnemonic('P');
            actionDescription.setAccelerator("ctrl shift S");
            actionDescription.setImageName("save_compile.png");
            actionDescription.setTooltip("Save and Compile current project");
            actionDescription.overrideWithExternalResources(theResourceBundle, "SaveWithCompile");

            return new SaveWithCompileAction(actionDescription);
        }

        SaveAction initSaveAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Save");
            actionDescription.setMnemonic('S');
            actionDescription.setAccelerator("ctrl S");
            actionDescription.setImageName("glp_save.png");
            actionDescription.setTooltip("Save current project");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Save");

            return new SaveAction(actionDescription);
        }

        RedoAction initRedoAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Redo");
            actionDescription.setMnemonic('R');
            actionDescription.setImageName("glp_redo.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Redo");

            return new RedoAction(actionDescription);
        }

        PasteSelectedObjectAction initPasteSelectedObjectAction () {
            String sLabel = getPasteAction().getLabel();
            char mnemonic = getPasteAction().getMnemonicKey();
            String sImageName = "glp_paste.png";

            return new PasteSelectedObjectAction(sLabel, mnemonic, null, sImageName, null);
        }

        PasteAction initPasteAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Paste");
            actionDescription.setMnemonic('P');
            actionDescription.setImageName("glp_paste.png");
            actionDescription.setTooltip("Paste");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Paste");

            return new PasteAction(actionDescription);
        }

        NewAction initNewAction () {
            String sLabel = "New";
            char mnemonic = 'N';
            KeyStroke keyStroke = null;
            String sImageName = "document_new.png";
            String sTooltip = "Create a new project";

            return new NewAction(sLabel, mnemonic, keyStroke, sImageName, sTooltip);
        }

        UndoAction initUndoAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Undo");
            actionDescription.setMnemonic('U');
            actionDescription.setImageName("glp_undo.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "Undo");

            return new UndoAction(actionDescription);
        }

        FontChooserAction initFontChooserAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Set Font");
            actionDescription.setMnemonic('F');
            actionDescription.setImageName("font.png");
            actionDescription.setTooltip("Set the font of the selected object");
            actionDescription.overrideWithExternalResources(theResourceBundle, "ChooseFont");

            return new FontChooserAction(actionDescription);
        }

        ForegroundColorChooserAction initForegroundColorChooserAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Set Foreground Color");
            actionDescription.setMnemonic('O');
            actionDescription.setImageName("palette2.png");
            actionDescription.setTooltip("Set the foreground color of the selected object");
            actionDescription.overrideWithExternalResources(theResourceBundle, "ForegroundColor");

            return new ForegroundColorChooserAction(actionDescription);
        }

        BackgroundColorChooserAction initBackgroundColorChooserAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("Set Background Color");
            actionDescription.setMnemonic('B');
            actionDescription.setImageName("colors.png");
            actionDescription.setTooltip("Set the background color of the selected object");
            actionDescription.overrideWithExternalResources(theResourceBundle, "BackgroundColor");

            return new BackgroundColorChooserAction(actionDescription);
        }

        AboutAction initAboutAction () {
            ActionDescription actionDescription = new ActionDescription();
            actionDescription.setLabel("About");
            actionDescription.setMnemonic('A');
            actionDescription.setImageName("glp_about.png");
            actionDescription.overrideWithExternalResources(theResourceBundle, "About");

            return new AboutAction(actionDescription);
        }

    }

    class ActionDescription {
        private String theLabel;
        private char theMnemonic;
        private KeyStroke theAccelerator;
        private String theImageName;
        private String theTooltip;

        public String getLabel () {
            return theLabel;
        }

        public void setLabel (String theLabel) {
            this.theLabel = theLabel;
        }

        public char getMnemonic () {
            return theMnemonic;
        }

        public void setMnemonic (char mnemonic) {
            this.theMnemonic = mnemonic;
        }

        public KeyStroke getAccelerator () {
            return theAccelerator;
        }

        public void setAccelerator (String sAccelerator) {
            this.theAccelerator = KeyStroke.getKeyStroke(sAccelerator);
        }

        public void setAccelerator (KeyStroke keyStroke) {
            this.theAccelerator = keyStroke;
        }

        public String getImageName () {
            return theImageName;
        }

        public void setImageName (String theImageName) {
            this.theImageName = theImageName;
        }

        public String getTooltip () {
            return theTooltip;
        }

        public void setTooltip (String theTooltip) {
            this.theTooltip = theTooltip;
        }

        public void overrideWithExternalResources (ResourceBundle bundle, String sKeyRoot) {
            if (bundle == null) return;

            // Note: all MissingResourceExceptions are ignored as it isn't an error to
            // not find a resource, as it may not be defined.
            try {
                theLabel = bundle.getString(sKeyRoot + ".label");
            } catch (MissingResourceException e) {
            }

            try {
                String sMnemonic = bundle.getString(sKeyRoot + ".mnemonic");
                if (sMnemonic.length() > 0)
                    theMnemonic = sMnemonic.charAt(0);
            } catch (MissingResourceException e) {
            }

            try {
                String sAccelerator = bundle.getString(sKeyRoot + ".accelerator");
                if (sAccelerator.length() > 0)
                    theAccelerator = KeyStroke.getKeyStroke(sAccelerator);
            } catch (MissingResourceException e) {
            }

            try {
                theTooltip = bundle.getString(sKeyRoot + ".tooltip");
            } catch (MissingResourceException e) {
            }

            try {
                theImageName = bundle.getString(sKeyRoot + ".image");
            } catch (MissingResourceException e) {
            }

        }

    }

}

